/*
 * Copyright (c) 2002, 2003, 2004, 2006, 2007, 2008, 2009, 2010, 2011, 2012, 2013, 2014, 2015, 2016, 2018, 2019, 2020, 2021, 2022, 2024
 * The Regents of the University of California. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *     * Redistributions of source code must retain the above copyright
 *       notice, this list of conditions and the following disclaimer.
 *     * Redistributions in binary form must reproduce the above copyright
 *       notice, this list of conditions and the following disclaimer in the
 *       documentation and/or other materials provided with the distribution.
 *     * Neither the name of the University nor the names of its contributors
 *       may be used to endorse or promote products derived from this software
 *       without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#ifndef lint
static const char rcsid[] __attribute__ ((unused)) =
    "@(#) $Id$ (LBL)";
#endif

#include <sys/types.h>
#include <sys/time.h>

#include <ctype.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sysexits.h>

#include "acld.h"
#include "cf.h"

/* Globals */
struct s2v str2acl[] = {
	{ "unknown",		ATYPE_UNKNOWN },

	{ "permitany",		ATYPE_PERMITANY },
	{ "blockany",		ATYPE_BLOCKANY },

	{ "permithost",		ATYPE_PERMITHOST },
	{ "blockhost",		ATYPE_BLOCKHOST },

	{ "permithosthost",	ATYPE_PERMITHOSTHOST },
	{ "blockhosthost",	ATYPE_BLOCKHOSTHOST },

	{ "blockmac",		ATYPE_BLOCKMAC },

	{ "permitnet",		ATYPE_PERMITNET },
	{ "blocknet",		ATYPE_BLOCKNET },

	{ "permitnethost",	ATYPE_PERMITNETHOST },
	{ "blocknethost",	ATYPE_BLOCKNETHOST },

	{ "permitudpport",	ATYPE_PERMITUDPPORT },
	{ "blockudpport",	ATYPE_BLOCKUDPPORT },
	{ "permittcpport",	ATYPE_PERMITTCPPORT },
	{ "blocktcpport",	ATYPE_BLOCKTCPPORT },

	{ "permittcpsynport",	ATYPE_PERMITTCPSYNPORT },
	{ "blocktcpsynport",	ATYPE_BLOCKTCPSYNPORT },

	{ "permitudphostport",	ATYPE_PERMITUDPHOSTPORT },
	{ "blockudphostport",	ATYPE_BLOCKUDPHOSTPORT },
	{ "permittcphostport",	ATYPE_PERMITTCPHOSTPORT },
	{ "blocktcphostport",	ATYPE_BLOCKTCPHOSTPORT },

	{ "permitudpdsthost",	ATYPE_PERMITUDPDSTHOST },
	{ "blockudpdsthost",	ATYPE_BLOCKUDPDSTHOST },
	{ "permittcpdsthost",	ATYPE_PERMITTCPDSTHOST },
	{ "blocktcpdsthost",	ATYPE_BLOCKTCPDSTHOST },

	{ "permitdsthost",	ATYPE_PERMITDSTHOST },
	{ "blockdsthost",	ATYPE_BLOCKDSTHOST },
	{ "permiticmpdsthost",	ATYPE_PERMITICMPDSTHOST },
	{ "blockicmpdsthost",	ATYPE_BLOCKICMPDSTHOST },

	{ "permitdstnet",	ATYPE_PERMITDSTNET },
	{ "blockdstnet",	ATYPE_BLOCKDSTNET },
	{ "permiticmpdstnet",	ATYPE_PERMITICMPDSTNET },
	{ "blockicmpdstnet",	ATYPE_BLOCKICMPDSTNET },

	{ "blockudpdsthostport", ATYPE_BLOCKUDPDSTHOSTPORT },
	{ "blocktcpdsthostport", ATYPE_BLOCKTCPDSTHOSTPORT },
	{ "permitudpdsthostport", ATYPE_PERMITUDPDSTHOSTPORT },
	{ "permittcpdsthostport", ATYPE_PERMITTCPDSTHOSTPORT },

	{ "permitudpnetport",	ATYPE_PERMITUDPNETPORT },
	{ "blockudpnetport",	ATYPE_BLOCKUDPNETPORT },
	{ "permittcpnetport",	ATYPE_PERMITTCPNETPORT },
	{ "blocktcpnetport",	ATYPE_BLOCKTCPNETPORT },

	{ "permitudphostsrcport", ATYPE_PERMITUDPHOSTSRCPORT },
	{ "blockudphostsrcport", ATYPE_BLOCKUDPHOSTSRCPORT },
	{ "permittcphostsrcport", ATYPE_PERMITTCPHOSTSRCPORT },
	{ "blocktcphostsrcport", ATYPE_BLOCKTCPHOSTSRCPORT },

	{ "permitudphostpairdstport", ATYPE_PERMITUDPHOSTPAIRDSTPORT },
	{ "blockudphostpairdstport", ATYPE_BLOCKUDPHOSTPAIRDSTPORT },
	{ "permittcphostpairdstport", ATYPE_PERMITTCPHOSTPAIRDSTPORT },
	{ "blocktcphostpairdstport", ATYPE_BLOCKTCPHOSTPAIRDSTPORT },

	{ "permitudpdstnet",	ATYPE_PERMITUDPDSTNET },
	{ "blockudpdstnet",	ATYPE_BLOCKUDPDSTNET },
	{ "permittcpdstnet",	ATYPE_PERMITTCPDSTNET },
	{ "blocktcpdstnet",	ATYPE_BLOCKTCPDSTNET },

	{ "permitudpdstnetport", ATYPE_PERMITUDPDSTNETPORT },
	{ "blockudpdstnetport",	ATYPE_BLOCKUDPDSTNETPORT },
	{ "permittcpdstnetport", ATYPE_PERMITTCPDSTNETPORT },
	{ "blocktcpdstnetport",	ATYPE_BLOCKTCPDSTNETPORT },

	{ "permitudpnethostport", ATYPE_PERMITUDPNETHOSTPORT },
	{ "blockudpnethostport", ATYPE_PERMITUDPNETHOSTPORT },
	{ "permittcpnethostport", ATYPE_PERMITTCPNETHOSTPORT },
	{ "blocktcpnethostport", ATYPE_PERMITTCPNETHOSTPORT },

	{ NULL,			0 }
};

/* Forwards */
static void acladdrlistcallback(FILE *, void *);
static int aclcmp(const void *, const void *);
static int aclgroupcmp(const void *, const void *);

/* Add an ACL to an aclgroup */
void
acladdacl(struct aclgroup *gp, struct acl *nal)
{
	acladdacl2(gp, NULL, nal);
}

/* Add an ACL to an acllist (acllist is optional) */
void
acladdacl2(struct aclgroup *gp, struct acllist *ap, struct acl *nal)
{
	struct acl *al;
	struct array *dp;

	if (ap == NULL)
		ap = aclfindlist(gp, nal);
	dp = &ap->acl_array;
	DYNARRAY(dp, ap->acl, dp->len + 1, "acl array");
	al = ap->acl + dp->len;
	aclcopy(al, nal);
	++dp->len;
	++gp->acltotallen;
	stats_sethiwater(&gp->stats, gp->acltotallen);

	if (gp->limitp != NULL)
		++gp->limitp->a_len;

	/*
	 * Sort on sequence number if we have them
	 * (and we're not doing junoscript)
	 */
	if (
#ifdef HAVE_JUNOSCRIPT
	    !cf->c_junoscript &&
#endif
	    (gp->lastseq >= 0 || gp->lastpermithostportseq >= 0 ||
	    gp->lastportseq >= 0))
		qsort(ap->acl, dp->len, sizeof(*al), aclcmp);

	/* Update ACL addr list file (if we're not initializing) */
	if (state.listedallacls && gp->fn != NULL &&
	    al->type == ATYPE_BLOCKHOST)
		aclupdatefile("acladdacl2", gp);
}

int
acladdaclgroup(const char *name, const char *cidr)
{
	const char *errmsg;
	struct addr *a;
	struct mac mac;
	struct acl *al;
	struct aclgroup *gp;
	struct acllist *ap;
	struct array *dp;
	char buf[128];

	gp = aclfindgroupbyname(name);
	if (gp == NULL) {
		/* Allocate the aclgroup */
		dp = &cf->c_aclgroup_array;
		DYNARRAY(dp, cf->c_aclgroup, dp->len + 1, "cf aclgroup");
		gp = cf->c_aclgroup + dp->len;
		gp->name = strsave(name);
		++dp->len;
		gp->lastseq = 0;
		gp->lastportseq = 0;
		gp->lastpermithostportseq = 0;

		dp = &gp->addr_array;
		dp->osize = sizeof(*gp->addr);

		/* Allocate the acllist */
		dp = &gp->acllist_array;
		dp->osize = sizeof(*gp->acllist);
		DYNARRAY(dp, gp->acllist, dp->len + 1, "cf acllist");
		++dp->len;
		ap = gp->acllist;
		ap->name = strsave(gp->name);

		dp = &ap->acl_array;
		dp->osize = sizeof(*ap->acl);
		dp->inclen = 1024;
	}

	stats_init(&gp->stats, 10, 60, "acladdaclgroup: stats");

	/* Try for a mac address first */
	errmsg = extractmac(cidr, &mac);
	if (errmsg == NULL) {
		/* Found a mac address */
		++gp->ismac;
		ap = gp->acllist;
		dp = &ap->acl_array;
		DYNARRAY(dp, ap->acl, dp->len + 1,
		    "acl array (acladdaclgroup)");
		al = ap->acl + dp->len;
		al->mac = mac;

		al->type = ATYPE_BLOCKMAC;
		++dp->len;

		/* Add an interface entry for this guy (if not junoscript) */
#ifdef HAVE_JUNOSCRIPT
		if (!cf->c_junoscript) {
#endif
			(void)snprintf(buf, sizeof(buf), "acld %s", gp->name);
			intlistadd(buf);
#ifdef HAVE_JUNOSCRIPT
		}
#endif
	} else {
		/* Must be a cidr */
		dp = &gp->addr_array;
		DYNARRAY(dp, gp->addr, dp->len + 1, "aclgroup addrs");
		a = gp->addr + dp->len;

		errmsg = extractaddr(cidr, NULL, a);
		if (errmsg != NULL) {
			lg(LOG_ERR, "acladdaclgroup: extractaddr: %s", errmsg);
			return (0);
		}
		if (dp->len > 0 && ACLGROUP_FAMILY(gp) != a->family) {
			lg(LOG_ERR,
			    "acladdaclgroup: Can't add %s to %s ACL list %s",
			    cidr, family2protostr(ACLGROUP_FAMILY(gp)),
			    gp->name);
			return (0);
		}
		++dp->len;
	}
	return (1);
}

static void
acladdrlistcallback(FILE *f, void *opaque)
{
	int i, j;
	struct acl *al;
	struct aclgroup *gp;
	struct acllist *ap;
	struct array *dp, *dp2;

	gp = (struct aclgroup *)opaque;
	if (gp->fn == NULL) {
		lg(LOG_ERR, "acladdrlistcallback: ACL %s missing fn", gp->name);
		return;
	}

	dp = &gp->acllist_array;
	for (i = 0, ap = gp->acllist; i < dp->len; ++i, ++ap) {
		dp2 = &ap->acl_array;
		for (j = 0, al = ap->acl; j < dp2->len; ++j, ++al)
			if (al->type == ATYPE_BLOCKHOST)
				fprintf(f, "%s\n", addr2str(&al->addr1));
	}
}

/* Free dynamic memory in an acl struct */
void
aclclean(struct acl *al)
{
	if (al->port1 != NULL) {
		free(al->port1);
		al->port1 = NULL;
	}
	if (al->port2 != NULL) {
		free(al->port2);
		al->port2 = NULL;
	}
	if (al->raw != NULL) {
		free(al->raw);
		al->raw = NULL;
	}
}

static int
aclcmp(const void *arg1, const void *arg2)
{
	const struct acl *al1, *al2;

	al1 = (const struct acl *)arg1;
	al2 = (const struct acl *)arg2;
	return (al1->seq - al2->seq);
}

/* Copy an ACL saving char fields as required */
void
aclcopy(struct acl *dst, const struct acl *src)
{

	*dst = *src;
	if (dst->port1 != NULL)
		dst->port1 = strsave(dst->port1);
	if (dst->port2 != NULL)
		dst->port2 = strsave(dst->port2);
	if (dst->raw != NULL)
		dst->raw = strsave(dst->raw);
}

int
acldeleteacl(struct aclgroup *gp, struct acl *oal)
{
	int i, n;
	struct acllist *ap;
	struct acl *al;
	struct array *dp;

	ap = aclfindlist(gp, oal);
	dp = &ap->acl_array;
	for (i = 0, al = ap->acl; i < dp->len; ++i, ++al) {
		if (al->type != oal->type)
			continue;

		/* addr1 */
		if (!ITEMEQ(&al->addr1, &oal->addr1))
			continue;

		/* addr2 */
		if (!ITEMEQ(&al->addr2, &oal->addr2))
			continue;

		/* mac */
		if (!ITEMEQ(&al->mac, &oal->mac))
			continue;

		/* port1 */
		if ((al->port1 != NULL) ^ (oal->port1 != NULL))
			continue;
		if (al->port1 != NULL && strcasecmp(al->port1, oal->port1) != 0)
			continue;

		/* port2 */
		if ((al->port2 != NULL) ^ (oal->port2 != NULL))
			continue;
		if (al->port2 != NULL && strcasecmp(al->port2, oal->port2) != 0)
			continue;
		break;
	}
	if (i >= dp->len)
		return (0);

	aclclean(al);

	n = dp->len - (i + 1);
	memmove(al, al + 1, n * sizeof(*al));
	--dp->len;
	--gp->acltotallen;
	stats_setvalue(&gp->stats, gp->acltotallen);
	if (gp->limitp != NULL)
		--gp->limitp->a_len;
	memset(ap->acl + dp->len, 0, sizeof(*ap->acl));

	/* Update ACL addr list file */
	if (gp->fn != NULL && oal->type == ATYPE_BLOCKHOST)
		aclupdatefile("acldeleteacl", gp);
	return (1);
}

/* If checksubnet, check for address in subnet */
struct acl *
aclfindacl(struct aclgroup *gp, struct acl *al, int checksubnet)
{
	int i;
	struct acllist *ap;
	struct acl *al2;
	struct array *dp;

	ap = aclfindlist(gp, al);
	dp = &ap->acl_array;
	for (i = 0, al2 = ap->acl; i < dp->len; ++i, ++al2) {
		if (al->type != al2->type &&
		    (!checksubnet ||
		    al->type != ATYPE_BLOCKHOST ||
		    al2->type != ATYPE_BLOCKNET))
			continue;

		switch (al->type) {

		case ATYPE_PERMITHOST:
		case ATYPE_BLOCKHOST:
			if (checksubnet && insubnet(&al2->addr1, &al->addr1))
				return (al2);
			if (ITEMEQ(&al->addr1, &al2->addr1))
				return (al2);
			break;

		case ATYPE_BLOCKMAC:
			if (ITEMEQ(&al->mac, &al2->mac))
				return (al2);
			break;

		case ATYPE_PERMITNET:
		case ATYPE_BLOCKNET:
			if (ITEMEQ(&al->addr1, &al2->addr1))
				return (al2);
			break;

		case ATYPE_BLOCKHOSTHOST:
			if (ITEMEQ(&al->addr1, &al2->addr1) &&
			    ITEMEQ(&al->addr2, &al2->addr2))
				return (al2);
			break;

		case ATYPE_BLOCKUDPPORT:
		case ATYPE_PERMITUDPPORT:
		case ATYPE_BLOCKTCPPORT:
		case ATYPE_PERMITTCPPORT:
		case ATYPE_BLOCKTCPSYNPORT:
		case ATYPE_PERMITTCPSYNPORT:
			if (strcasecmp(al->port1, al2->port1) == 0)
				return (al2);
			break;

		case ATYPE_PERMITUDPDSTHOSTPORT:
		case ATYPE_BLOCKUDPDSTHOSTPORT:
		case ATYPE_PERMITTCPDSTHOSTPORT:
		case ATYPE_BLOCKTCPDSTHOSTPORT:
			if (ITEMEQ(&al->addr1, &al2->addr1) &&
			    strcasecmp(al->port1, al2->port1) == 0)
				return (al2);
			break;

		default:
			break;
		}
	}
	return (NULL);
}

/* XXX when searching for a network, this doesn't find the widest mask match */
struct aclgroup *
aclfindgroupbyaddr(struct addr *a)
{
	int i, j;
	struct aclgroup *gp;
	struct addr *a2;
	struct array *dp, *dp2;

	dp = &cf->c_aclgroup_array;
	for (i = 0, gp = cf->c_aclgroup; i < dp->len; ++i, ++gp) {
		dp2 = &gp->addr_array;
		for (j = 0, a2 = gp->addr; j < dp2->len; ++j, ++a2)
			if (insubnet(a2, a))
				return (gp);
	}
	return (NULL);
}

struct aclgroup *
aclfindgroupbyname(const char *name)
{
	int i;
	struct aclgroup *gp;
	struct array *dp;

	dp = &cf->c_aclgroup_array;
	for (i = 0, gp = cf->c_aclgroup; i < dp->len; ++i, ++gp)
		if (strcmp(gp->name, name) == 0)
			return (gp);
	return (NULL);
}

/* Find the default ACL list for the address family of the passed addr */
struct aclgroup *
aclfindgroupdefault(struct addr *a)
{
	int i, j;
	struct addr *a2;
	struct aclgroup *gp;
	struct array *dp, *dp2;

	/*
	 * XXX We could probably move the family check to the outer
	 * loop but we would also have to make sure that each acl
	 * list contained homogenious address families
	 * (Which we should do!)
	 */
	dp = &cf->c_aclgroup_array;
	for (i = 0, gp = cf->c_aclgroup; i < dp->len; ++i, ++gp) {
		dp2 = &gp->addr_array;
		for (j = 0, a2 = gp->addr; j < dp2->len; ++j, ++a2)
			if (a->family == a2->family && isdefaultaddr(a2))
				return (gp);
	}
	return (NULL);
}

struct acllist *
aclfindlistbyname(const char *name)
{
	int i, j;
	struct acllist *ap;
	struct aclgroup *gp;
	struct array *dp, *dp2;

	dp = &cf->c_aclgroup_array;
	for (i = 0, gp = cf->c_aclgroup; i < dp->len; ++i, ++gp) {
		dp2 = &gp->acllist_array;
		for (j = 0, ap = gp->acllist; j < dp2->len; ++j, ++ap)
			if (strcmp(ap->name, name) == 0)
				return (ap);
	}
	return (NULL);
}

struct acllist *
aclfindlist(struct aclgroup *gp, struct acl *al)
{
	int i;
	const char *aclname;
	struct acllist *ap;
	struct array *dp;
	struct state *sp;

	/* Map the ACL group name to a potentially different ACL list name */
	sp = &state;
	if (sp->f_clientmapaclname != NULL)
		aclname = (sp->f_clientmapaclname)(gp->name, al);
	else
		aclname = gp->name;

	dp = &gp->acllist_array;
	for (i = 0, ap = gp->acllist; i < dp->len; ++i, ++ap)
		if (strcmp(aclname, ap->name) == 0)
			return (ap);

	lg(LOG_ERR, "aclfindlist: Can't find %s in %s", aclname, gp->name);
	exit(EX_SOFTWARE);
}

const char *
aclformat(struct acl *al)
{
	const char *p;
	char *cp;
	size_t size, len;
	static char buf[1024];

	p = val2str(str2acl, al->type);
	cp = buf;
	size = sizeof(buf);
	(void)snprintf(cp, size, "%d %llu %s", al->seq, al->count, p);
	len = strlen(cp);
	size -= len;
	cp += len;

	switch (al->type) {

	case ATYPE_PERMITANY:
	case ATYPE_BLOCKANY:
		break;

	case ATYPE_PERMITHOST:
	case ATYPE_BLOCKHOST:
	case ATYPE_PERMITUDPDSTHOST:
	case ATYPE_BLOCKUDPDSTHOST:
	case ATYPE_PERMITTCPDSTHOST:
	case ATYPE_BLOCKTCPDSTHOST:
	case ATYPE_PERMITDSTHOST:
	case ATYPE_BLOCKDSTHOST:
	case ATYPE_PERMITICMPDSTHOST:
	case ATYPE_BLOCKICMPDSTHOST:
	case ATYPE_PERMITNET:
	case ATYPE_BLOCKNET:
	case ATYPE_PERMITDSTNET:
	case ATYPE_BLOCKDSTNET:
	case ATYPE_PERMITICMPDSTNET:
	case ATYPE_BLOCKICMPDSTNET:
	case ATYPE_PERMITUDPDSTNET:
	case ATYPE_BLOCKUDPDSTNET:
	case ATYPE_PERMITTCPDSTNET:
	case ATYPE_BLOCKTCPDSTNET:
		(void)snprintf(cp, size, " %s", addr2str(&al->addr1));
		break;

	case ATYPE_BLOCKMAC:
		(void)snprintf(cp, size, " %s", mac2str(&al->mac));
		break;

	case ATYPE_BLOCKHOSTHOST:
	case ATYPE_PERMITHOSTHOST:
		(void)snprintf(cp, size, " %s", addr2str(&al->addr1));

		len = strlen(cp);
		size -= len;
		cp += len;
		(void)snprintf(cp, size, " %s", addr2str(&al->addr2));
		break;

	case ATYPE_BLOCKNETHOST:
	case ATYPE_PERMITNETHOST:
		(void)snprintf(cp, size, " %s", addr2str(&al->addr1));

		len = strlen(cp);
		size -= len;
		cp += len;
		(void)snprintf(cp, size, " %s", addr2str(&al->addr2));
		break;

	case ATYPE_PERMITUDPPORT:
	case ATYPE_BLOCKUDPPORT:
	case ATYPE_PERMITTCPPORT:
	case ATYPE_BLOCKTCPPORT:
	case ATYPE_PERMITTCPSYNPORT:
	case ATYPE_BLOCKTCPSYNPORT:
		(void)snprintf(cp, size, " %s", al->port1);
		break;

	case ATYPE_PERMITUDPHOSTPORT:
	case ATYPE_BLOCKUDPHOSTPORT:
	case ATYPE_PERMITTCPHOSTPORT:
	case ATYPE_BLOCKTCPHOSTPORT:
	case ATYPE_PERMITUDPDSTHOSTPORT:
	case ATYPE_BLOCKUDPDSTHOSTPORT:
	case ATYPE_PERMITTCPDSTHOSTPORT:
	case ATYPE_BLOCKTCPDSTHOSTPORT:
	case ATYPE_PERMITUDPHOSTSRCPORT:
	case ATYPE_BLOCKUDPHOSTSRCPORT:
	case ATYPE_PERMITTCPHOSTSRCPORT:
	case ATYPE_BLOCKTCPHOSTSRCPORT:
	case ATYPE_PERMITUDPDSTNETPORT:
	case ATYPE_BLOCKUDPDSTNETPORT:
	case ATYPE_PERMITTCPDSTNETPORT:
	case ATYPE_BLOCKTCPDSTNETPORT:
		(void)snprintf(cp, size, " %s %s",
		    addr2str(&al->addr1), al->port1);
		break;

	case ATYPE_PERMITUDPNETPORT:
	case ATYPE_BLOCKUDPNETPORT:
	case ATYPE_PERMITTCPNETPORT:
	case ATYPE_BLOCKTCPNETPORT:
		(void)snprintf(cp, size, " %s %s",
		    addr2str(&al->addr1), al->port1);
		break;

	case ATYPE_PERMITUDPHOSTPAIRDSTPORT:
	case ATYPE_BLOCKUDPHOSTPAIRDSTPORT:
	case ATYPE_PERMITTCPHOSTPAIRDSTPORT:
	case ATYPE_BLOCKTCPHOSTPAIRDSTPORT:
		(void)snprintf(cp, size, " %s", addr2str(&al->addr1));

		len = strlen(cp);
		size -= len;
		cp += len;
		(void)snprintf(cp, size, " %s %s",
		    addr2str(&al->addr2), al->port1);
		break;

	case ATYPE_PERMITUDPNETHOSTPORT:
	case ATYPE_BLOCKUDPNETHOSTPORT:
	case ATYPE_PERMITTCPNETHOSTPORT:
	case ATYPE_BLOCKTCPNETHOSTPORT:
		(void)snprintf(cp, size, " %s", addr2str(&al->addr1));

		len = strlen(cp);
		size -= len;
		cp += len;
		(void)snprintf(cp, size, " %s %s",
		    addr2str(&al->addr2), al->port1);
		break;

	case ATYPE_UNKNOWN:
		(void)snprintf(buf, sizeof(buf), "# %d unsupported ACL \"%s\"",
		    al->seq, al->raw != NULL ? al->raw : "");
		break;

#ifdef notdef
	default:
		lg(LOG_ERR, "aclformat: bad type %d", al->type);
		exit(EX_SOFTWARE);
#endif
	}
	return (buf);
}

void
aclfree(struct aclgroup *gp)
{
	int i, j;
	struct acllist *ap;
	struct acl *al;
	struct req *rp;
	struct array *dp, *dp2;

	if (gp->name != NULL) {
		free(gp->name);
		gp->name = NULL;
	}
	if (gp->fn != NULL) {
		free(gp->fn);
		gp->fn = NULL;
	}

	/* Compaction cleanup */
	gp->compactclient = NULL;
	if ((rp = gp->compactreq) != NULL) {
		rp->state = RSTATE_DONE;
		gp->compactreq = NULL;
	}

	if (gp->acllist != NULL) {
		dp = &gp->acllist_array;
		for (i = 0, ap = gp->acllist; i < dp->len; ++i, ++ap) {
			dp2 = &ap->acl_array;
			for (j = 0, al = ap->acl; j < dp2->len; ++j, ++al)
				aclclean(al);
			if (ap->name != NULL) {
				free(ap->name);
				ap->name = NULL;
			}
			FREEDYNARRAY(dp2, ap->acl);
		}
		FREEDYNARRAY(dp, gp->acllist);
	}
	dp = &gp->addr_array;
	FREEDYNARRAY(dp, gp->addr);
	gp->sentlistacl = 0;

	gp->limitp = NULL;

	stats_free(&gp->stats);
}

struct aclgroup *
aclgetmacgroup(void)
{
	int i;
	struct aclgroup *gp;
	struct array *dp;

	dp = &cf->c_aclgroup_array;
	for (i = 0, gp = cf->c_aclgroup; i < dp->len; ++i, ++gp)
		if (gp->ismac)
			return (gp);
	return (NULL);
}

static int
aclgroupcmp(const void *arg1, const void *arg2)
{
	const struct aclgroup *gp, *gp1, *gp2;

	gp1 = (const struct aclgroup *)arg1;
	gp2 = (const struct aclgroup *)arg2;

	gp = aclgetmacgroup();
	if (gp != NULL) {
		if (gp == gp1)
			return (1);
		if (gp == gp2)
			return (-1);
	}

	/* Sort on the first address (good enough to move default to the end) */
	return (addrcmp(gp1->addr, gp2->addr));
}

/* Free just the ACL entries from all ACL groups */
void
aclgroupsfree(void)
{
	int i, j, k;
	struct aclgroup *gp;
	struct acllist *ap;
	struct acl *al;
	struct array *dp, *dp2, *dp3;

	dp = &cf->c_aclgroup_array;
	for (i = 0, gp = cf->c_aclgroup; i < dp->len; ++i, ++gp) {
		gp->acltotallen = 0;
		gp->sentlistacl = 0;
		gp->sentattr = 0;
		if (gp->limitp != NULL)
			gp->limitp->a_len = 0;
		if (gp->acllist == NULL)
			continue;
		dp2 = &gp->acllist_array;
		for (j = 0, ap = gp->acllist; j < dp2->len; ++j, ++ap) {
			dp3 = &ap->acl_array;
			for (k = 0, al = ap->acl; k < dp3->len; ++k, ++al)
				aclclean(al);
			FREEDYNARRAY(dp3, ap->acl);
		}
	}
	state.listedallacls = 0;
}

/* Returns number of matching host, hosthost or subnet acl entries reported */
int
aclquery(struct aclgroup *gp, struct req *rp)
{
	int i, cnt;
	struct acllist *ap;
	struct acl *al;
	struct addr *addrp;
	struct iobuf *ip;
	char buf[128];
	struct array *dp;

	addrp = &rp->acl.addr1;
	ip = &rp->payload;
	ap = aclfindlist(gp, &rp->acl);
	cnt = 0;
	dp = &ap->acl_array;
	for (i = 0, al = ap->acl; i < dp->len; ++i, ++al) {
		switch (al->type) {

		case ATYPE_BLOCKHOST:
			if (ITEMEQ(addrp, &al->addr1)) {
				ioappendfmt(ip, "%s",
				    val2str(str2acl, al->type));
				++cnt;
			}
			break;

		case ATYPE_BLOCKHOSTHOST:
			if (ITEMEQ(addrp, &al->addr1)) {
				strlcpy(buf, addr2str(&al->addr1), sizeof(buf));
				ioappendfmt(ip, "%s %s %s",
				    val2str(str2acl, al->type),
				    buf,
				    addr2str(&al->addr2));
				++cnt;
			}
			break;

		case ATYPE_BLOCKNET:
			/*
			 * Look for ATYPE_BLOCKNET vs. ATYPE_BLOCKNET
			 * or ATYPE_BLOCKNET vs. ATYPE_BLOCKHOST
			 */
			if (ITEMEQ(addrp, &al->addr1) ||
			    insubnet(&al->addr1, addrp)) {
				ioappendfmt(ip, "%s %s",
				    val2str(str2acl, al->type),
				    addr2str(&al->addr1));
				++cnt;
			}
			break;

		default:
			break;
		}
	}
	return (cnt);
}

void
aclsortacls(void)
{
	int i;
	struct aclgroup *gp;
	struct addr *a;
	struct array *dp, *dp2;

	gp = cf->c_aclgroup;
	dp = &cf->c_aclgroup_array;
	qsort(gp, dp->len, sizeof(*gp), aclgroupcmp);
	for (i = 0, gp = cf->c_aclgroup; i < dp->len; ++i, ++gp) {
		dp2 = &gp->addr_array;
		if (dp2->len > 1) {
			a = gp->addr;
			qsort(a, dp2->len, sizeof(*a), addrcmp);
		}
#ifdef HAVE_CFORCE
		/* The rest of this isn't needed for the cForce */
		if (cf->c_cforceaddr != NULL)
			continue;
#endif
		if (cf->c_lowportseq >= 0 &&
		    (gp->lastportseq < 0 || gp->lastportseq < cf->c_lowportseq))
			gp->lastportseq = cf->c_lowportseq - 1;
		if (cf->c_lowpermithostportseq >= 0 &&
		    (gp->lastpermithostportseq < 0 ||
		    gp->lastpermithostportseq < cf->c_lowpermithostportseq))
			gp->lastpermithostportseq =
			    cf->c_lowpermithostportseq - 1;
		if (cf->c_incrseq) {
			if (cf->c_lowseq >= 0 &&
			    (gp->lastseq < 0 || gp->lastseq < cf->c_lowseq))
				gp->lastseq = cf->c_lowseq - 1;
		} else {
			if (cf->c_highseq >= 0 &&
			    (gp->lastseq < 0 || gp->lastseq > cf->c_highseq))
				gp->lastseq = cf->c_highseq + 1;
		}
	}
}

int
aclstraddacl(struct aclgroup *gp, const char *str)
{
	int an;
	const char *p;
	char **av;
	struct acl *al;
	struct acl acl;

	/* Eat leading dot; done if there was only one */
	if (*str == '.') {
		++str;
		if (*str == '\n'|| *str == '\0')
			return (1);
	}

	/* syslog any comments we get */
	if (*str == '#') {
		++str;
		while (*str == ' ')
			++str;
		lg(LOG_INFO, "aclstraddacl: \"%s\"", str);
		return (0);
	}

	al = &acl;
	memset(al, 0, sizeof(*al));
	an = makeargv(str, &av);

	if (an < 3) {
		lg(LOG_ERR, "aclstraddacl: missing args \"%s\"", str);
		freeargv(av);
		return (0);
	}

	al->seq = atoi(av[0]);
	al->count = strtoull(av[1], NULL, 10);
	al->type = str2val(str2acl, av[2]);
	switch (al->type) {

	case ATYPE_PERMITANY:
	case ATYPE_BLOCKANY:
		/* No args */
		if (an != 3) {
			lg(LOG_ERR, "aclstraddacl: wrong number of args \"%s\"",
			    str);
			al->type = ATYPE_UNKNOWN;
			al->raw = (char *)str;
			break;
		}
		break;

	case ATYPE_PERMITHOST:
	case ATYPE_BLOCKHOST:
	case ATYPE_PERMITUDPDSTHOST:
	case ATYPE_BLOCKUDPDSTHOST:
	case ATYPE_PERMITTCPDSTHOST:
	case ATYPE_BLOCKTCPDSTHOST:
	case ATYPE_PERMITDSTHOST:
	case ATYPE_BLOCKDSTHOST:
	case ATYPE_PERMITICMPDSTHOST:
	case ATYPE_BLOCKICMPDSTHOST:
	case ATYPE_PERMITNET:
	case ATYPE_BLOCKNET:
	case ATYPE_PERMITUDPDSTNET:
	case ATYPE_BLOCKUDPDSTNET:
	case ATYPE_PERMITTCPDSTNET:
	case ATYPE_BLOCKTCPDSTNET:
	case ATYPE_PERMITDSTNET:
	case ATYPE_BLOCKDSTNET:
	case ATYPE_PERMITICMPDSTNET:
	case ATYPE_BLOCKICMPDSTNET:
		/* One host or network */
		if (an != 4) {
			lg(LOG_ERR, "aclstraddacl: wrong number of args \"%s\"",
			    str);
			al->type = ATYPE_UNKNOWN;
			al->raw = (char *)str;
			break;
		}
		p = extractaddr(av[3], NULL, &al->addr1);
		if (p != NULL) {
			lg(LOG_ERR, "aclstraddacl: bad addr: %s", p);
			al->type = ATYPE_UNKNOWN;
			al->raw = (char *)str;
			break;
		}
		break;

	case ATYPE_BLOCKMAC:
		if (an != 4) {
			lg(LOG_ERR, "aclstraddacl: wrong number of args \"%s\"",
			    str);
			al->type = ATYPE_UNKNOWN;
			al->raw = (char *)str;
			break;
		}
		p = extractmac(av[3], &al->mac);
		if (p != NULL) {
			lg(LOG_ERR, "aclstraddacl: bad mac: %s", p);
			al->type = ATYPE_UNKNOWN;
			al->raw = (char *)str;
			break;
		}
		break;

	case ATYPE_PERMITHOSTHOST:
	case ATYPE_BLOCKHOSTHOST:
	case ATYPE_PERMITNETHOST:
	case ATYPE_BLOCKNETHOST:
		/* Two hosts or networks */
		if (an != 5) {
			lg(LOG_ERR, "aclstraddacl: wrong number of args \"%s\"",
			    str);
			al->type = ATYPE_UNKNOWN;
			al->raw = (char *)str;
			break;
		}
		if ((p = extractaddr(av[3], NULL, &al->addr1)) != NULL ||
		    (p = extractaddr(av[4], NULL, &al->addr2)) != NULL) {
			lg(LOG_ERR, "aclstraddacl: bad addr: %s", p);
			al->type = ATYPE_UNKNOWN;
			al->raw = (char *)str;
			break;
		}
		break;

	case ATYPE_PERMITUDPPORT:
	case ATYPE_BLOCKUDPPORT:
	case ATYPE_PERMITTCPPORT:
	case ATYPE_BLOCKTCPPORT:
	case ATYPE_PERMITTCPSYNPORT:
	case ATYPE_BLOCKTCPSYNPORT:
		/* A port */
		if (an != 4) {
			lg(LOG_ERR, "aclstraddacl: wrong number of args \"%s\"",
			    str);
			al->type = ATYPE_UNKNOWN;
			al->raw = (char *)str;
			break;
		}
		al->port1 = av[3];
		break;

	case ATYPE_PERMITUDPHOSTPORT:
	case ATYPE_BLOCKUDPHOSTPORT:
	case ATYPE_PERMITTCPHOSTPORT:
	case ATYPE_BLOCKTCPHOSTPORT:
	case ATYPE_PERMITUDPDSTHOSTPORT:
	case ATYPE_BLOCKUDPDSTHOSTPORT:
	case ATYPE_PERMITTCPDSTHOSTPORT:
	case ATYPE_BLOCKTCPDSTHOSTPORT:
	case ATYPE_PERMITUDPHOSTSRCPORT:
	case ATYPE_BLOCKUDPHOSTSRCPORT:
	case ATYPE_PERMITTCPHOSTSRCPORT:
	case ATYPE_BLOCKTCPHOSTSRCPORT:
	case ATYPE_PERMITUDPNETPORT:
	case ATYPE_BLOCKUDPNETPORT:
	case ATYPE_PERMITTCPNETPORT:
	case ATYPE_BLOCKTCPNETPORT:
	case ATYPE_PERMITUDPDSTNETPORT:
	case ATYPE_BLOCKUDPDSTNETPORT:
	case ATYPE_PERMITTCPDSTNETPORT:
	case ATYPE_BLOCKTCPDSTNETPORT:
		/* A host or network and a port */
		if (an != 5) {
			lg(LOG_ERR, "aclstraddacl: wrong number of args \"%s\"",
			    str);
			al->type = ATYPE_UNKNOWN;
			al->raw = (char *)str;
			break;
		}
		p = extractaddr(av[3], NULL, &al->addr1);
		if (p != NULL) {
			lg(LOG_ERR, "aclstraddacl: bad addr/net: %s", p);
			al->type = ATYPE_UNKNOWN;
			al->raw = (char *)str;
			break;
		}
		al->port1 = av[4];
		break;

	case ATYPE_PERMITUDPHOSTPAIRDSTPORT:
	case ATYPE_BLOCKUDPHOSTPAIRDSTPORT:
	case ATYPE_PERMITTCPHOSTPAIRDSTPORT:
	case ATYPE_BLOCKTCPHOSTPAIRDSTPORT:
	case ATYPE_PERMITUDPNETHOSTPORT:
	case ATYPE_BLOCKUDPNETHOSTPORT:
	case ATYPE_PERMITTCPNETHOSTPORT:
	case ATYPE_BLOCKTCPNETHOSTPORT:
		/* Two hosts or networks and a port */
		if (an != 6) {
			lg(LOG_ERR, "aclstraddacl: wrong number of args \"%s\"",
			    str);
			al->type = ATYPE_UNKNOWN;
			al->raw = (char *)str;
			break;
		}
		if ((p = extractaddr(av[3], NULL, &al->addr1)) != NULL ||
		    (p = extractaddr(av[4], NULL, &al->addr2)) != NULL) {
			lg(LOG_ERR, "aclstraddacl: bad addr: %s", p);
			al->type = ATYPE_UNKNOWN;
			al->raw = (char *)str;
			break;
		}
		al->port1 = av[5];
		break;

	case ATYPE_UNKNOWN:
		/* Save the raw text so we can print it out later */
		p = strchr(str, '#');
		if (p == NULL)
			p = str;
		else {
			++p;
			while (isspace((int)*p))
				++p;
		}
		al->raw = (char *)p;
		lg(LOG_ERR, "aclstraddacl: unknown \"%s\"", str);
		break;

#ifdef notdef
	default:
		lg(LOG_ERR, "aclstraddacl: no such type \"%s\"", str);
		exit(EX_SOFTWARE);
#endif
	}

	acladdacl(gp, al);

	if (IN_SEQRANGE(al->seq)) {
		if (cf->c_incrseq) {
			if (gp->lastseq < al->seq)
				gp->lastseq = al->seq;
		} else {
			if (gp->lastseq < 0 || gp->lastseq > al->seq)
				gp->lastseq = al->seq;
		}
	} else if (IN_PERMITHOSTPORTSEQRANGE(al->seq)) {
		if (gp->lastpermithostportseq < al->seq)
			gp->lastpermithostportseq = al->seq;
	} else if (IN_PORTSEQRANGE(al->seq)) {
		if (gp->lastportseq < al->seq)
			gp->lastportseq = al->seq;
	} else {
#ifdef HAVE_JUNOSCRIPT
		if (cf->c_junoscript)
			lg(LOG_ERR, "seq %d in ACL %s not in any valid range",
			    al->seq, gp->name);
#endif
	}

	freeargv(av);
	return (0);
}

void
aclupdatefile(const char *what, struct aclgroup *gp)
{
	const char *errmsg;

	errmsg = writefile(gp->fn, (void *)gp, acladdrlistcallback);
	if (errmsg != NULL)
		lg(LOG_ERR, "%s: %s", what, errmsg);
}

int
addrcmp(const void *arg1, const void *arg2)
{
	int i, r1, r2;
	const struct addr *a1, *a2;

	a1 = (const struct addr *)arg1;
	a2 = (const struct addr *)arg2;

	/* IPv4 before IPv6 */
	if (a1->family != a2->family)
		return ((a1->family == AF_INET) ? -1 : 1);

	switch (a1->family) {

	case AF_INET:
		/* Address */
		if (ntohl(a1->addr4) < ntohl(a2->addr4))
			r1 = -1;
		else if (ntohl(a1->addr4) > ntohl(a2->addr4))
			r1 = 1;
		else
			r1 = 0;

		/* Mask */
		if (ntohl(a1->mask4) < ntohl(a2->mask4))
			r2 = 1;
		else if (ntohl(a1->mask4) > ntohl(a2->mask4))
			r2 = -1;
		else
			r2 = 0;
		break;

	case AF_INET6:
		/* Address */
		r1 = 0;
		for (i = 0; i < 16; ++i) {
			if (ntohl(a1->addr6[i]) < ntohl(a2->addr6[i])) {
				r1 = -1;
				break;
			}
			if (ntohl(a1->addr6[i]) > ntohl(a2->addr6[i])) {
				r1 = 1;
				break;
			}
		}

		/* Mask */
		r2 = 0;
		for (i = 0; i < 16; ++i) {
			if (a1->mask6[i] < a2->mask6[i]) {
				r2 = 1;
				break;
			}
			if (a1->mask6[i] > a2->mask6[i]) {
				r2 = -1;
				break;
			}
		}
		break;

	default:
		abort();
	}

	/* If the masks are the same, sort on addr */
	if (r2 == 0)
		return (r1);

	/* Otherwise, sort on mask */
	return (r2);
}
