/* @(#) $Id$ (LBL) */
#ifndef acld_acl_h
#define acld_acl_h

#include <sys/socket.h>
#include <netinet/in.h>

#include "mac.h"
#include "io.h"
#include "stats.h"
#include "util.h"

/* Address (network order) */
struct addr {
	u_int family;
	union {
		struct in_addr _addr4;
		struct in6_addr _addr6;
	} addr;
	union {
		struct in_addr _mask4;
		struct in6_addr _mask6;
	} mask;
};
#define addr4 addr._addr4.s_addr
#define mask4 mask._mask4.s_addr
#define addr6 addr._addr6.s6_addr
#define mask6 mask._mask6.s6_addr

/*
 * The normal case when blocking hosts or nets is "source"
 * The normal case when blocking ports is "dest"
 * If we're doing something else, we specify
 */
enum acltype {
	ATYPE_UNKNOWN = 0,
	ATYPE_PERMITANY,		/* allow everything */
	ATYPE_BLOCKANY,			/* block everything */

	ATYPE_PERMITHOST,		/* permit a src-host */
	ATYPE_BLOCKHOST,		/* block a src-host */

	ATYPE_BLOCKMAC,			/* block a src-mac-address */

	ATYPE_PERMITHOSTHOST,		/* permit a host to host */
	ATYPE_BLOCKHOSTHOST,		/* block a host to host */

	ATYPE_PERMITNET,		/* permit a src-network */
	ATYPE_BLOCKNET,			/* block a src-network */

	ATYPE_PERMITNETHOST,		/* permit a src-network-host */
	ATYPE_BLOCKNETHOST,		/* block a src-network-host */

	ATYPE_PERMITUDPPORT,		/* permit a dst-udp port */
	ATYPE_BLOCKUDPPORT,		/* block a dst-udp port */
	ATYPE_PERMITTCPPORT,		/* permit a dst-tcp port */
	ATYPE_BLOCKTCPPORT,		/* block a dst-tcp port */

	ATYPE_PERMITTCPSYNPORT,		/* permit a dst-tcp syn port */
	ATYPE_BLOCKTCPSYNPORT,		/* block a dst-tcp syn port */

	ATYPE_PERMITUDPDSTHOST,		/* permit a dst-udp-host */
	ATYPE_BLOCKUDPDSTHOST,		/* block a dst-udp-host */
	ATYPE_PERMITTCPDSTHOST,		/* permit a dst-tcp-host */
	ATYPE_BLOCKTCPDSTHOST,		/* block a dst-tcp-host */

	ATYPE_PERMITDSTHOST,		/* permit a dst-host */
	ATYPE_BLOCKDSTHOST,		/* block a dst-host */
	ATYPE_PERMITICMPDSTHOST,	/* permit a dst-icmp-host */
	ATYPE_BLOCKICMPDSTHOST,		/* block a dst-icmp-host */

	ATYPE_PERMITDSTNET,		/* permit a dst-net */
	ATYPE_BLOCKDSTNET,		/* block a dst-net */
	ATYPE_PERMITICMPDSTNET,		/* permit a dst-icmp-net */
	ATYPE_BLOCKICMPDSTNET,		/* block a dst-icmp-net */

	ATYPE_PERMITUDPHOSTPORT,	/* permit a src-host/dst-udp port */
	ATYPE_BLOCKUDPHOSTPORT,		/* block a src-host/dst-udp port */
	ATYPE_PERMITTCPHOSTPORT,	/* permit a src-host/dst-tcp port */
	ATYPE_BLOCKTCPHOSTPORT,		/* block a src-host/dst-tcp port */

	ATYPE_PERMITUDPDSTHOSTPORT,	/* permit a dst-host/dst-udp port */
	ATYPE_BLOCKUDPDSTHOSTPORT,	/* block a dst-host/dst-udp port */
	ATYPE_PERMITTCPDSTHOSTPORT,	/* permit a dst-host/dst-tcp port */
	ATYPE_BLOCKTCPDSTHOSTPORT,	/* block a dst-host/dst-tcp port */

	ATYPE_PERMITUDPNETPORT,		/* permit a src-network/dst-udp port */
	ATYPE_BLOCKUDPNETPORT,		/* block a src-network/dst-udp port */
	ATYPE_PERMITTCPNETPORT,		/* permit a src-network/dst-tcp port */
	ATYPE_BLOCKTCPNETPORT,		/* block a src-network/dst-tcp port */

	ATYPE_PERMITUDPHOSTSRCPORT,	/* permit a src-host/src-udp port */
	ATYPE_BLOCKUDPHOSTSRCPORT,	/* block a src-host/src-udp port */
	ATYPE_PERMITTCPHOSTSRCPORT,	/* permit a src-host/src-tcp port */
	ATYPE_BLOCKTCPHOSTSRCPORT,	/* block a src-host/src-tcp port */

	ATYPE_PERMITUDPHOSTPAIRDSTPORT,	/* permit a host pair/dst-udp port */
	ATYPE_BLOCKUDPHOSTPAIRDSTPORT,	/* block a host pair/dst-udp port */
	ATYPE_PERMITTCPHOSTPAIRDSTPORT,	/* permit a host pair/dst-tcp port */
	ATYPE_BLOCKTCPHOSTPAIRDSTPORT,	/* block a host pair/dst-tcp port */

	ATYPE_PERMITUDPDSTNET,		/* permit a udp-dst-net */
	ATYPE_BLOCKUDPDSTNET,		/* block a udp-dst-net */
	ATYPE_PERMITTCPDSTNET,		/* permit a tcp-dst-net */
	ATYPE_BLOCKTCPDSTNET,		/* block a tcp-dst-net */

	ATYPE_PERMITUDPDSTNETPORT,	/* permit a udp-dst-net port */
	ATYPE_BLOCKUDPDSTNETPORT,	/* block a udp-dst-net port */
	ATYPE_PERMITTCPDSTNETPORT,	/* permit a tcp-dst-net port */
	ATYPE_BLOCKTCPDSTNETPORT,	/* block a tcp-dst-net port */

	ATYPE_PERMITUDPNETHOSTPORT,	/* permit a udp-net-host-port */
	ATYPE_BLOCKUDPNETHOSTPORT,	/* block a udp-net-host-port */
	ATYPE_PERMITTCPNETHOSTPORT,	/* permit a tcp-net-host-port */
	ATYPE_BLOCKTCPNETHOSTPORT,	/* block a tcp-net-host-port */
};

/* ACL */
struct acl {
	enum acltype type;
	int seq;			/* sequence number */
	unsigned long long count;
	struct addr addr1;
	struct addr addr2;
	struct mac mac;			/* ethernet mac address */
	char *port1;
	char *port2;
	char *raw;			/* text as returned from the router */
};

#define ISHOST(a) \
    (((a)->family == AF_INET && (a)->mask4 == 0xffffffff) || \
    ((a)->family == AF_INET6 && maskwidth(a) == 128))

#define ITEMEQ(a1, a2) (memcmp(a1, a2, sizeof(*(a1))) == 0)
#define ITEMZERO(a) memset(a, 0, sizeof(*(a)));

/* ACL list */
struct acllist {
	char *name;
	struct acl *acl;
	struct array acl_array;
};

/* ACL group */
struct aclgroup {
	char *name;
	char *fn;				/* optional addr list file */
	int sentlistacl;
	int sentattr;
	int ismac;
	struct acllist *acllist;
	struct array acllist_array;
	size_t acltotallen;
	struct acllimit *limitp;
	int lastseq;
	int lastportseq;
	int lastpermithostportseq;
	int compactseq;
	int compactportseq;
	int compactpermithostportseq;
	struct client *compactclient;
	struct req *compactreq;
	struct addr *addr;			/* used to choose acl group */
	struct array addr_array;
	struct intlist *intlist;
	struct stats stats;
};

#define ACLGROUP_FAMILY(gp) ((gp)->addr->family)
#define ACLGROUP_ISINET4(gp) ((gp)->addr->family == AF_INET)
#define ACLGROUP_ISINET6(gp) ((gp)->addr->family == AF_INET6)

struct acllimit {
	int a_len;
	int a_maxseq;
	int a_nacls;
	char **a_acls;
};

struct intlist {
	char *i_acl;
	char *i_name;
};

extern struct s2v str2acl[];

extern void acladdacl(struct aclgroup *, struct acl *);
extern void acladdacl2(struct aclgroup *, struct acllist *, struct acl *);
extern int acladdaclgroup(const char *, const char *);
extern void aclclean(struct acl *);
extern void aclcopy(struct acl *, const struct acl *);
extern int acldeleteacl(struct aclgroup *, struct acl *);
extern struct acl *aclfindacl(struct aclgroup *, struct acl *, int);
extern struct aclgroup *aclfindgroupbyaddr(struct addr *);
extern struct aclgroup *aclfindgroupbyname(const char *);
extern struct aclgroup *aclfindgroupdefault(struct addr *);
extern struct acllist *aclfindlistbyname(const char *);
extern struct acllist *aclfindlist(struct aclgroup *, struct acl *);
extern const char *aclformat(struct acl *);
extern void aclfree(struct aclgroup *);
extern struct aclgroup *aclgetmacgroup(void);
extern void aclgroupsfree(void);
extern int aclquery(struct aclgroup *, struct req *);
extern void aclsortacls(void);
extern int aclstraddacl(struct aclgroup *, const char *);
extern void aclupdatefile(const char *, struct aclgroup *);
extern int addrcmp(const void *, const void *);
#endif
