/*
 * Copyright (c) 2002, 2003, 2004, 2005, 2006, 2007, 2008, 2009, 2010, 2011, 2012, 2013, 2014, 2015, 2016, 2018, 2019, 2020, 2022, 2024
 * The Regents of the University of California. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *     * Redistributions of source code must retain the above copyright
 *       notice, this list of conditions and the following disclaimer.
 *     * Redistributions in binary form must reproduce the above copyright
 *       notice, this list of conditions and the following disclaimer in the
 *       documentation and/or other materials provided with the distribution.
 *     * Neither the name of the University nor the names of its contributors
 *       may be used to endorse or promote products derived from this software
 *       without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#ifndef lint
static const char copyright[] __attribute__ ((unused)) =
    "@(#) Copyright (c) 2002, 2003, 2004, 2005, 2006, 2007, 2008, 2009, 2010, 2011, 2012, 2013, 2014, 2015, 2016, 2018, 2019, 2020, 2022, 2024\n\
The Regents of the University of California.  All rights reserved.\n";
static const char rcsid[] __attribute__ ((unused)) =
    "@(#) $Id$ (LBL)";
#endif

/*
 * acld - manage ACLs on a router
 */

#include <sys/types.h>
#include <sys/time.h>
#include <sys/wait.h>

#include <arpa/inet.h>

#include <errno.h>
#include <fcntl.h>
#include <signal.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sysexits.h>
#include <time.h>
#include <unistd.h>

#include "gnuc.h"
#ifdef HAVE_OS_PROTO_H
#include "os-proto.h"
#endif

#include "acld.h"
#include "cf.h"
#ifdef HAVE_CFORCE
#include "cforcep.h"
#endif
#include "child.h"
#include "child2.h"
#ifdef __FreeBSD__
#include "freebsd.h"
#include "history.h"
#endif
#ifdef HAVE_JUNOSCRIPT
#include "junoscript.h"
#endif
#ifdef HAVE_CORSA
#include "corsa.h"
#endif
#include "mock_client.h"
#include "server.h"
#include "setsignal.h"

/* Globals */
int debug;			/* turn on debugging and don't fork */
int verbose;			/* not really used for anything! */
int reload;			/* kill child and reload config */
int foreground;
const char *prog;
const char *configfn = "/usr/local/etc/acld.conf";
const char *logfile;
char hostname[64];
struct state state;

/* acld states */
struct s2v astate2str[] = {
	{ "connected",		ASTATE_CONNECTED },
	{ "connecting",		ASTATE_CONNECTING },
	{ "loggedin",		ASTATE_LOGGEDIN },
	{ "notconnected",	ASTATE_NOTCONNECTED },
	{ "readacl",		ASTATE_READACL },
	{ "readerror",		ASTATE_READERROR },
	{ "readresponse",	ASTATE_READRESPONSE },
	{ "readroute",		ASTATE_READROUTE },
	{ "sentattr",		ASTATE_SENTATTR },
	{ "sentlistacl",	ASTATE_SENTLISTACL },
	{ "sentlistroute",	ASTATE_SENTLISTROUTE },
	{ "sentlogin",		ASTATE_SENTLOGIN },
	{ NULL,			0 }
};

/* request states */
struct s2v rstate2str[] = {
	{ "child",		RSTATE_CHILD },
//	{ "child2",		RSTATE_CHILD2 },
	{ "done",		RSTATE_DONE },
//	{ "done2",		RSTATE_DONE2 },
	{ "pending",		RSTATE_PENDING },
	{ "pending2",		RSTATE_PENDING2 },
	{ "readcomment",	RSTATE_READCOMMENT },
	{ "readresponse",	RSTATE_READRESPONSE },
	{ NULL,			0 }
};

/* Locals */
static int sawterm;

static const char *pidfile = "/var/run/acld/acld.pid";

/* Externs */
extern int optind;
extern int opterr;
extern char *optarg;

/* Forwards */
int expect(void);
void handleclientreqs(void);
void initiateclient(int, enum clienttype);
int main(int, char **);
int opensocket(struct addr *, u_short);
int reapdeadclients(void);
void settimeout(void);
void sigchild(int);
void sighup(int);
#ifdef notdef
void sigpipe(int);
#endif
void sigterm(int);
int update_fd_set(fd_set *, fd_set *);
__dead void usage(void) __attribute__((noreturn));

/* Append the new request to end of list */
void
appendreq(struct req **reqp, struct req *rp)
{
	struct req *rp2;

	if (*reqp == NULL)
		*reqp = rp;
	else {
		rp2 = *reqp;
		while (rp2->next != NULL)
			rp2 = rp2->next;
		rp2->next = rp;
	}
}

/* Create the expect process */
int
expect(void)
{
	pid_t pid;
	int xerrno;
	int wfds[2], rfds[2];
	char **av, *argv[5], *envp[1];

	if (access(cf->c_script, R_OK) < 0) {
		lg(LOG_ERR, "expect: %s: %s", cf->c_script, strerror(errno));
		exit(EX_OSFILE);
	}

	if (access(cf->c_expect, X_OK) < 0) {
		lg(LOG_ERR, "expect: %s: %s", cf->c_expect, strerror(errno));
		exit(EX_OSFILE);
	}

	if (pipe(rfds) < 0)
		return (-1);
	if (pipe(wfds) < 0) {
		xerrno = errno;
		(void)close(rfds[0]);
		(void)close(rfds[1]);
		errno = xerrno;
		return (-1);
	}

	pid = fork();
	if (pid < 0) {
		xerrno = errno;
		(void)close(rfds[0]);
		(void)close(rfds[1]);
		(void)close(wfds[0]);
		(void)close(wfds[1]);
		errno = xerrno;
		return (-1);
	}
	if (pid == 0) {
		/* Child */
		/* XXX close all descriptors?? */
		/* Setup stdin, stdout and stderr */
		if (rfds[0] != STDIN_FILENO) {
			(void)dup2(rfds[0], STDIN_FILENO);
			(void)close(rfds[0]);
		}
		(void)close(rfds[1]);

		if (wfds[1] != STDOUT_FILENO) {
			(void)dup2(wfds[1], STDOUT_FILENO);
			(void)close(wfds[1]);
		}
		(void)dup2(STDOUT_FILENO, STDERR_FILENO);
		(void)close(wfds[0]);

		av = argv;
		*av++ = cf->c_expect;
		*av++ = cf->c_script;
		if (cf->c_expectlog != NULL) {
			*av++ = "-l";
			*av++ = cf->c_expectlog;
		}
		*av++ = NULL;
		envp[0] = NULL;
		(void)execve(cf->c_expect, argv, envp);
		lg(LOG_ERR, "execve: %s", strerror(errno));
		exit(EX_OSERR);
	}

	/* Parent */
	(void)close(rfds[0]);
	(void)close(wfds[1]);

	/* The child's write is our read, etc. */
	state.rfd = wfds[0];
	state.wfd = rfds[1];
	state.pid = pid;
	lg(LOG_DEBUG, "expect: spawed child %d", (int)pid);
	return (0);
}

void
freereq(struct req *rp)
{
	int i;
	struct aclgroup *gp;
	struct array *dp;

	if (rp == NULL)
		return;

	if (rp->next != NULL) {
		lg(LOG_ERR, "freereq: rp->next not NULL");
		exit(EX_SOFTWARE);
	}

	/* Clean up curreq in child2 children */
	child2freereq(rp);

	if (rp->rereq) {
		if (rp->rereq->state == RESTATE_DONE)
			replicantfreerereq(rp->rereq);
		else {
			/* Unlink from replicant request from request */
			if (rp->rereq != NULL) {
				rp->rereq->req = NULL;
				rp->rereq = NULL;
			}
		}
		rp->rereq = NULL;
	}
	if (rp->aclname != NULL) {
		free(rp->aclname);
		rp->aclname = NULL;
	}
	if (rp->prefixname != NULL) {
		free(rp->prefixname);
		rp->prefixname = NULL;
	}
	if (rp->comment != NULL) {
		free(rp->comment);
		rp->comment = NULL;
	}
	iofree(&rp->payload);
	if (rp->av != NULL)  {
		freeargv(rp->av);
		rp->av = NULL;
	}
	if (rp->acl.port1 != NULL) {
		free(rp->acl.port1);
		rp->acl.port1 = NULL;
	}
	if (rp->acl.port2 != NULL) {
		free(rp->acl.port2);
		rp->acl.port2 = NULL;
	}
	if (rp->acl.raw != NULL) {
		free(rp->acl.raw);
		rp->acl.raw = NULL;
	}
	if (rp->nullzero.raw != NULL) {
		free(rp->nullzero.raw);
		rp->nullzero.raw = NULL;
	}
	free(rp);

	/* Clean up compaction state */
	dp = &cf->c_aclgroup_array;
	for (i = 0, gp = cf->c_aclgroup; i < dp->len; ++i, ++gp)
		if (gp->compactreq == rp) {
			gp->compactclient = NULL;
			gp->compactreq = NULL;
			break;
		}
}

int
getastatesecs(void)
{
	return (time(NULL) - state.loginout_ts);
}

struct client *
getreqclient(struct req *rp)
{
	struct client *cl;
	struct req *rp2;

	for (cl = state.clients; cl != NULL; cl = cl->next)
		for (rp2 = cl->req; rp2 != NULL; rp2 = rp2->next)
			if (rp2 == rp)
				return (cl);

	return (NULL);
}

/*
 * Loop across clients attempting to process requests.
 * If multiball is not active, return as soon as the child is busy.
 * If multiball is active process all requests.
 * Either way, round robin to give each client a fair chance.
 */
void
handleclientreqs(void)
{
	int busy, didany;
	struct client *cl;
	struct req *rp;
	struct state *sp;

	/* Handle child2 requests */
	sp = &state;
	if (sp->chp2 != NULL && sp->f_child2process != NULL)
		(sp->f_child2process)();

	/* Initialize temp req list pointers, determine if child is busy */
	busy = 0;
	for (cl = sp->clients; cl != NULL; cl = cl->next) {
		cl->hreq = cl->req;
		for (rp = cl->req; rp != NULL; rp = rp->next)
			if (rp->state == RSTATE_CHILD)
				++busy;
	}

	while (sp->multiball || !busy) {
		didany = 0;
		for (cl = sp->clients; cl != NULL; cl = cl->next) {
			/* Find next request that's ready to process */
			for (rp = cl->hreq; rp != NULL; rp = rp->next) {
				cl->hreq = rp;
				if (rp->state == RSTATE_PENDING)
					break;
			}
			if (rp == NULL)
				continue;

			/* Let the replicant have a look */
			if (cf->c_portreplicant > 0 &&
			    rp->rereq == NULL && !cl->replicant)
				replicantreqhook(rp);

			/* Attempt to process this request */
			clientprocess(cl, rp, busy);

			/* If no change in request state, skip to next client */
			if (rp->state == RSTATE_PENDING)
				continue;

			++didany;
			if (rp->state == RSTATE_CHILD)
				++busy;
		}
		if (!didany)
			break;
	}

	/* Call optional child hook */
	/* Would like to only commit if no pending client input */
	if (sp->f_childcommit != NULL)
		(sp->f_childcommit)();

	/* Rotate the first client to the last position */
	if (sp->clients != NULL && sp->clients->next != NULL) {
		cl = sp->clients;
		while (cl->next != NULL)
			cl = cl->next;
		cl->next = sp->clients;
		sp->clients = sp->clients->next;
		cl->next->next = NULL;
	}
}

struct req hello = {
	NULL,				/* next */
	NULL,				/* rereq */
	REQ_UNKNOWN,			/* REQ_* */
	RSTATE_PENDING,			/* RSTATE_* */
	RFLAG_CONTINUE,			/* request flags */
	"acld",				/* client command name */
	NULL,				/* aclgroup pointer */
	NULL,				/* ACL name */
	NULL,				/* prefix name */
	0,				/* cookie */
	0,				/* multiball request */
	{ 0, 0 },			/* arrival timestamp */
	{ 0, 0 },			/* completion timestamp */
	0,				/* retries */
	0,				/* child2 pending */
	0,				/* child2 not done */
	0,				/* total number of child2 children */
	0,				/* bitmask of child2 QUERY* results */
	0,				/* argument count */
	NULL,				/* argument vector */
	NULL,				/* comment */
	{ NULL, { 1, IOBUF_BYTES, 0, 0 }}, /* dynarray */
	{ ROUTE_UNKNOWN },		/* route */
	{ 0 },				/* filter */
	{ 0 },				/* prefix */
	{ 0 },				/* whitelist */
	{ ATYPE_UNKNOWN },		/* ACL */
};

/* XXX limit number of clients? */
void
initiateclient(int s, enum clienttype ctype)
{
	int c;
	struct client *cl;
	socklen_t salen;
	u_int16_t port;
	struct sockaddr_storage sa;
	struct sockaddr *sap;
	struct req *rp;

	memset(&sa, 0, sizeof(sa));
	sap = (struct sockaddr *)&sa;
	salen = sizeof(sa);
	c = accept(s, sap, &salen);
	if (c < 0) {
		lg(LOG_ERR, "accept: %s", strerror(errno));
		if (errno == EBADF)
			exit(EX_OSERR);
		return;
	}
	blocking(c, 0);
	cl = newclient(c);
	cl->type = ctype;
	sa2addr(sap, NULL, &cl->addr, &port);

	lg(LOG_DEBUG, "client #%d connect from %s.%hu (%s)",
	    cl->n, addr2str(&cl->addr), port, val2str(ctype2str, cl->type));

	rp = &hello;
	getts(&rp->cts);
	clientsendfmt(cl, rp, "ready for action");
}

int
main(int argc, char **argv)
{
	int n, lastn, i, nfds, op, done;
	char *cp;
	const char *p;
	time_t t;
	FILE *fp;
	ssize_t cc;
	struct req *rp, *nextrp, *lastrp;
	struct state *sp;
	struct client *cl, *nextcl;
	fd_set readfds, writefds;
	struct stat sbuf;
	char buf[132];

	if (argv[0] == NULL)
		prog = "acld";
	else if ((cp = strrchr(argv[0], '/')) != NULL)
		prog = cp + 1;
	else
		prog = argv[0];

	openlog(prog, 0, LOG_DAEMON);

	opterr = 0;
	while ((op = getopt(argc, argv, "c:dDfo:P:v")) != EOF)
		switch (op) {

		case 'c':
			configfn = optarg;
			break;

		case 'd':
			++debug;
			break;

		case 'f':
			++foreground;
			break;

		case 'o':
			logfile = optarg;
			break;

		case 'P':
			pidfile = optarg;
			break;

		case 'v':
			++verbose;
			break;

		default:
			usage();
			/* NOTREACHED */
		}

	if (argc != optind)
		usage();

	if (logfile != NULL) {
		log_lf.fn = logfile;
		checklog(&log_lf);
	} else
		log_lf.f = stderr;

	if (!foreground && daemon(0, 1) < 0) {
		lg(LOG_ERR, "daemon: %s", strerror(errno));
		exit(EX_OSERR);
	}

	/* Update pidfile */
	fp = fopen(pidfile, "w");
	if (fp == NULL)
		lg(LOG_ERR, "fopen: %s: %s", pidfile, strerror(errno));
	else {
		fprintf(fp, "%d\n", (int)getpid());
		(void)fclose(fp);
	}

	buf[0] = '\0';
#ifdef TC_VERSION_STRING
	(void)snprintf(buf, sizeof(buf), " (with %s)", TC_VERSION_STRING);
#endif
	lg(LOG_INFO, "version %s%s starting", version, buf);

	(void)setsignal(SIGTERM, sigterm);
	(void)setsignal(SIGINT, sigterm);
	(void)setsignal(SIGHUP, sighup);
	(void)setsignal(SIGCHLD, sigchild);
	(void)setsignal(SIGPIPE, SIG_IGN);

	sp = &state;
	sp->epoc_ts = time(NULL);
	sp->loginout_ts = time(NULL);
	sp->state = ASTATE_NOTCONNECTED;
	sp->rfd = -1;
	sp->wfd = -1;
	sp->refd = -1;
	sp->pid = -1;
	sp->s = -1;
	sp->s6 = -1;
	sp->ro = -1;
	sp->ro6 = -1;
	sp->web = -1;
	sp->web6 = -1;
	IOBUF_INIT(&sp->re_rbuf);
	IOBUF_INIT(&sp->re_wbuf);
	IOBUF_INIT(&sp->rbuf);
	sp->routes_array.osize = sizeof(*sp->routes);
	sp->routes_array.inclen = 1024;
	sp->chp2_array.osize = sizeof(*sp->chp2);
	reload = 1;

	sp->chp2 = NULL;

	/* Get the hostname */
	if (gethostname(hostname, sizeof(hostname)) < 0)
		strcpy(hostname, "?");

	/* Child hooks */
	sp->f_childcheckreq = childcheckreq;
	sp->f_childcommit = NULL;
	sp->f_childconnect = NULL;
	sp->f_childinput = childinput;
	sp->f_childkill = childkill;
	sp->f_childlistacl = childlistacl;
	sp->f_childlistroute = childlistroute;
	sp->f_childlogin = childlogin;
	sp->f_childread = ioread;
	sp->f_childready = NULL;
	sp->f_childrequest = childrequest;
	sp->f_childsend = childsend;
	sp->f_childsendattr = childsendattr;

	/* Client hooks */
	sp->f_clientcompact = clientcompact;
	sp->f_clientdroprestore = clientdroprestore;
	sp->f_clientlistroute = clientlistroute;
	sp->f_clientnullzero = clientnullzero;
	sp->f_clientprefix = clientprefix;
	sp->f_clientquerynullzero = clientquerynullzero;

	/* Server hooks */
	sp->f_serverayt = serverayt;
	sp->f_servercompact = servercompact;
	sp->f_serversync = serversync;

	/* Second client hooks */
	sp->f_clientlistroute2 = NULL;
	sp->f_clientnullzero2 = NULL;
	sp->f_clientquerynullzero2 = NULL;

	/* Initial config file parsing */
	parsecf(configfn);

#ifdef HAVE_CFORCE
	/* cForce */
	if (cf->c_cforceaddr != NULL)
		cforceinit();
#endif

#ifdef HAVE_JUNOSCRIPT
	/* junoscript */
	if (cf->c_junoscript)
		junoscriptinit();
#endif

#ifdef __FreeBSD__
	/* FreeBSD routes (called later so it can override junoscript) */
	if (cf->c_freebsdroutes) {
		freebsdinit();
		history_init();
	}
#endif

#ifdef HAVE_CORSA
	/* corsa */
	if (cf->c_corsa)
		corsainit();
#endif

	if (cf->c_mockclient)
		mock_init();

	/* nullzero routes */
	routeinit();

	/* Replicant initialization */
	replicantinit();

	lastn = 0;
	for (;;) {
		if (sawterm) {
			lg(LOG_NOTICE, "exiting");
			/* Explicitly terminate the child */
			if (sp->pid > 0 &&
			    kill(sp->pid, SIGTERM) < 0 && errno != ESRCH)
				lg(LOG_ERR, "kill %d: %s",
				    (int)sp->pid, strerror(errno));
			/* Simulate /bin/sh exit status */
			exit(128 + SIGTERM);
		}
		if (reload) {
			/* Terminate the child */
			if (sp->wfd >= 0 && sp->state == ASTATE_LOGGEDIN)
				(sp->f_childkill)();

			/* Terminate the second child */
			/* XXX fix comment */
			if (sp->chp2 != NULL)
				child2reload();

			/* Reload config file if it's changed */
			if (cf != NULL) {
				/* Get the config file mod time */
				if (stat(configfn, &sbuf) < 0)
					lg(LOG_ERR, "stat %s: %s",
					    configfn, strerror(errno));
				else {
					t = (sbuf.st_mtime > sbuf.st_ctime) ?
					    sbuf.st_mtime : sbuf.st_ctime;
					if (t > cf->c_time) {
						lg(LOG_INFO, "reloading %s",
						    configfn);
						freecf();
						routefree();
					}
				}
			}
			reload = 0;
			timerset(&sp->t_login, 1);
		}

		/* Read the config file */
		if (cf == NULL)
			parsecf(configfn);

		/* Create new expect process */
		if (cf->c_script != NULL && sp->wfd < 0 && expect() < 0) {
			lg(LOG_ERR, "expect child: %s", strerror(errno));
			exit(EX_OSERR);
		}

		if (cf->c_mockclient && sp->state == ASTATE_NOTCONNECTED) {
			sp->state = ASTATE_LOGGEDIN;
			sp->sentattrs++;
			sp->listedroutes++;
			sp->listedallacls++;
		}

		/* Internal processing */
		if (
#ifdef HAVE_CFORCE
		    cf->c_cforceaddr != NULL ||
#endif
#ifdef HAVE_JUNOSCRIPT
		    cf->c_junoscript ||
#endif
		    sp->wfd >= 0) {
			switch (sp->state) {

			case ASTATE_CONNECTED:
#if defined(HAVE_CFORCE) || defined(HAVE_JUNOSCRIPT)
			case ASTATE_NOTCONNECTED:
#endif
				/* Reset timers if not pending */
				if (timercheck(&sp->t_login) < 0)
					timerset(&sp->t_login,
					    cf->c_login_secs);

				if (timerdue(&sp->t_login))
					(sp->f_childlogin)();
				break;

			case ASTATE_LOGGEDIN:
				/*
				 * List routes before ACLs so FreeBSD
				 * route requests work when we're not
				 * logged into the router
				 */
				if (!sp->sentattrs)
					(sp->f_childsendattr)();
				else if (!sp->listedroutes)
					(sp->f_childlistroute)();
				else if (!sp->listedallacls)
					(sp->f_childlistacl)();
				break;

			default:
				break;
			}
		}

		/* Open the listen sockets */
		if (cf->c_listen6 && sp->s6 < 0)
			sp->s6 = opensocket(&cf->c_bindaddr6, cf->c_port);
		if (sp->s < 0)
			sp->s = opensocket(&cf->c_bindaddr, cf->c_port);

		/* Optional read/only request socket */
		if (cf->c_portro > 0) {
			if (cf->c_listen6 && sp->ro6 < 0)
				sp->ro6 = opensocket(&cf->c_bindaddr6,
				    cf->c_portro);
			if (sp->ro < 0)
				sp->ro = opensocket(&cf->c_bindaddr,
				    cf->c_portro);
		}

		/* Optional web registration request socket */
		if (cf->c_portweb > 0) {
			if (cf->c_listen6 && sp->web6 < 0)
				sp->web6 = opensocket(&cf->c_bindaddr6,
				    cf->c_portweb);
			if (sp->web < 0)
				sp->web = opensocket(&cf->c_bindaddr,
				    cf->c_portweb);
		}

		/* Update fd_sets */
		nfds = update_fd_set(&readfds, &writefds);

		/* Poll instead of wait if we had ready descriptors last time */
		if (lastn > 0)
			memset(&sp->timeout, 0, sizeof(sp->timeout));
		else
			settimeout();
		n = select(nfds, &readfds, &writefds, NULL, &sp->timeout);
		lastn = n;
		if (n < 0) {
			/* Don't choke if we get ptraced */
			if (errno == EINTR)
				continue;

			lg(LOG_ERR, "select: %s", strerror(errno));
			/*
			 * There's a race between the client and
			 * our call to select(). It's possible for
			 * the client to close his connection just
			 * before we call select(). When this happens
			 * we get EBADF. If we can find the client
			 * and clean him up, we can continue on.
			 * Otherwise, exit.
			 */
			if (errno == EBADF && reapdeadclients() > 0)
				continue;
			exit(EX_OSERR);
		}

		/* Time for maintenance */
		// XXX this is broken
		// t = time(NULL);
		if (n == 0) {
			/* Did the child fall asleep? */
			if (((cf->c_expect != NULL && sp->wfd < 0) ||
			    sp->state != ASTATE_LOGGEDIN) &&
			    timerdue(&sp->t_ayt)) {
				(sp->f_childkill)();
			}

			/* Only do maintenance if there's nothing else to do */
			if (sp->req == NULL && !childbusy()) {
				if (timerdue(&sp->t_compact))
					(sp->f_servercompact)();
				else if (timerdue(&sp->t_sync))
					(sp->f_serversync)();
				else if (timerdue(&sp->t_ayt))
					(sp->f_serverayt)();
			}
		}

		/* Open optional replicant connection */
		if (cf->c_portreplicant > 0 && sp->refd < 0 &&
		    timerdue(&sp->t_re_connect))
			replicantlogin();

		/* Did any clients say anything? */
		/* XXX limit about of client data to throttle them? */
		for (cl = sp->clients; cl != NULL; cl = nextcl) {
			/* Ok to free clients inside this loop */
			nextcl = cl->next;
			if (cl->c >= 0 && FD_ISSET(cl->c, &readfds)) {
				cc = ioread(cl->c, &cl->rbuf);
				if (cc < 0) {
					lg(LOG_ERR, "client #%d read: %s",
					    cl->n, strerror(errno));
					freeclient(cl);
				} else if (cc == 0) {
					lg(LOG_INFO,
					    "client #%d disconnected", cl->n);
					freeclient(cl);
				}
			}
		}

		/* Timeout clients due to waiting to login to the router */
		if (sp->state != ASTATE_LOGGEDIN &&
		    getastatesecs() > CHILD_REQUEST_TIMEOUT_SECS) {
			/* Trick clientsendallerr() into doing all clients */
			i = sp->multiball;
			sp->multiball = 0;
			clientsendallerr(
			    "timeout waiting to login to the router");
			sp->multiball = i;
		}

		/* Check for new clients */
		if (sp->s >= 0 && FD_ISSET(sp->s, &readfds))
			initiateclient(sp->s, CTYPE_PRIV);
		if (sp->s6 >= 0 && FD_ISSET(sp->s6, &readfds))
			initiateclient(sp->s6, CTYPE_PRIV);
		if (sp->ro >= 0 && FD_ISSET(sp->ro, &readfds))
			initiateclient(sp->ro, CTYPE_RO);
		if (sp->ro6 >= 0 && FD_ISSET(sp->ro6, &readfds))
			initiateclient(sp->ro6, CTYPE_RO);
		if (sp->web >= 0 && FD_ISSET(sp->web, &readfds))
			initiateclient(sp->web, CTYPE_WEB);
		if (sp->web6 >= 0 && FD_ISSET(sp->web6, &readfds))
			initiateclient(sp->web6, CTYPE_WEB);

		/* Did we connect to the child? */
		if (sp->wfd >= 0 && FD_ISSET(sp->wfd, &writefds) &&
		    sp->f_childconnect != NULL)
			(sp->f_childconnect)();

		/* Did we connect to the second child? */
		if (sp->chp2 != NULL)
			child2connect(&readfds, &writefds);

		/* Did the child say something? */
		if (sp->rfd >= 0 && FD_ISSET(sp->rfd, &readfds)) {
			cc = (sp->f_childread)(sp->rfd, &sp->rbuf);
			if (cc < 0) {
				lg(LOG_ERR, "child read: %s", strerror(errno));
				setastate(ASTATE_NOTCONNECTED);
				sp->cmd = NULL;
				(sp->f_childkill)();
			} else if (cc == 0) {
				lg(LOG_DEBUG, "child disconnect");
				setastate(ASTATE_NOTCONNECTED);
				sp->cmd = NULL;
				(sp->f_childkill)();
			}
		}

		/* Did the second child say something? */
		/* Second child read/write */
		if (sp->chp2 != NULL)
			child2poll(&readfds, &writefds);

		/* Process child input */
		done = 0;
		while (IOBUF_LEN(&sp->rbuf) > 0 && !done) {
			done = (sp->f_childinput)();

			/* Remove server request from list when completed */
			if (sp->req != NULL && sp->req->state == RSTATE_DONE) {
				/* Remove from request list */
				rp = sp->req;
				sp->req = rp->next;
				rp->next = NULL;
				freereq(rp);
			}
		}

		/* Process second child input */
		if (sp->chp2 != NULL)
			child2input();

		/*
		 * Process input from each client input, possibly
		 * appending a request to the end of the client's
		 * queue.
		 */
		for (cl = sp->clients; cl != NULL; cl = cl->next)
			while (iohaveline(&cl->rbuf))
				clientinput(cl);

		/* Did the replicant say anything? */
		if (sp->refd >= 0 && FD_ISSET(sp->refd, &readfds)) {
			timerset(&sp->t_re_timeout,
			    cf->c_replicant_session_secs);
			cc = ioread(sp->refd, &sp->re_rbuf);
			if (cc < 0) {
				lg(LOG_ERR, "replicant read: %s",
				    strerror(errno));
			} else if (cc == 0) {
				p = "replicant disconnect";
				lg(LOG_DEBUG, "%s", p);
				replicantterminate(p);
			}
		}

		/* Process replicant input */
		if (IOBUF_LEN(&sp->re_rbuf) > 0)
			replicantinput();

		/* Clean up completed requests */
		for (cl = sp->clients; cl != NULL; cl = cl->next) {
			lastrp = NULL;
			for (rp = cl->req; rp != NULL; rp = nextrp) {
				nextrp = rp->next;
				if (!REQUESTDONE(rp)) {
					lastrp = rp;
					continue;
				}

				/* Remove from request list */
				if (lastrp == NULL)
					cl->req = rp->next;
				else
					lastrp->next = rp->next;
				rp->next = NULL;
				freereq(rp);
			}
		}

		/* Drain replicant output */
		if (sp->refd >= 0 &&
		    FD_ISSET(sp->refd, &writefds) &&
		    IOBUF_LEN(&sp->re_wbuf) > 0 &&
		    iowrite(sp->refd, &sp->re_wbuf) < 0) {
			(void)snprintf(buf, sizeof(buf), "replicant write: %s",
			    strerror(errno));
			lg(LOG_ERR, "%s", buf);
			replicantterminate(buf);
		}

		/* Did the replicant go to sleep? */
		if (timerdue(&sp->t_re_timeout)) {
			p = "replicant timeout";
			lg(LOG_DEBUG, "%s", p);
			replicantterminate(p);
		}

		/* XXX todo: ping replicant when we ping the router? */

		/* Reset keepalive if necessary */
		if (sp->state == ASTATE_LOGGEDIN &&
		    cf->c_ayt_secs > 0 && timercheck(&sp->t_ayt) < 0)
			timerset(&sp->t_ayt, cf->c_ayt_secs);

		/* Handle server requests */
		while (sp->state == ASTATE_LOGGEDIN &&
		    (rp = sp->req) != NULL && rp->state == RSTATE_PENDING) {
			serverprocess(rp);
			if (rp->state == RSTATE_DONE) {
				/* Remove from request list */
				rp = sp->req;
				sp->req = rp->next;
				rp->next = NULL;
				freereq(rp);
			}
		}

		/* Handle client requests */
		if (sp->req == NULL)
			handleclientreqs();

		/* Send new replicant requests */
		if (sp->new_rereq != NULL)
			replicantprocess();

		/* Drain client output */
		for (cl = sp->clients; cl != NULL; cl = nextcl) {
			/* Ok to free clients inside this loop */
			nextcl = cl->next;
			if (cl->c >= 0 && FD_ISSET(cl->c, &writefds) &&
			    IOBUF_LEN(&cl->wbuf) > 0) {
				if (iowrite(cl->c, &cl->wbuf) < 0) {
					lg(LOG_ERR, "client #%d write: %s",
					    cl->n, strerror(errno));
					freeclient(cl);
					continue;
				}
				/* If this client is done free him */
				if (cl->close && IOBUF_LEN(&cl->wbuf) == 0) {
					freeclient(cl);
					continue;
				}
			}
		}
	}
}

struct req *
newreq(const char *what)
{
	struct req *rp;

	rp = new(1, sizeof(*rp), what);
	IOBUF_INIT(&rp->payload);
	return (rp);
}

int
opensocket(struct addr *addr, u_short port)
{
	int s, flags;
	socklen_t salen;
	struct sockaddr_storage sa;
	struct sockaddr *sap;
	static int on = 1;
	char buf[64];

	sap = (struct sockaddr *)&sa;
	salen = addr2sa(addr, port, sap);
	(void)snprintf(buf, sizeof(buf), "%s.%hu", addr2str(addr), port);

	s = socket(addr->family, SOCK_STREAM, 0);
	if (s < 0) {
		lg(LOG_ERR, "socket: %s: %s", buf, strerror(errno));
		exit(EX_OSERR);
	}

	/* Close on exec */
	if (fcntl(s, F_SETFD, FD_CLOEXEC) < 0) {
		lg(LOG_ERR, "opensocket: fcntl FD_CLOEXEC: %s: %s",
		    buf, strerror(errno));
		exit(EX_OSERR);
	}

	if (setsockopt(s, SOL_SOCKET, SO_REUSEADDR,
	    (char *)&on, sizeof (on)) < 0) {
		lg(LOG_ERR, "SO_REUSEADDR: %s: %s", buf, strerror(errno));
		exit(EX_OSERR);
	}
	if (bind(s, sap, salen) < 0) {
		lg(LOG_ERR, "bind: %s: %s", buf, strerror(errno));
		exit(EX_OSERR);
	}
	if (listen(s, SOMAXCONN) < 0) {
		lg(LOG_ERR, "listen: %s: %s", buf, strerror(errno));
		exit(EX_OSERR);
	}

	/* Enable non-blocking I/O */
	if ((flags = fcntl(s, F_GETFL, 0)) < 0) {
		lg(LOG_ERR, "nbio: F_GETFL: %s", strerror(errno));
		exit(EX_OSERR);
	}
	flags |= O_NONBLOCK;
	if ((flags = fcntl(s, F_SETFL, flags)) < 0) {
		lg(LOG_ERR, "nbio: F_SETFL: %s", strerror(errno));
		exit(EX_OSERR);
	}
	return (s);
}

int
reapdeadclients(void)
{
	int numbad;
	struct client *cl, *nextcl;

	numbad = 0;
	for (cl = state.clients; cl != NULL; cl = nextcl) {
		/* Ok to free clients inside this loop */
		nextcl = cl->next;
		if (cl->c < 0)
			continue;
		if (fcntl(cl->c, F_GETFL, 0) >= 0 || errno != EBADF)
			continue;
		lg(LOG_ERR, "reapdeadclients: reaping client #%d", cl->n);
		freeclient(cl);
		++numbad;
	}
	return (numbad);
}

void
setastate(enum acldstate astate)
{
	struct state *sp;

	/* Track logged in vs. not logged in */
	sp = &state;
	if ((sp->state == ASTATE_LOGGEDIN) ^ (astate == ASTATE_LOGGEDIN))
		sp->loginout_ts = time(NULL);
	sp->state = astate;
}

/* Set our select timeout */
void
settimeout(void)
{
	int t, secs;
	struct client *cl;
	struct req *rp;
	struct state *sp;

	sp = &state;
	secs = cf->c_select_secs;
	t = timercheck(&sp->t_login);
	if (t >= 0 && secs > t)
		secs = t;
	if (sp->chp2 != NULL) {
		t = child2settimeout(secs);
		if (t >= 0 && secs > t)
			secs = t;
	}
	t = timercheck(&sp->t_sync);
	if (t >= 0 && secs > t)
		secs = t;
	t = timercheck(&sp->t_compact);
	if (t >= 0 && secs > t)
		secs = t;
	t = timercheck(&sp->t_ayt);
	if (t >= 0 && secs > t)
		secs = t;
	t = timercheck(&sp->t_re_connect);
	if (t >= 0 && secs > t)
		secs = t;
	t = timercheck(&sp->t_re_timeout);
	if (t >= 0 && secs > t)
		secs = t;
	t = timercheck(&sp->t_child);
	if (t >= 0 && secs > t)
		secs = t;

	/* Don't screw around if we have things to do */
	if ((sp->state == ASTATE_LOGGEDIN || sp->state == ASTATE_SENTLISTACL) &&
	    (!sp->sentattrs || !sp->listedallacls || !sp->listedroutes))
		secs = 0;

	/* Don't delay if there are client requests we can process */
	if (sp->req == NULL && !childbusy()) {
		for (cl = sp->clients; cl != NULL; cl = cl->next) {
			rp = cl->req;
			if (rp != NULL && rp->state == RSTATE_PENDING) {
				secs = 0;
				break;
			}
		}
	}

	sp->timeout.tv_sec = secs;
	sp->timeout.tv_usec = 0;
}

/* Reap exiting child */
void
sigchild(int signo)
{
	DECLWAITSTATUS status;

	(void)waitpid(0, &status, WNOHANG);
}

void
sighup(int signo)
{

	++reload;
}

#ifdef notdef
void
sigpipe(int signo)
{
	struct state *sp;

	sp = &state;
	if (sp->c >= 0) {
		(void)close(sp->c);
		sp->c = -1;
	}
}
#endif

void
sigterm(int signo)
{
	struct state *sp;
	struct client *cl;

	sp = &state;
	for (cl = sp->clients; cl != NULL; cl = cl->next) {
		(void)close(cl->c);
		cl->c = -1;
	}

	++sawterm;
}

int
update_fd_set(fd_set *rfp, fd_set *wfp)
{
	int nfds;
	struct client *cl;
	struct state *sp;

	FD_ZERO(rfp);
	FD_ZERO(wfp);

	/* Privileged sockets */
	sp = &state;
	FD_SET(sp->s, rfp);
	nfds = sp->s;

	if (sp->s6 >= 0) {
		FD_SET(sp->s6, rfp);
		nfds = MAX(nfds, sp->s6);
	}

	if (sp->rfd >= 0) {
		FD_SET(sp->rfd, rfp);
		nfds = MAX(nfds, sp->rfd);
	}

	/* Only watch for child write when we're waiting to connect */
	if (sp->wfd >= 0 && sp->f_childconnect != NULL) {
		FD_SET(sp->wfd, wfp);
		nfds = MAX(nfds, sp->wfd);
	}

	/* Second child */
	if (sp->chp2 != NULL)
		nfds = child2update_fd_set(rfp, wfp, nfds);

	/* Replicant */
	if (sp->refd >= 0) {
		FD_SET(sp->refd, rfp);
		nfds = MAX(nfds, sp->refd);
		if (IOBUF_LEN(&sp->re_wbuf) > 0)
			FD_SET(sp->refd, wfp);
	}

	/* R/O sockets */
	if (sp->ro >= 0) {
		FD_SET(sp->ro, rfp);
		nfds = MAX(nfds, sp->ro);
	}

	if (sp->ro6 >= 0) {
		FD_SET(sp->ro6, rfp);
		nfds = MAX(nfds, sp->ro6);
	}

	/* Web registration sockets */
	if (sp->web >= 0) {
		FD_SET(sp->web, rfp);
		nfds = MAX(nfds, sp->web);
	}

	if (sp->web6 >= 0) {
		FD_SET(sp->web6, rfp);
		nfds = MAX(nfds, sp->web6);
	}

	/* Client sockets */
	for (cl = sp->clients; cl != NULL; cl = cl->next) {
		if (cl->c >= 0) {
			FD_SET(cl->c, rfp);
			nfds = MAX(nfds, cl->c);

			if (IOBUF_LEN(&cl->wbuf) > 0)
				FD_SET(cl->c, wfp);
		}
	}
	++nfds;
	return (nfds);
}

__dead void
usage(void)
{

	fprintf(stderr, "%s version %s\n", prog, version);
#ifdef TC_VERSION_STRING
	fputs(" with " TC_VERSION_STRING " (tcmalloc)\n", stderr);
#endif
	fprintf(stderr,
	    "Usage: %s [-dfv] [-c config] [-o logfile] [-P pidfile]\n", prog);
	exit(EX_USAGE);
}
