/* @(#) $Id$ (LBL) */
#ifndef acld_h
#define acld_h

#ifdef __FreeBSD__
#include <kvm.h>
#endif

#ifdef HAVE_LIBTCMALLOC
#include <gperftools/tcmalloc.h>
#endif

#include "cf.h"
#include "io.h"
#include "stats.h"
#include "mac.h"
#include "timer.h"
#include "version.h"

#ifndef MAX
#define MAX(x, y) ((x) > (y) ? (x) : (y))
#endif

/* Maximum number of "sent" replicant requests */
#define REPLICANT_DEPTH 100

/* acld states */
enum acldstate {
	ASTATE_NOTCONNECTED = 0,
	ASTATE_CONNECTED,
	ASTATE_CONNECTING,
	ASTATE_LOGGEDIN,
	ASTATE_READACL,
	ASTATE_READERROR,
	ASTATE_READRESPONSE,
	ASTATE_READROUTE,
	ASTATE_SENTATTR,
	ASTATE_SENTLISTACL,
	ASTATE_SENTLISTROUTE,
	ASTATE_SENTLOGIN,
};

#define ACLDREADY() \
    (state.state == ASTATE_LOGGEDIN && \
    state.sentattrs && state.listedroutes && state.listedallacls)

/* Both the req and the optional replicant request are done */
#define REQUESTDONE(rp) \
    ((rp) != NULL && \
    (rp)->state == RSTATE_DONE && \
    (rp)->child2notdone == 0 && \
    ((rp)->rereq == NULL || (rp)->rereq->state == RESTATE_DONE))

/* Seconds to wait for acld to login to the router before giving up */
#define CHILD_REQUEST_TIMEOUT_SECS	30

/* Seconds of client connection time to be considered persistent */
#define PERSISTENT_CLIENT_SECS 10

/* Seconds that a child is down before failing requests for it */
#define CHILD2_FAIL_SECS 10

/* Child */
struct child {
	char *name;			/* name of this child */
	int n;				/* child2 sequence number */
	enum acldstate state;		/* ASTATE_* */
	time_t loginout_ts;		/* time we last logged in or out */
	int numsends;			/* number of rest api requests sent */
	int rfd;			/* read descriptor */
	int wfd;			/* write descriptor */
	struct iobuf rbuf;		/* read buffer */
	struct iobuf wbuf;		/* write buffer */
	pid_t pid;			/* child pid */
	struct req *curreq;		/* current request */
	// XXX listed list of *server* requests?
	struct req *req;		/* linked list of requests */
	struct timer t_ayt;		/* timer for "ayt" */
	struct timer t_login;		/* timer for next login attempt */
	void *opaque;			/* child specific opaque data pointer */
	int eatresponse;		/* abort the in-progress request */

	/* Child hooks */
	void (*f_childayt)(struct child *);
	int (*f_childbusy)(struct child *);
	const char *(*f_childcheckreq)(struct req *);
	void (*f_childcommit)(struct child *);
	void (*f_childconnect)(struct child *);
	// void (*f_childprocess)(struct child *);
	int (*f_childinput)(struct child *);
	void (*f_childkill)(struct child *);
	void (*f_childlistacl)(struct child *);
	void (*f_childlistroute)(struct child *);
	void (*f_childlogin)(struct child *);
	int (*f_childread)(struct child *);
	int (*f_childready)(struct child *);
	void (*f_childrequest)(struct child *, struct aclgroup *, struct req *);
	void (*f_childsend)(struct child *, const char *, ...)
	    __attribute__ ((format (printf, 2, 3)));
	void (*f_childsendattr)(struct child *);
	int (*f_childwrite)(struct child *);
};

/* State */
struct state {
	enum acldstate state;		/* ASTATE_* */
	time_t loginout_ts;		/* time we last logged in or out */
	time_t epoc_ts;			/* time we started running */
	const char *cmd;		/* command sent */
	int s;				/* IPv4 listen socket */
	int s6;				/* IPv6 listen socket */
	int ro;				/* IPv4 listen socket (read only) */
	int ro6;			/* IPv6 listen socket (read only) */
	int web;			/* IPv4 listen socket (web reg) */
	int web6;			/* IPv6 listen socket (web reg) */
	int rfd;			/* expect/junoscript child read pipe */
	int wfd;			/* expect/junoscript child write pipe */
	int refd;			/* replicant socket */
	u_int32_t recookie;		/* replicant cookie */
	int re_secs;			/* replicant connect retry seconds */
	int replicantdepth;		/* number of outstanding requests */
	struct iobuf re_rbuf;		/* replicant read buffer */
	struct iobuf re_wbuf;		/* replicant write buffer */
	struct rereq *new_rereq;	/* new replicant requests */
	struct rereq *sent_rereq;	/* pending replicant requests */

#ifdef __FreeBSD__
	int routefd;			/* FreeBSD routing raw socket */
	kvm_t *kvmd;			/* FreeBSD kvm descriptor */
#endif
	pid_t pid;			/* child pid */
	u_int nclients;			/* next client number */

	/*
	 * Submit all ready requests to the child
	 * Commit when there are no ready requests left
	 */
	int multiball;

	int listedallacls;		/* have we listed all the acl's? */
	int listedroutes;		/* have we listed the routes? */
	int sentattrs;			/* have we sent the attributes? */

	struct aroute *routes;		/* list of routes from router */
	struct array routes_array;

	int nullzerolen;		/* current number of nullzero routes */
	struct stats nullzerostats;	/* nullzero statistics */
	struct stats filterstats;	/* filter statistics */

	struct timeval timeout;		/* select timeout */
	struct iobuf rbuf;		/* child read buffer */
	struct req *req;		/* linked list of server requests */
	struct client *clients;		/* linked list of clients */
	struct timer t_ayt;		/* timer for "ayt" */
	struct timer t_compact;		/* timer for "compact" */
	struct timer t_login;		/* timer for next login attempt */
	struct timer t_sync;		/* timer for next sync */
	struct timer t_re_connect;	/* timer for replicant connect */
	struct timer t_re_timeout;	/* timer for replicant session */
	struct timer t_child;		/* timer for generic child use */

	struct child *chp2;		/* Second children */
	struct array chp2_array;

	/* Child hooks */
	const char *(*f_childcheckreq)(struct req *);
	void (*f_childcommit)(void);
	void (*f_childconnect)(void);
	int (*f_childinput)(void);
	void (*f_childkill)(void);
	void (*f_childlistacl)(void);
	void (*f_childlistroute)(void);
	void (*f_childlogin)(void);
	int (*f_childread)(int, struct iobuf *);
	int (*f_childready)(void);
	void (*f_childrequest)(struct aclgroup *, struct req *);
	void (*f_childsend)(const char *, ...)
	    __attribute__ ((format (printf, 1, 2)));
	void (*f_childsendattr)(void);

	/* Child2 hooks */
	void (*f_child2process)(void);

	/* Client hooks */
	void (*f_clientcompact)(struct client *cl, struct req *rp);
	void (*f_clientdroprestore)(struct client *, struct req *);
	void (*f_clientfilter)(struct client *, struct req *);
	void (*f_clientlistroute)(struct client *, struct req *, int);
	const char *(*f_clientmapaclname)(const char *, struct acl *);
	void (*f_clientnullzero)(struct client *, struct req *);
	void (*f_clientprefix)(struct client *, struct req *);
#ifdef notdef
	void (*f_clientqueryfilter)(struct client *, struct req *);
#endif
	void (*f_clientquerynullzero)(struct client *, struct req *);

	/* Server hooks */
	void (*f_serverayt)(void);
	void (*f_servercompact)(void);
	void (*f_serversync)(void);

	/* Second client hooks */
	void (*f_clientlistroute2)(struct client *, struct req *, int);
	void (*f_clientnullzero2)(struct client *, struct req *);
	void (*f_clientquerynullzero2)(struct client *, struct req *);
};

extern struct state state;

#include "acl.h"
#include "client.h"
#include "route.h"

#include "whitelist.h"

/* Request types */
enum reqtype {
	REQ_UNKNOWN = 0,
	REQ_ADDMACWHITELIST,	/* add whitelist mac */
	REQ_ADDWHITELIST,	/* add whitelist address */
	REQ_AYT,		/* check that router is still alive */
	REQ_BLOCKHOSTHOST,	/* block a host to host pair */
	REQ_BLOCKMAC,		/* block a mac address */
	REQ_COMPACT,		/* schedule compaction of specified acl */
	REQ_DROP,		/* block a host or subnet */
	REQ_DROPTCPDSTHOSTPORT,	/* block tcp dst-host/port */
	REQ_DROPTCPPORT,	/* block a tcp port */
	REQ_DROPTCPSYNPORT,	/* block a tcp syn port */
	REQ_DROPUDPPORT,	/* block a udp port */
	REQ_EXIT,		/* terminate client session */
	REQ_FILTER,		/* add a corsa filter address */
	REQ_HELP,		/* display brief help message */
	REQ_LISTACL,		/* list contents of specified acl */
	REQ_LISTMAC,		/* list contents of the mac block acl */
	REQ_LISTNULLZERO,	/* list nullzero routes */
	REQ_LISTPREFIX,		/* list addresses on a prefix list */
	REQ_LISTROUTE,		/* list routes (with route type character) */
	REQ_MACWHITELIST,	/* list contents of mac block whitelist */
	REQ_NOFILTER,		/* remove corsa filter address */
	REQ_NONULLZERO,		/* remove null zero host route */
	REQ_NOPREFIX,		/* remove an address from a prefix list */
	REQ_NULLZERO,		/* create null zero host route */
	REQ_PERMITTCPDSTHOSTPORT, /* permit tcp dst-host/port */
	REQ_PERMITUDPDSTHOSTPORT, /* permit udp dst-host/port */
	REQ_PREFIX,		/* add an address to a prefix list */
	REQ_QUERY,		/* check host, host to host or subnet block */
	REQ_QUERYFILTER,	/* check for a corsa filter address */
	REQ_QUERYMAC,		/* check for mac address blocks */
	REQ_QUERYMACWHITELIST,	/* check for mac address blocks */
	REQ_QUERYNULLZERO,	/* check for a nullzero host route */
	REQ_QUERYWHITELIST,	/* check for a whitelist host/net */
	REQ_RELOAD,		/* kill expect child process & reload config */
	REQ_REMMACWHITELIST,	/* remove whitelist mac */
	REQ_REMWHITELIST,	/* remove whitelist address */
	REQ_REPLICANT,		/* register as a replicant provider */
	REQ_RESTORE,		/* unblock a host or subnet */
	REQ_RESTOREHOSTHOST,	/* unblock a host to host pair */
	REQ_RESTOREMAC,		/* unblock a mac address */
	REQ_RESTORETCPDSTHOSTPORT, /* remove tcp dst-host/port */
	REQ_RESTORETCPPORT,	/* unblock a tcp port */
	REQ_RESTORETCPSYNPORT,	/* unblock a tcp syn port */
	REQ_RESTOREUDPPORT,	/* unblock a udp port */
	REQ_STATE,		/* display current acld state */
	REQ_SYNC,		/* copy router config to nonvolatile memory */
	REQ_TEST,		/* (undocumented) */
	REQ_UNPERMITTCPDSTHOSTPORT, /* remove permit tcp dst-host/port */
	REQ_UNPERMITUDPDSTHOSTPORT, /* remove permit udp dst-host/port */
	REQ_WHITELIST,		/* list whitelist */
};

/* Include after route.h and enum reqtype */
#include "replicant.h"

/* Request states */
enum reqstate {
	RSTATE_PENDING = 0,	/* not processed yet */
	RSTATE_CHILD,		/* waiting for child to reply */
//	RSTATE_CHILD2,		/* used for child2 request processing */
	RSTATE_DONE,		/* ok to remove from queue */
//	RSTATE_DONE2,		/* used for child2 request processing */
	RSTATE_READCOMMENT,	/* reading client comments */
	RSTATE_READRESPONSE,	/* reading child response */
	RSTATE_PENDING2,	/* used for child2 request processing */
};

/* Request flags */
enum reqflags {
	RFLAG_CONTINUE	= 0x1,	/* request has a payload */
	RFLAG_FAILED	= 0x2,	/* request resulted in a failed status */
	RFLAG_IGNORE	= 0x4,	/* request didn't result in any changes */
	RFLAG_CONSOLIDATE = 0x8, /* consolidate payload messages */
};

/* Request */
struct req {
	struct req *next;
	struct rereq *rereq;		/* link to replicant request */
	enum reqtype type;		/* REQ_* */
	enum reqstate state;		/* RSTATE_* */
	enum reqflags flags;		/* request flags */
	const char *cmd;		/* client command name */
	struct aclgroup *aclgroup;	/* aclgroup pointer */
	char *aclname;			/* listacl, compact, drop udp/tcp ACL */
	char *prefixname;		/* prefix, noprefix and listprefix */
	u_int32_t cookie;
	u_int32_t multiball;		/* multiball request */
	struct timeval ats;		/* arrival timestamp */
	struct timeval cts;		/* completion timestamp */
	int retries;			/* number of retries */
	int child2pending;		/* child2 pending */
	int child2notdone;		/* child2 not done */
	int child2num;			/* total number of child2 children */
	u_int32_t child2results;	/* bitmask of child2 QUERY* results */
	int an;				/* argument count */
	char **av;			/* argument vector */
	char *comment;
	struct iobuf payload;
	struct aroute nullzero;
	struct addr filter;
	struct addr prefix;
	struct addr whitelist;
	struct acl acl;
};

#define LOGIN_SECS		15
#define LOGIN_SECS2		LOGIN_SECS
#define AYT_SECS		(10 * 60)
#define SYNC_SECS		(30 * 60)
#define SELECT_SECS		15
#define REPLICANT_SESSION_SECS	45	/* half of acl.exp's timeout */

/* Must be greater than SELECT_SECS */
#define IDLE_SECS		(2 * SELECT_SECS)

#define MAX_IPV4_WIDTH		16	/* widest IPv4 acl block */
#define MAX_IPV6_WIDTH		32	/* widest IPv6 acl block */

#define DEFAULT_IPV4_WIDTH	24	/* default IPv4 acl width */
#define DEFAULT_IPV6_WIDTH	64	/* default IPv6 acl width */

#define MAX_IPV4_NULLZERO_WIDTH	24	/* widest IPv4 nullzero block */
#define MAX_IPV6_NULLZERO_WIDTH	64	/* widest IPv6 nullzero block */

#define DEFAULT_IPV4_NULLZERO_WIDTH 32	/* default IPv4 acl width */
#define DEFAULT_IPV6_NULLZERO_WIDTH 128	/* default IPv6 acl width */

#define DEFAULT_RETRIES		5	/* per request/child */

/* https://docs.freebsd.org/info/cpp/cpp.info.Stringification.html */
#define _HERESTR1(fn, ln) _HERESTR2(fn, ln)
#define _HERESTR2(fn, ln) fn ":" #ln
#define HERESTR _HERESTR1(__FILE__, __LINE__)

/* Globals */
extern int debug;
extern int verbose;
extern int reload;
extern const char *prog;
extern struct s2v astate2str[];
extern struct s2v rstate2str[];
extern const char *configfn;
extern const char *logfile;
extern char hostname[];

/* Functions */
extern void appendreq(struct req **, struct req *);
extern void freereq(struct req *);
extern int getastatesecs(void);
extern struct client *getreqclient(struct req *);
extern struct req *newreq(const char *);
extern void setastate(enum acldstate);
#endif
