/*
 * Copyright (c) 2002, 2003, 2004, 2006, 2008, 2009, 2010, 2011, 2012, 2013, 2014, 2015, 2016, 2018, 2019, 2020, 2021, 2022, 2023, 2024
 * The Regents of the University of California. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *     * Redistributions of source code must retain the above copyright
 *       notice, this list of conditions and the following disclaimer.
 *     * Redistributions in binary form must reproduce the above copyright
 *       notice, this list of conditions and the following disclaimer in the
 *       documentation and/or other materials provided with the distribution.
 *     * Neither the name of the University nor the names of its contributors
 *       may be used to endorse or promote products derived from this software
 *       without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#ifndef lint
static const char rcsid[] __attribute__ ((unused)) =
    "@(#) $Id$ (LBL)";
#endif

#include <sys/types.h>
#include <sys/time.h>

#include <arpa/inet.h>

#include <ctype.h>
#include <errno.h>
#ifdef HAVE_JUNOSCRIPT
#include <netdb.h>
#endif
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sysexits.h>

#include "acld.h"
#include "cf.h"
#ifdef HAVE_CORSA
#include "corsa.h"
#endif
#include "prefix.h"

/* Globals */
struct cf *cf;

/* Locals */
static int errors;

/* Forwards */
static void acllimitadd(const char *);
static void acllimitfree(struct acllimit *);
static void freeone(char **);
static void intlistfree(struct intlist *);
static int readcf(const char *, int);

static void
acllimitadd(const char *str)
{
	long v;
	int an;
	char **av;
	char *cp;
	struct acllimit *alp;
	struct array *dp;

	v = strtol(str, &cp, 10);
	if (!isspace(*cp)) {
		lg(LOG_ERR, "acllimitadd: trailing garbage (%s)", str);
		++errors;
	}
	if (v <= 0) {
		lg(LOG_ERR, "acllimitadd: bad value (%ld)", v);
		++errors;
	}

	++cp;
	while (isspace(*cp))
		++cp;

	/* Make a list of ACL names */
	an = makeargv(cp, &av);
	if (an <= 0) {
		lg(LOG_ERR, "acllimitadd: missing ACL name(s) \"%s\"", str);
		return;
	}

	dp = &cf->c_acllimit_array;
	DYNARRAY(dp, cf->c_acllimit, dp->len + 1, "cf acllimit");
	alp = cf->c_acllimit + dp->len;
	++dp->len;

	alp->a_maxseq = v;
	alp->a_nacls = an;
	alp->a_acls = av;
}

static void
acllimitfree(struct acllimit *alp)
{

	if (alp->a_acls != NULL) {
		freeargv(alp->a_acls);
		alp->a_acls = NULL;
		alp->a_nacls = 0;
	}
}

void
freecf(void)
{
	int i;
	struct aclgroup *gp;
	struct acllimit *alp;
	struct intlist *ip;
	char **prefixes;
	struct array *dp;

	/* Single line options */
#ifdef HAVE_CFORCE
	freeone(&cf->c_cforceaddr);
#endif
	freeone(&cf->c_cuser);
	freeone(&cf->c_cpass1);
	freeone(&cf->c_cpass2);
	freeone(&cf->c_euser);
	freeone(&cf->c_epass1);
	freeone(&cf->c_epass2);
#ifdef __FreeBSD__
	freeone(&cf->c_nullzeroif);
#endif
#ifdef HAVE_JUNOSCRIPT
	freeone(&cf->c_sshkeyfn);
	freeone(&cf->c_sshpubfn);
	freeone(&cf->c_junoscriptlog);
#endif
#ifdef HAVE_CORSA
	freeone(&cf->c_corsacacert);
	freeone(&cf->c_corsalog);
#endif
	freeone(&cf->c_expect);
	freeone(&cf->c_expectlog);
	freeone(&cf->c_router);
	freeone(&cf->c_script);
	freeone(&cf->c_id);
	freeone(&cf->c_device);
#ifdef HAVE_CFORCE
	freeone(&cf->c_cforceaddr);
	freeone(&cf->c_cforcedata);
#endif
	freeone(&cf->c_macwhitelistfn);
	freeone(&cf->c_whitelistfn);

	/* ACL limits */
	dp = &cf->c_acllimit_array;
	for (i = 0, alp = cf->c_acllimit; i < dp->len; ++i, ++alp)
		acllimitfree(alp);
	FREEDYNARRAY(dp, cf->c_acllimit);

	/* ACLs */
	dp = &cf->c_aclgroup_array;
	for (i = 0, gp = cf->c_aclgroup; i < dp->len; ++i, ++gp)
		aclfree(gp);
	FREEDYNARRAY(dp, cf->c_aclgroup);

	/* Interfaces */
	dp = &cf->c_intlist_array;
	for (i = 0, ip = cf->c_intlist; i < dp->len; ++i, ++ip)
		intlistfree(ip);
	FREEDYNARRAY(dp, cf->c_intlist);

	/* Null zero routes */
	dp = &cf->c_nullzeronets_array;
	FREEDYNARRAY(dp, cf->c_nullzeronets);

	/* Prefix names */
	dp = &cf->c_prefixes_array;
	for (i = 0, prefixes = cf->c_prefixes; i < dp->len; ++i, ++prefixes)
		free(*prefixes);
	FREEDYNARRAY(dp, cf->c_prefixes);

	/* Whitelists */
	FREEDYNARRAY(&cf->c_macwhitelist_array, cf->c_macwhitelist);
	FREEDYNARRAY(&cf->c_whitelist_array, cf->c_whitelist);

#ifdef HAVE_CORSA
	FREEDYNARRAY(&cf->c_corsaservers_array, cf->c_corsaservers);
	FREEDYNARRAY(&cf->c_corsalocalnets_array, cf->c_corsalocalnets);
#endif

	free(cf);
	cf = NULL;
}

static void
freeone(char **pp)
{
	if (*pp != NULL) {
		free(*pp);
		*pp = NULL;
	}
}

void
intlistadd(const char *str)
{
	struct intlist *ip;
	struct array *dp;
	int an, i;
	char **av;
	char buf[1024];
	char *cp;
	size_t size, len;

	an = makeargv(str, &av);
	if (an < 2) {
		lg(LOG_ERR, "intlistadd: wrong number of args: \"%s\"", str);
		freeargv(av);
		return;
	}

	dp = &cf->c_intlist_array;
	DYNARRAY(dp, cf->c_intlist, dp->len + 1, "intlistadd");
	ip = cf->c_intlist + dp->len;
	++dp->len;
	ip->i_acl = strsave(av[an - 1]);

	size = sizeof(buf);
	cp = buf;
	for (i = 0; i < an - 1; ++i) {
		(void)snprintf(cp, size, " %s", av[i]);
		len = strlen(cp);
		cp += len;
		size -= len;
	}
	cp = buf + 1;
	ip->i_name = strsave(cp);

	freeargv(av);
}

static void
intlistfree(struct intlist *ip)
{

	if (ip->i_acl != NULL) {
		free(ip->i_acl);
		ip->i_acl = NULL;
	}
	if (ip->i_name != NULL) {
		free(ip->i_name);
		ip->i_name = NULL;
	}
}

void
parsecf(const char *fn)
{
	int i, j, n, seqrange, portseqrange, permithostportseqrange;
	char *cp;
	const char *errmsg;
	struct acllimit *alp;
	struct aclgroup *gp;
	struct intlist *ip;
	struct array *dp, *dp2;
#ifdef HAVE_JUNOSCRIPT
	const char *p;
	struct servent *sp;
#endif

	if (cf != NULL)
		free(cf);
	cf = (struct cf *)new(1, sizeof(*cf), "parsecf cf");

	/* Defaults */
	cf->c_ayt_secs = AYT_SECS;
	cf->c_login_secs = LOGIN_SECS;
	cf->c_login_secs2 = LOGIN_SECS2;
	cf->c_select_secs = SELECT_SECS;
	cf->c_sync_secs = SYNC_SECS;
	cf->c_replicant_session_secs = REPLICANT_SESSION_SECS;
	cf->c_incrseq = 1;
	cf->c_lowseq = -1;
	cf->c_highseq = -1;
	cf->c_lowportseq = -1;
	cf->c_highportseq = -1;
	cf->c_lowpermithostportseq = -1;
	cf->c_highpermithostportseq = -1;
	cf->c_retries = DEFAULT_RETRIES;

	cf->c_ipv4_maxwidth = DEFAULT_IPV4_WIDTH;
	cf->c_ipv6_maxwidth = DEFAULT_IPV6_WIDTH;
	cf->c_ipv4_maxnullzerowidth = DEFAULT_IPV4_NULLZERO_WIDTH;
	cf->c_ipv6_maxnullzerowidth = DEFAULT_IPV6_NULLZERO_WIDTH;
#ifdef HAVE_JUNOSCRIPT
	cf->c_portssh = -1;
#endif

	/* dynarray foo */
	cf->c_acllimit_array.osize = sizeof(*cf->c_acllimit);
	cf->c_aclgroup_array.osize = sizeof(*cf->c_aclgroup);
	cf->c_intlist_array.osize = sizeof(*cf->c_intlist);
	cf->c_nullzeronets_array.osize = sizeof(*cf->c_nullzeronets);
	cf->c_prefixes_array.osize = sizeof(*cf->c_prefixes);
	cf->c_macwhitelist_array.osize = sizeof(*cf->c_macwhitelist);
	cf->c_whitelist_array.osize = sizeof(*cf->c_whitelist);
	cf->c_whitelist_array.inclen = 1024;
#ifdef HAVE_CORSA
	cf->c_corsaservers_array.osize = sizeof(*cf->c_corsaservers);
	cf->c_corsalocalnets_array.osize = sizeof(*cf->c_corsalocalnets);
#endif

	errmsg = extractaddr("127.0.0.1", NULL, &cf->c_bindaddr);
	if (errmsg != NULL)
		abort();
	errmsg = extractaddr("::1", NULL, &cf->c_bindaddr6);
	if (errmsg != NULL)
		abort();

	errors += readcf(fn, 1);

	/* Check for a conflicting config */
	n = 0;

	/* Expect script */
	if (cf->c_expect != NULL) {
		++n;
		if ((cp = "script", cf->c_script == NULL) ||
		    (cp = "router", cf->c_router == NULL)) {
			lg(LOG_ERR, "error: \"%s\" missing but required", cp);
			++errors;
		}
	}

	/* Voight-Kampff tests */
	if (cf->c_replicate_acl ||
	    cf->c_replicate_nullzero ||
	    cf->c_replicate_prefix ||
	    cf->c_replicate_whitelist) {
		if (cf->c_portreplicant <= 0) {
			lg(LOG_ERR,
			    "error: portreplicant required with replicate");
			++errors;
		}
		if (cf->c_replicantaddr.family == 0) {
			lg(LOG_ERR,
			    "error: replicantaddr required with replicate");
			++errors;
		}
	} else {
		if (cf->c_portreplicant > 0) {
			lg(LOG_ERR, "error: portreplicant specificied"
			    " without replicate");
			++errors;
		}
		if (cf->c_replicantaddr.family != 0) {
			lg(LOG_ERR, "error: replicantaddr specificied"
			    " without replicate");
			++errors;
		}
	}

	if (cf->c_portreplicant > 0 && cf->c_replicantaddr.family == 0) {
		lg(LOG_ERR,
		    "error: replicantaddr required with portreplicant");
		++errors;
	}

#ifdef __FreeBSD__
	if (((cf->c_nullzeroaddr.family == 0) ^
	    (cf->c_nullzeroaddr6.family == 0)) != 0) {
		lg(LOG_ERR, "error: both \"nullzeroaddr\" and"
		    " \"nullzeroaddr6\" must be specified");
		    ++errors;
	}

	if (cf->c_nullzeroaddr.family == 0 &&
	    cf->c_nullzeroaddr6.family == 0) {
		if (cf->c_freebsdroutes) {
			/* Default nullzero interface name */
			if (cf->c_nullzeroif == NULL)
				cf->c_nullzeroif = strsave("lo0");
		} else {
			if (cf->c_nullzeroif != NULL) {
				lg(LOG_ERR, "error: can't use \"nullzeroif\""
				    " without \"freebsdroutes\"");
				    ++errors;
			}
		}

	} else if (cf->c_nullzeroif != NULL) {
		if (cf->c_freebsdroutes)
			lg(LOG_ERR, "error: can't use \"nullzeroif\""
			    " with \"nullzeroaddr\" or \"nullzeroaddr6\"");
		else
			lg(LOG_ERR, "error: can't use \"nullzeroif\""
			    " without \"freebsdroutes\"");
		++errors;
	}
#endif

#ifdef HAVE_CFORCE
	/* cForce appliance */
	if (cf->c_cforceaddr != NULL) {
		++n;
		if (cf->c_portcforce <= 0) {
			lg(LOG_ERR, "error: \"%s\" missing but required", cp);
			++errors;
		}

		/* cForce: sequence numbers are assigned by the appliance */
		if (cf->c_lowseq >= 0 || cf->c_highseq >= 0) {
			lg(LOG_ERR, "error: cForce does not use seqrange");
			++errors;
		}

		/* cForce: nullzero routes are not support */
		if (cf->c_nullzeronets != NULL || cf->c_nullzeromax != 0) {
			lg(LOG_ERR,
			    "error: cForce does not support nullzero routes");
			++errors;
		}
		if (!cf->c_incrseq) {
			lg(LOG_ERR, "error: cForce does not use incrseq");
			++errors;
		}
		if (!cf->c_sync_secs > 0) {
			lg(LOG_ERR, "error: cForce does not use sync_secs");
			++errors;
		}

		/* Default cforce mmap() file */
		if (cf->c_cforcedata == NULL)
			cf->c_cforcedata = "/var/db/cforce.data";
	}
#endif

#ifdef HAVE_JUNOSCRIPT
	if (cf->c_junoscript) {
		++n;
		if (cf->c_sshkeyfn == NULL) {
			lg(LOG_ERR, "error: junoscript requires sshkeyfn");
			++errors;
		}
		if (cf->c_sshpubfn == NULL) {
			lg(LOG_ERR, "error: junoscript requires sshpubfn");
			++errors;
		}
		if (cf->c_portssh < 0) {
			p = "ssh";
			sp = getservbyname(p, "tcp");
			if (sp != NULL)
				cf->c_portssh = ntohs(sp->s_port);
			else {
				lg(LOG_ERR,
				    "error: can't determine the %s port", p);
				++errors;
			}
		} else if (!VALID_PORT(cf->c_portssh)) {
			lg(LOG_ERR, "error: \"portssh\" is out of range");
			++errors;
		}

		if (cf->c_intlist_array.len > 0) {
			lg(LOG_ERR,
			    "error: \"interface\" not used with junoscript");
			++errors;
		}
	} else {
#endif
		if (cf->c_intlist_array.len == 0) {
			lg(LOG_ERR,
			    "error: \"interface\" missing but required");
			++errors;
		}
		if (cf->c_prefixes_array.len != 0) {
			lg(LOG_ERR,
			    "error: \"prefix\" specified but not supported");
			++errors;
		}
#ifdef HAVE_JUNOSCRIPT
	}
#endif

#ifdef HAVE_CORSA
	if (cf->c_corsa) {
		if (cf->c_corsaservers == NULL) {
			lg(LOG_ERR,
			    "error: corsa requires at least one corsaserver");
			++errors;
		}
		if (cf->c_corsalocalnets == NULL) {
			lg(LOG_ERR, "error: corsa requires corsalocalnet");
			++errors;
		}
	}
#endif

	j = 0;
	dp = &cf->c_aclgroup_array;
	for (i = 0, gp = cf->c_aclgroup; i < dp->len; ++i, ++gp)
		if (gp->ismac)
			++j;
	if (j > 1) {
		lg(LOG_ERR, "error: At most one mac address block acl"
		    " may be configured");
		++errors;
	}

	if (cf->c_mockclient)
		++n;

	/* Need example one control method */
	if (n == 0) {
		lg(LOG_ERR, "error: No control method configured");
		++errors;
	} else if (n > 1) {
		lg(LOG_ERR, "error: Only one control method may be configured");
		++errors;
	}

	if (cf->c_aclgroup == NULL) {
		lg(LOG_ERR, "error: \"acl\" missing but required");
		++errors;
	}

	if (cf->c_ipv4_maxwidth < MAX_IPV4_WIDTH) {
		lg(LOG_ERR, "error: \"ipv4_maxwidth\" must > %d",
		    MAX_IPV4_WIDTH);
		++errors;
	} else if (cf->c_ipv4_maxwidth > 32) {
		lg(LOG_ERR, "error: \"ipv4_maxwidth\" must <= 32");
		++errors;
	}

	if (cf->c_ipv6_maxwidth < MAX_IPV6_WIDTH) {
		lg(LOG_ERR, "error: \"ipv6_maxwidth\" must > %d",
		    MAX_IPV6_WIDTH);
		++errors;
	} else if (cf->c_ipv6_maxwidth > 128) {
		lg(LOG_ERR, "error: \"ipv6_maxwidth\" must <= 128");
		++errors;
	}

	if (cf->c_ipv4_maxnullzerowidth < MAX_IPV4_NULLZERO_WIDTH) {
		lg(LOG_ERR, "error: \"ipv4_maxnullzerowidth\" must > %d",
		    MAX_IPV4_WIDTH);
		++errors;
	} else if (cf->c_ipv4_maxnullzerowidth > 32) {
		lg(LOG_ERR, "error: \"ipv4_maxnullzerowidth\" must <= 32");
		++errors;
	}

	if (cf->c_ipv6_maxnullzerowidth < MAX_IPV6_NULLZERO_WIDTH) {
		lg(LOG_ERR, "error: \"ipv6_maxnullzerowidth\" must > %d",
		    MAX_IPV6_WIDTH);
		++errors;
	} else if (cf->c_ipv6_maxnullzerowidth > 128) {
		lg(LOG_ERR, "error: \"ipv6_maxnullzerowidth\" must <= 128");
		++errors;
	}

	if ((cp = "login_secs", cf->c_login_secs <= 0) ||
	    (cp = "login_secs2", cf->c_login_secs2 <= 0) ||
	    (cp = "port", cf->c_port <= 0) ||
	    (cp = "replicant_session_secs",
		cf->c_replicant_session_secs <= 0) ||
	    (cp = "select_secs", cf->c_select_secs <= 0)) {
		lg(LOG_ERR, "error: \"%s\" must be non-zero", cp);
		++errors;
	}

	if ((cp = "port", !VALID_PORT(cf->c_port)) ||
	    (cp = "portreplicant", !VALID_PORT(cf->c_portreplicant) &&
	    cf->c_portreplicant != 0) ||
	    (cp = "portro", !VALID_PORT(cf->c_portro) && cf->c_portro != 0) ||
	    (cp = "portweb", !VALID_PORT(cf->c_portweb) &&
	    cf->c_portweb != 0)) {
		lg(LOG_ERR, "error: \"%s\" is out of range", cp);
		++errors;
	}

	if ((cf->c_portro > 0 && (cp = "portro", cf->c_portro == cf->c_port)) ||
	    (cf->c_portweb > 0 && (cp = "portweb",
		cf->c_portweb == cf->c_port))) {
		lg(LOG_ERR, "error: \"%s\" can't be the same as \"port\"", cp);
		++errors;
	}
	if (cf->c_portro > 0 && cf->c_portweb > 0 &&
	    cf->c_portro == cf->c_portweb) {
		lg(LOG_ERR,
		    "error: \"portro\" can't be the same as \"portweb\"");
		++errors;
	}

	seqrange = (cf->c_lowseq >= 0 || cf->c_highseq >= 0);
	portseqrange = (cf->c_lowportseq > 0 || cf->c_highportseq > 0);
	permithostportseqrange = (cf->c_lowpermithostportseq >= 0 ||
	    cf->c_highpermithostportseq >= 0);
#ifdef HAVE_JUNOSCRIPT
	if (cf->c_junoscript) {
		if (seqrange || portseqrange || permithostportseqrange) {
			lg(LOG_ERR, "error: junoscript doesn't use seqrange"
			    ", portseqrange and permithostportseqrange");
			++errors;
		}
	}
#endif

	if (seqrange) {
		if (cf->c_lowseq >= cf->c_highseq) {
			lg(LOG_ERR, "error: \"seqrange\" must be increasing");
			++errors;
		}
		if (portseqrange && (IN_SEQRANGE(cf->c_lowportseq) ||
		    IN_SEQRANGE(cf->c_highportseq))) {
			lg(LOG_ERR, "error: \"seqrange\" and"
			" \"portseqrange\" overlap");
			++errors;
		}
	}

	if (portseqrange) {
		if (cf->c_lowportseq >= cf->c_highportseq) {
			lg(LOG_ERR,
			    "error: \"portseqrange\" must be increasing");
			++errors;
		}
		if (permithostportseqrange &&
		    (IN_PERMITHOSTPORTSEQRANGE(cf->c_lowseq) ||
		    IN_PERMITHOSTPORTSEQRANGE(cf->c_highseq))) {
			lg(LOG_ERR, "error: \"portseqrange\" and"
			    " \"permithostportseqrange\" overlap");
			++errors;
		}
	}

	if (permithostportseqrange) {
		if (cf->c_lowpermithostportseq >= cf->c_highpermithostportseq) {
			lg(LOG_ERR, "error: \"permithostportseqrange\" must"
			" be increasing");
			++errors;
		}
		if (seqrange && (IN_SEQRANGE(cf->c_lowpermithostportseq) ||
		    IN_SEQRANGE(cf->c_highpermithostportseq))) {
			lg(LOG_ERR, "error: \"seqrange\" and"
			    " \"permithostportseqrange\" overlap");
			++errors;
		}
	}
	if (cf->c_port != 0 && !VALID_PORT(cf->c_port)) {
		lg(LOG_ERR, "error: bad \"port\" port value");
		++errors;
	}
	if (cf->c_portro != 0 && !VALID_PORT(cf->c_portro)) {
		lg(LOG_ERR, "error: bad \"portro\" port value");
		++errors;
	}
	if (cf->c_portweb != 0 && !VALID_PORT(cf->c_portweb)) {
		lg(LOG_ERR, "error: bad \"portweb\" port value");
		++errors;
	}

	if (cf->c_retries <= 0) {
		lg(LOG_ERR, "error: \"retries\" must > 0");
		++errors;
	}

#define ESCAPESTR(str) { \
    cp = str; \
    str = strsave(escapestr(cp)); \
    if (cp != NULL) \
	    free(cp); \
    }

	ESCAPESTR(cf->c_cuser);
	ESCAPESTR(cf->c_cpass1);
	ESCAPESTR(cf->c_cpass2);
	ESCAPESTR(cf->c_euser);
	ESCAPESTR(cf->c_epass1);
	ESCAPESTR(cf->c_epass2);

	/* Hook interfaces into ACLs */
	dp = &cf->c_aclgroup_array;
	for (i = 0, gp = cf->c_aclgroup; i < dp->len; ++i, ++gp) {
		dp2 = &cf->c_intlist_array;
		for (j = 0, ip = cf->c_intlist; j < dp2->len; ++j, ++ip)
			if (strcmp(gp->name, ip->i_acl) == 0) {
				gp->intlist = ip;
				break;
			}
	}

#ifdef HAVE_JUNOSCRIPT
	if (!cf->c_junoscript) {
#endif
		dp = &cf->c_aclgroup_array;
		for (i = 0, gp = cf->c_aclgroup; i < dp->len; ++i, ++gp) {
			if (gp->intlist == NULL && !gp->ismac) {
				lg(LOG_ERR, "error: ACL %s missing interface",
				    gp->name);
				++errors;
			}
		}
#ifdef HAVE_JUNOSCRIPT
	}
#endif

	/* Hook ACL limits into ACLs */
	dp = &cf->c_acllimit_array;
	for (i = 0, alp = cf->c_acllimit; i < dp->len; ++i, ++alp)
		for (j = 0; j < alp->a_nacls; ++j) {
			gp = aclfindgroupbyname(alp->a_acls[j]);
			if (gp == NULL) {
				lg(LOG_ERR, "error: ACL %s not defined",
				    alp->a_acls[j]);
				++errors;
				continue;
			}
			if (gp->limitp != NULL) {
				lg(LOG_ERR, "error: ACL %s already limited",
				    gp->name);
				++errors;
				continue;
			}
			gp->limitp = alp;
		}

	/* If any ACLs are limited, check for unlimited */
	if (cf->c_acllimit_array.len > 0) {
		dp = &cf->c_aclgroup_array;
		for (i = 0, gp = cf->c_aclgroup; i < dp->len; ++i, ++gp)
			if (gp->limitp == NULL) {
				lg(LOG_ERR, "error: ACL %s is unlimited",
				    gp->name);
				++errors;
			}
	}

	/* Whitelist */
	macwhitelistread();
	whitelistread();

	if (errors > 0)
		exit(EX_CONFIG);
}

/* Open, read and parse one config file */
static int
readcf(const char *fn, int mustexist)
{
	int i, n, an, errs, *ip, *ip2;
	char *cp, *cp2, **av, **pp;
	const char *errmsg;
	time_t t;
	FILE *f;
	struct addr *a;
#ifdef HAVE_CORSA
	struct array *dp;
#endif
	struct aclgroup *gp;
	struct stat sbuf;
	char buf[1024];
	char at[1024];
#ifdef HAVE_CORSA
	struct corsa *csp;
#endif

	errs = 0;
	f = fopen(fn, "r");
	if (f == NULL) {
		if (mustexist) {
			lg(LOG_ERR, "readcf: fopen %s: %s",
			    fn, strerror(errno));
			++errs;
		}
		return (errs);
	}

	/* Get the config file mod time */
	if (fstat(fileno(f), &sbuf) >= 0) {
		t = (sbuf.st_mtime > sbuf.st_ctime) ?
		    sbuf.st_mtime : sbuf.st_ctime;
		if (cf->c_time < t)
			cf->c_time = t;
	} else {
		lg(LOG_ERR, "readcf: fstat %s: %s", fn, strerror(errno));
		++errs;
	}

	n = 0;
	while (fgets(buf, sizeof(buf), f) != NULL) {
		++n;

		/* Eat comments */
		cp = strchr(buf, '#');
		if (cp)
			*cp = '\0';

		/* Eat trailing whitesapce */
		cp = buf + strlen(buf) - 1;
		while (cp >= buf && isspace((int)*cp))
			*cp-- = '\0';
		cp = buf;

		/* Skip blank lines */
		if (*cp == '\n' || *cp == '\0')
			continue;

		/* Get command */
		cp2 = cp;
		while (!isspace((int)*cp) && *cp != '\0')
			++cp;
		*cp++ = '\0';

		(void)sprintf(at, "%s:%d", fn, n);

		/* Find start of next keyword */
		while (isspace((int)*cp))
			++cp;

		/* Quoted one line commands */
		if ((pp = &cf->c_cuser, strcasecmp(cp2, "cuser") == 0) ||
#ifdef HAVE_CFORCE
		    (pp = &cf->c_cforceaddr,
			strcasecmp(cp2, "cforceaddr") == 0) ||
		    (pp = &cf->c_cforcedata,
			strcasecmp(cp2, "cforcedata") == 0) ||
#endif
		    (pp = &cf->c_cpass1, strcasecmp(cp2, "cpass1") == 0) ||
		    (pp = &cf->c_cpass2, strcasecmp(cp2, "cpass2") == 0) ||
		    (pp = &cf->c_device, strcasecmp(cp2, "device") == 0) ||
		    (pp = &cf->c_epass1, strcasecmp(cp2, "epass1") == 0) ||
		    (pp = &cf->c_epass2, strcasecmp(cp2, "epass2") == 0) ||
		    (pp = &cf->c_euser, strcasecmp(cp2, "euser") == 0) ||
		    (pp = &cf->c_expect, strcasecmp(cp2, "expect") == 0) ||
		    (pp = &cf->c_expectlog,
			strcasecmp(cp2, "expectlog") == 0) ||
		    (pp = &cf->c_id, strcasecmp(cp2, "id") == 0) ||
#ifdef __FreeBSD__
		    (pp = &cf->c_nullzeroif,
			strcasecmp(cp2, "nullzeroif") == 0) ||
#endif
		    (pp = &cf->c_router, strcasecmp(cp2, "router") == 0) ||
		    (pp = &cf->c_script, strcasecmp(cp2, "script") == 0) ||
#ifdef HAVE_JUNOSCRIPT
		    (pp = &cf->c_sshkeyfn, strcasecmp(cp2, "sshkeyfn") == 0) ||
		    (pp = &cf->c_sshpubfn, strcasecmp(cp2, "sshpubfn") == 0) ||
		    (pp = &cf->c_junoscriptlog,
			strcasecmp(cp2, "junoscriptlog") == 0) ||
#endif
#ifdef HAVE_CORSA
		    (pp = &cf->c_corsacacert,
			strcasecmp(cp2, "corsacacert") == 0) ||
		    (pp = &cf->c_corsalog,
			strcasecmp(cp2, "corsalog") == 0) ||
#endif
		    (pp = &cf->c_macwhitelistfn,
			strcasecmp(cp2, "macwhitelist") == 0) ||
		    (pp = &cf->c_whitelistfn,
			strcasecmp(cp2, "whitelist") == 0)) {
			if (*cp++ != '"') {
				lg(LOG_ERR, "%s missing leading double quote",
				    at);
				++errors;
				continue;
			}
			if (*pp != NULL)  {
				lg(LOG_ERR, "%s only one \"%s\" is allowed",
				    at, cp2);
				++errors;
				continue;
			}
			*pp = strsave(cp);
			cp = *pp;
			cp2 = cp + strlen(cp) - 1;
			if (cp > cp2 || *cp2 != '"') {
				lg(LOG_ERR, "%s missing trailing double quote",
				    at);
				++errors;
				free(*pp);
				*pp = NULL;
				continue;
			}
			*cp2 = '\0';
			continue;
		}

		/* Commands with numeric arguments */
		if ((ip = &cf->c_ayt_secs, strcasecmp(cp2, "ayt_secs") == 0) ||
		    (ip = &cf->c_ayt_secs2,
			strcasecmp(cp2, "ayt_secs2") == 0) ||
#ifdef __FreeBSD__
		    (ip = &cf->c_freebsdroutes,
			strcasecmp(cp2, "freebsdroutes") == 0) ||
#endif
		    (ip = &cf->c_incrseq, strcasecmp(cp2, "incrseq") == 0) ||
		    (ip = &cf->c_ipv4_maxwidth,
			strcasecmp(cp2, "ipv4_maxwidth") == 0) ||
		    (ip = &cf->c_ipv6_maxwidth,
			strcasecmp(cp2, "ipv6_maxwidth") == 0) ||
		    (ip = &cf->c_ipv4_maxnullzerowidth,
			strcasecmp(cp2, "ipv4_maxnullzerowidth") == 0) ||
		    (ip = &cf->c_ipv6_maxnullzerowidth,
			strcasecmp(cp2, "ipv6_maxnullzerowidth") == 0) ||
		    (ip = &cf->c_login_secs,
			strcasecmp(cp2, "login_secs") == 0) ||
		    (ip = &cf->c_login_secs2,
			strcasecmp(cp2, "login_secs2") == 0) ||
		    (ip = &cf->c_nullzeromax,
			strcasecmp(cp2, "nullzeromax") == 0) ||
		    (ip = &cf->c_maxseq, strcasecmp(cp2, "maxseq") == 0) ||
		    (ip = &cf->c_port, strcasecmp(cp2, "port") == 0) ||
#ifdef HAVE_CFORCE
		    (ip = &cf->c_portcforce,
			strcasecmp(cp2, "portcforce") == 0) ||
#endif
#ifdef HAVE_JUNOSCRIPT
		    (ip = &cf->c_junoscript,
			strcasecmp(cp2, "junoscript") == 0) ||
		    (ip = &cf->c_portssh, strcasecmp(cp2, "portssh") == 0) ||
#endif
#ifdef HAVE_CORSA
		    (ip = &cf->c_corsa, strcasecmp(cp2, "corsa") == 0) ||
#endif
		    (ip = &cf->c_mockclient,
			strcasecmp(cp2, "mock_client") == 0) ||
		    (ip = &cf->c_portreplicant,
			strcasecmp(cp2, "portreplicant") == 0) ||
		    (ip = &cf->c_portro, strcasecmp(cp2, "portro") == 0) ||
		    (ip = &cf->c_portweb, strcasecmp(cp2, "portweb") == 0) ||
		    (ip = &cf->c_replicant_session_secs,
			strcasecmp(cp2, "replicant_session_secs") == 0) ||
		    (ip = &cf->c_retries, strcasecmp(cp2, "retries") == 0) ||
		    (ip = &cf->c_select_secs,
			strcasecmp(cp2, "select_secs") == 0) ||
		    (ip = &cf->c_sync_secs,
			strcasecmp(cp2, "sync_secs") == 0)) {
			*ip = atoi(cp);
			if (*cp == '-' || *cp == '+')
				++cp;
			while (isdigit((int)*cp))
				++cp;
			while (isspace((int)*cp))
				++cp;
			if (*cp != '\0') {
				lg(LOG_ERR, "%s trailing garbage (%s)", at, cp);
				++errors;
			}
			continue;
		}

		/* Commands with two numeric arguments */
		if ((ip = &cf->c_lowseq, ip2 = &cf->c_highseq,
			strcasecmp(cp2, "seqrange") == 0) ||
		    (ip = &cf->c_lowportseq, ip2 = &cf->c_highportseq,
			strcasecmp(cp2, "portseqrange") == 0) ||
		    (ip = &cf->c_lowpermithostportseq,
			ip2 = &cf->c_highpermithostportseq,
			strcasecmp(cp2, "permithostportseqrange") == 0)) {
			/* Low value */
			*ip = atoi(cp);
			if (*cp == '-' || *cp == '+')
				++cp;
			if (!isdigit((int)*cp)) {
				lg(LOG_ERR, "%s missing low %s", at, cp2);
				++errors;
			}
			while (isdigit((int)*cp))
				++cp;
			if (!isspace((int)*cp)) {
				lg(LOG_ERR, "%s missing %s whitespace",
				    at, cp2);
				++errors;
			}
			while (isspace((int)*cp))
				++cp;

			/* High value */
			*ip2 = atoi(cp);
			if (*cp == '-' || *cp == '+')
				++cp;
			if (!isdigit((int)*cp)) {
				lg(LOG_ERR, "%s missing high %s", at, cp2);
				++errors;
			}
			while (isdigit((int)*cp))
				++cp;
			while (isspace((int)*cp))
				++cp;
			if (*cp != '\0') {
				lg(LOG_ERR, "%s %s trailing garbage", at, cp2);
				++errors;
			}
			continue;
		}

		/* Commands with ip address arguments */
		if ((a = &cf->c_bindaddr, strcasecmp(cp2, "bindaddr") == 0)
		    || (a = &cf->c_bindaddr6, strcasecmp(cp2, "bindaddr6") == 0)
#ifdef __FreeBSD__
		    || (a = &cf->c_nullzeroaddr,
			strcasecmp(cp2, "nullzeroaddr") == 0)
		    || (a = &cf->c_nullzeroaddr6,
			strcasecmp(cp2, "nullzeroaddr6") == 0)
#endif
		    || (a = &cf->c_replicantaddr,
			strcasecmp(cp2, "replicantaddr") == 0)
		    ) {
			errmsg = extractaddr(cp, NULL, a);
			if (errmsg != NULL) {
				lg(LOG_ERR, "%s bogus ip address: %s",
				    at, errmsg);
				++errors;
			}

			if ((a == &cf->c_bindaddr && a->family != AF_INET)
#ifdef __FreeBSD__
			    || (a == &cf->c_nullzeroaddr &&
				a->family != AF_INET)
#endif
			) {
				lg(LOG_ERR, "%s Not IPv4: %s", at, cp);
				++errors;
			}

			if ((a == &cf->c_bindaddr6 && a->family != AF_INET6)
#ifdef __FreeBSD__
			    || (a == &cf->c_nullzeroaddr6 &&
				a->family != AF_INET6)
#endif
			) {
				lg(LOG_ERR, "%s Not IPv6: %s", at, cp);
				++errors;
			}
			if (a == &cf->c_bindaddr6)
				++cf->c_listen6;
			continue;
		}

		/* ACL limits */
		if (strcasecmp(cp2, "limit") == 0) {
			acllimitadd(cp);
			continue;
		}

		/* ACLs */
		if (strcasecmp(cp2, "acl") == 0) {
			an = makeargv(cp, &av);
			if (an < 2 || an > 3) {
				lg(LOG_ERR, "acl: wrong number of args: \"%s\"",
				    cp);
				if (an > 0)
					freeargv(av);
				++errors;
				continue;
			}
			if (!acladdaclgroup(av[0], av[1])) {
				freeargv(av);
				++errors;
				continue;
			}

			gp = aclfindgroupbyname(av[0]);
			if (an == 3) {
				if (gp == NULL) {
					lg(LOG_ERR, "readcf: cannot find the"
					    " ACL group we just created");
					abort();
				}

				/* Store acl filename */
				gp->fn = strsave(av[2]);
			}
			freeargv(av);
			continue;
		}

		/* Corsa servers */
		if (strcasecmp(cp2, "corsaserver") == 0) {
#ifdef HAVE_CORSA
			an = makeargv(cp, &av);
			if (an != 2) {
				lg(LOG_ERR,
				    "error: %s: requires addr and token path",
				    cp2);
				++errors;
				if (an > 0)
					freeargv(av);
				continue;
			}
			dp = &cf->c_corsaservers_array;
			DYNARRAY(dp, cf->c_corsaservers, dp->len + 1,
			    "corsaserver");
			csp = cf->c_corsaservers + dp->len;
			++dp->len;
			strlcpy(csp->saddr, av[0], sizeof(csp->saddr));
			strlcpy(csp->tokenfn, av[1],
			    sizeof(csp->tokenfn));

			/* corsareadtoken() logs any error */
			cp = corsareadtoken(csp->tokenfn);
			if (cp == NULL)
				++errors;
			else {
				strlcpy(csp->token, cp, sizeof(csp->token));
				if (strcmp(csp->token, cp) != 0) {
					lg(LOG_ERR, "error: %s: token file is"
					    " too big (%s)", cp2, csp->tokenfn);
					++errors;
				}
				free(cp);
			}
#else
			lg(LOG_ERR, "error: %s: corsa support not available",
			    cp2);
			++errors;
#endif
			continue;
		}

		/* Corsa local nets */
		if (strcasecmp(cp2, "corsalocalnet") == 0) {
#ifdef HAVE_CORSA
			if (!corsaaddnet(cp))
				++errors;
#else
			lg(LOG_ERR, "error: %s: corsa support not available",
			    cp2);
			++errors;
#endif
			continue;
		}

		/* Interfaces */
		if (strcasecmp(cp2, "interface") == 0) {
			intlistadd(cp);
			continue;
		}

		/* Null zero nets */
		if (strcasecmp(cp2, "nullzeronet") == 0) {
			if (!nullzeronetadd(cp))
				++errors;
			continue;
		}

		/* Prefix names */
		if (strcasecmp(cp2, "prefix") == 0) {
			if (!prefixadd(cp))
				++errors;
			continue;
		}

		/* nets_log() syslog facility */
		if (strcasecmp(cp2, "netsfac") == 0) {
			cf->c_netsfac = str2val(syslog2str, cp);
			if (cf->c_netsfac < 0) {
				lg(LOG_ERR, "error: bogus netpri value \"%s\"",
				    cp);
				++errors;
			}
			continue;
		}

		/* Parse the replicant command */
		if (strcasecmp(cp2, "replicate") == 0) {
			an = makeargv(cp, &av);
			if (an <= 0) {
				lg(LOG_ERR,
				    "error: replicate statement requires"
				    " at least one argument: \"%s\"", cp);
				++errors;
				continue;
			}
			for (i = 0, pp = av; i < an; ++i, ++pp) {
				if (strcmp(*pp, "acl") == 0) {
					++cf->c_replicate_acl;
					continue;
				}
				if (strcmp(*pp, "nullzero") == 0) {
					++cf->c_replicate_nullzero;
					continue;
				}
				if (strcmp(*pp, "prefix") == 0) {
					++cf->c_replicate_prefix;
					continue;
				}
				if (strcmp(*pp, "primary") == 0) {
					++cf->c_replicate_primary;
					continue;
				}
				if (strcmp(*pp, "whitelist") == 0) {
					++cf->c_replicate_whitelist;
					continue;
				}
				lg(LOG_ERR,
				    "error: unknown replicate keyword \"%s\"",
				    *pp);
				++errors;
			}
			freeargv(av);
			continue;
		}

		/* Bad commands */
		lg(LOG_ERR, "%s unknown command \"%s\"", at, cp2);
		++errors;
	}
	(void)fclose(f);

	return (errs);
}
