/*
 * Copyright (c) 2009, 2010, 2011, 2012, 2013, 2014, 2015, 2016, 2018, 2019, 2020, 2024
 * The Regents of the University of California. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *     * Redistributions of source code must retain the above copyright
 *       notice, this list of conditions and the following disclaimer.
 *     * Redistributions in binary form must reproduce the above copyright
 *       notice, this list of conditions and the following disclaimer in the
 *       documentation and/or other materials provided with the distribution.
 *     * Neither the name of the University nor the names of its contributors
 *       may be used to endorse or promote products derived from this software
 *       without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#ifndef lint
static const char rcsid[] __attribute__ ((unused)) =
    "@(#) $Id$ (LBL)";
#endif

#include <sys/types.h>
#include <sys/mman.h>
#include <sys/time.h>

#include <arpa/inet.h>

#include <ctype.h>
#include <errno.h>
#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sysexits.h>
#include <unistd.h>

#include <cforce.h>

#include "acld.h"
#include "cforcep.h"

/* Locals */
struct s2v cforceerr2str[] = {
	{ "GENERAL",			ERROR_GENERAL },
	{ "COMMUNICATION",		ERROR_COMMUNICATION },
	{ "INVALID_IP_ADDRESS",		ERROR_INVALID_IP_ADDRESS },
	{ "ADDRESS_ALREADY_PROVISIONED", ERROR_ADDRESS_ALREADY_PROVISIONED },
	{ "HOST_ALREADY_BLOCKED",	ERROR_HOST_ALREADY_BLOCKED },
	{ "INVALID_ID",			ERROR_INVALID_ID },
	{ "CAPACITY_EXCEEDED",		ERROR_CAPACITY_EXCEEDED },
	{ "INVALID_RULE",		ERROR_INVALID_RULE },
	{ "BOTH_NOT_SUPPORTED",		ERROR_BOTH_NOT_SUPPORTED },
	{ "CONVERSATION_ALREADY_BLOCKED", ERROR_CONVERSATION_ALREADY_BLOCKED },
	{ "NOT_INITIALIZED",		ERROR_NOT_INITIALIZED },
	{ "INVALID_ARGUMENTS",		ERROR_INVALID_ARGUMENTS },
	{ "INVALID_CONNECTION_PARAMETERS",
					ERROR_INVALID_CONNECTION_PARAMETERS },
	{ "ALL_RULE_PROVISIONED",	ERROR_ALL_RULE_PROVISIONED },
	{ "RULE_CONFLICT",		ERROR_RULE_CONFLICT },
	{ "INVALID_COMBINATION",	ERROR_INVALID_COMBINATION },
	{ "COULD_NOT_OPEN_FILE",	ERROR_COULD_NOT_OPEN_FILE },
	{ "COULD_NOT_WRITE_TO_FD",	ERROR_COULD_NOT_WRITE_TO_FD },
	{ "DATA_STRUCTURE_INCONSISTENCY", ERROR_DATA_STRUCTURE_INCONSISTENCY },
};

struct cforceseq {
	enum acltype type;
	int dir;
	struct addr addr1;
	struct addr addr2;
};

static int md = -1;
struct cforceseq *cforceseq;

/* XXX doesn't include hosthost or IPv6 rules */
#define NUMRULES 64000
size_t cforceseqsize = sizeof(*cforceseq) * NUMRULES;

/* Forwards */
static int acl2dir(struct aclgroup *);
static int cforce_drophost(int, struct addr *);
static int cforce_drophosthost(int, struct addr *, struct addr *);
static void cforcechildlistroute(void);
static void cforceclientcompact(struct client *, struct req *);
static void cforceclientlistroute(struct client *, struct req *, int);
static void cforcedroprestore(struct client *, struct req *);
static int cforceinput(void);
static void cforcekill(void);
static void cforcelistacl(void);
static void cforcelogin(void);
static void cforcemapfile(const char *fn);
static void cforcemsync(void);
static void cforcenullzero(struct client *, struct req *);
static void cforcequerynullzero(struct client *, struct req *);
static void cforcesend(const char *, ...) __attribute__
    ((format (printf, 2, 3)));
static void cforcesendattr(void);
static void cforceserverayt(void);
static void cforceservercompact(void);
static void cforceserversync(void);
static void cforceunmapfile(void);

static int
acl2dir(struct aclgroup *gp)
{
	const char *interface;

	interface = gp->intlist->i_name;
	if (strcasecmp(interface, "outside") == 0)
		return (OUTSIDE);
	if (strcasecmp(interface, "inside") == 0)
		return (INSIDE);
	return (-1);
}

static int
cforce_drophost(int dir, struct addr *addr)
{
	int (*drop_host_func)(block_direction_enum_t, char *);

	if (addr->family == AF_INET)
		drop_host_func = drop_host;
	else
		drop_host_func = drop_ipv6_host;
	return ((drop_host_func)(dir, (char *)addr2str(addr)));
}

static int
cforce_drophosthost(int dir, struct addr *addr1, struct addr *addr2)
{
	char saddr1[128];

	int (*drop_hosthost_func)(rule_port_enum_t, char *, char *);

	if (addr1->family == AF_INET)
		drop_hosthost_func = drop_conversation;
	else
		drop_hosthost_func = drop_ipv6_conversation;
	strlcpy(saddr1, addr2str(addr1), sizeof(saddr1));
	return ((drop_hosthost_func)(dir, saddr1, (char *)addr2str(addr2)));
}

static void
cforcechildlistroute(void)
{
	++state.listedroutes;
}

static void
cforceclientcompact(struct client *cl, struct req *rp)
{
	rp->flags |= RFLAG_IGNORE;
	clientsendfmt(cl, rp, "cForce does not require ACL compaction");
}

static void
cforceclientlistroute(struct client *cl, struct req *rp, int nullzeroonly)
{
	clientsenderr(cl, rp, "cForce doesn't support null zero routes");
}

static void
cforcedroprestore(struct client *cl, struct req *rp)
{
	int rc, dir, seq;
	const char *p, *what;
	struct aclgroup *gp;
	struct acl *al;
	struct addr *wp, *tp;
	struct cforceseq *cs;
	char buf[132];

	/* Find ACL list */
	switch (rp->acl.type) {

	case ATYPE_BLOCKHOST:
	case ATYPE_BLOCKHOSTHOST:
		/* By address */
		gp = aclfindgroupbyaddr(&rp->acl.addr1);
		if (gp == NULL) {
			clientsenderr(cl, rp, "can't find aclgroup for %s"
			    " (is the default ACL missing?)",
			    addr2str(&rp->acl.addr1));
			return;
		}
		rp->aclgroup = gp;
		break;

	default:
		clientsenderr(cl, rp, "ATYPE %s not implemented",
		    val2str(str2acl, rp->acl.type));
		return;
	}

	/* Check if ACL should or should not already exist */
	al = aclfindacl(gp, &rp->acl, 0);
	switch (rp->type) {

	case REQ_DROP:
		tp = &rp->acl.addr1;
		wp = whitelist(tp);
		if (wp != NULL) {
			snprintf(buf, sizeof(buf), "%s", addr2str(tp));
			clientsenderr(cl, rp, "%s is on the whitelist (%s)",
			    buf, addr2str(wp));

			/* Log for NETS */
			nets_log(cl, rp);
			return;
		}

		if (al != NULL) {
			rp->flags |= RFLAG_IGNORE;
			clientsendfmt(cl, rp,
			    "Note: %s is already blocked in ACL %s",
			    addr2str(tp), gp->name);

			/* Log for NETS */
			nets_log(cl, rp);
			return;
		}
		stats_setrate(&gp->stats);
		break;

	case REQ_BLOCKHOSTHOST:
		tp = &rp->acl.addr1;
		wp = whitelist(tp);
		if (wp == NULL) {
			/* Check second address too */
			tp = &rp->acl.addr2;
			wp = whitelist(tp);
		}
		if (wp != NULL) {
			snprintf(buf, sizeof(buf), "%s", addr2str(tp));
			clientsenderr(cl, rp, "%s is on the whitelist (%s)",
			    buf, addr2str(wp));

			/* Log for NETS */
			nets_log(cl, rp);
			return;
		}

		if (al != NULL) {
			rp->flags |= RFLAG_IGNORE;
			snprintf(buf, sizeof(buf), "%s", addr2str(tp));
			clientsendfmt(cl, rp,
			    "Note: %s/%s is already blocked in ACL %s",
			    buf, addr2str(&rp->acl.addr2), gp->name);

			/* Log for NETS */
			nets_log(cl, rp);
			return;
		}
		stats_setrate(&gp->stats);
		break;

	case REQ_RESTORE:
	case REQ_RESTOREHOSTHOST:
		if (al == NULL) {
			rp->flags |= RFLAG_IGNORE;
			clientsenderr(cl, rp, "%s not found in ACL %s",
			    addr2str(&rp->acl.addr1), gp->name);

			/* Log for NETS */
			nets_log(cl, rp);
			return;
		}
		break;

	default:
		lg(LOG_ERR, "cforcedroprestore: bad type %d (1)", rp->type);
		exit(EX_SOFTWARE);
	}

	/* Add or remove the ACL */
	switch (rp->type) {

	case REQ_DROP:
	case REQ_BLOCKHOSTHOST:
		/* Check cap and limits */
		p = checklimits(gp);
		if (p != NULL) {
			clientsenderr(cl, rp, "%s", p);
			syslog(LOG_INFO, "%s", p);
			return;
		}

		/* Add the rule */
		if (rp->type == REQ_DROP) {
			/* Determine cForce direction */
			dir = acl2dir(gp);
			if (dir < 0) {
				(void)snprintf(buf, sizeof(buf),
				    "Direction for ACL %s undefined", gp->name);
				clientsenderr(cl, rp, "%s", buf);
				syslog(LOG_INFO, "%s", buf);
				return;
			}
			rc = cforce_drophost(dir, &rp->acl.addr1);
			what = "cforce_drophost";
		} else {
			dir = CFORCE_BOTH_PORTS;
			rc = cforce_drophosthost(dir, &rp->acl.addr1,
			    &rp->acl.addr2);
			what = "cforce_drophosthost";
		}
		if (rc < 0) {
			(void)snprintf(buf, sizeof(buf), "%s failed: %s",
			    what, val2str(cforceerr2str, rc));
			lg(LOG_ERR, "cforcedroprestore: %s", buf);
			clientsenderr(cl, rp, "%s", buf);
			/*
			 * Force reinitialization of the appliance for
			 * most errors
			 */
			if (rc != ERROR_CAPACITY_EXCEEDED)
				cforcekill();
			return;
		}

		/* Assign sequence number */
		rp->acl.seq = rc;

		/* Sanity */
		cs = cforceseq + rp->acl.seq;
		if (cs->type != ATYPE_UNKNOWN) {
			lg(LOG_ERR, "cforcedroprestore: %s failed:"
			    " cforceseq slot %d already in use",
			    what, rp->acl.seq);
			exit(EX_SOFTWARE);
		}

		/* Update our data structure */
		acladdacl(gp, &rp->acl);

		/* Update mmap file */
		cs->dir = dir;
		if (rp->type == REQ_DROP) {
			cs->type = ATYPE_BLOCKHOST;
			cs->addr1 = rp->acl.addr1;
		} else {
			cs->type = ATYPE_BLOCKHOSTHOST;
			cs->addr1 = rp->acl.addr1;
			cs->addr2 = rp->acl.addr2;
		}

		/* Always sort after assigning new sequence numbers */
		aclsortacls();
		break;

	case REQ_RESTORE:
	case REQ_RESTOREHOSTHOST:
		/* Save a copy of the sequence number before it gets zeroed */
		seq = al->seq;

		/* Delete the rule */
		rc = delete_rule(seq);
		if (rc < 0) {
			snprintf(buf, sizeof(buf), "delete_rule(%d) failed: %s",
			    seq, val2str(cforceerr2str, rc));
			lg(LOG_ERR, "cforcedroprestore: %s", buf);
			clientsenderr(cl, rp, "%s", buf);

			/* Force reinitialization of the appliance */
			cforcekill();
		}

		/* Remove from our data structure */
		if (!acldeleteacl(gp, &rp->acl)) {
			lg(LOG_ERR, "cforcedroprestore: acldeleteacl failed");
			exit(EX_SOFTWARE);
		}

		/* Update mmap file */
		cs = cforceseq + seq;
		memset(cs, 0, sizeof(*cs));
		break;

	default:
		lg(LOG_ERR, "cforcedroprestore: bad type %d (2)", rp->type);
		exit(EX_SOFTWARE);
	}

	/* Update mmap file */
	cforcemsync();

	/* Log for NETS */
	nets_log(cl, rp);

	clientsend(cl, rp);
}

void
cforceinit(void)
{
	struct state *sp;

	/* Child hooks */
	sp = &state;
	sp->f_childinput = cforceinput;
	sp->f_childkill = cforcekill;
	sp->f_childlistacl = cforcelistacl;
	sp->f_childlistroute = cforcechildlistroute;
	sp->f_childlogin = cforcelogin;
	sp->f_childsend = cforcesend;
	sp->f_childsendattr = cforcesendattr;
	sp->f_clientcompact = cforceclientcompact;

	/* Client hooks */
	sp->f_clientdroprestore = cforcedroprestore;
	sp->f_clientlistroute = cforceclientlistroute;
	sp->f_clientnullzero = cforcenullzero;
	sp->f_clientquerynullzero = cforcequerynullzero;

	/* Server hooks */
	sp->f_serverayt = cforceserverayt;
	sp->f_servercompact = cforceservercompact;
	sp->f_serversync = cforceserversync;
}

static int
cforceinput(void)
{
	lg(LOG_ERR, "cforceinput: unused");
	exit(EX_SOFTWARE);
}

static void
cforcekill(void)
{
	struct state *sp;

	sp = &state;
	setastate(ASTATE_NOTCONNECTED);
	timerreset(&sp->t_ayt);
	timerreset(&sp->t_login);
	timerreset(&sp->t_sync);

	/* Free ACL lists so we'll re-aquire them when we re-block them */
	aclgroupsfree();
}

static void
cforcelistacl(void)
{
	aclsortacls();
	++state.listedallacls;
}

static void
cforcelogin(void)
{
	int rc, i;
	const char *what;
	struct aclgroup *gp;
	struct cforceseq *cforceseq2, *cs, *cs2;
	struct acl acl;
	static int init = 0;

	if (!init) {
		rc = cforce_initialize(cf->c_cforceaddr, cf->c_portcforce);
		if (rc < 0) {
			lg(LOG_ERR, "cforcelogin: cforce_initialize failed: %s",
			    val2str(cforceerr2str, rc));
			return;
		}
		++init;
	} else {
		lg(LOG_INFO, "cforcelogin: unmapping cforce data file");
		cforceunmapfile();
	}

	/* Zero rules in appliance */
	rc = clear_rules();
	if (rc < 0) {
		lg(LOG_ERR, "cforcelogin: clear_rules failed: %s",
		    val2str(cforceerr2str, rc));
		return;
	}

	/* Reinstall saved rules */
	cforcemapfile(cf->c_cforcedata);

	/* Reblock hosts */
	cs = cforceseq;
	cforceseq2 = malloc(cforceseqsize);
	if (cforceseq2 == NULL) {
		lg(LOG_ERR, "cforcelogin: malloc");
		exit(EX_OSERR);
	}
	memmove(cforceseq2, cforceseq, cforceseqsize);
	memset(cforceseq, 0, cforceseqsize);

	for (i = 0, cs2 = cforceseq2; i < NUMRULES; ++i, ++cs2) {
		switch (cs2->type) {

		case ATYPE_UNKNOWN:
			/* Unused slot */
			break;

		case ATYPE_BLOCKHOST:
			/* Install rule in appliance */
			what = "cforce_drophost";
			rc = cforce_drophost(cs2->dir, &cs2->addr1);
			if (rc < 0) {
				lg(LOG_ERR, "cforcelogin: drop_host failed: %s",
				    val2str(cforceerr2str, rc));
				exit(EX_SOFTWARE);
			}

			/* Sanity */
			if (rc >= NUMRULES) {
				lg(LOG_ERR, "cforcelogin: drop_host"
				    " seq number too boku %d", rc);
				exit(EX_SOFTWARE);
			}
			cs = cforceseq + rc;
			if (cs->type != ATYPE_UNKNOWN) {
				lg(LOG_ERR, "cforcelogin: %s failed:"
				    " cforceseq seq %d already in use",
				    what, rc);
				exit(EX_SOFTWARE);
			}

			/* Update our data structure */
			gp = aclfindgroupbyaddr(&cs2->addr1);
			if (gp == NULL) {
				lg(LOG_ERR, "can't find aclgroup for %s",
				    addr2str(&cs2->addr1));
				exit(EX_SOFTWARE);
			}
			memset(&acl, 0, sizeof(acl));
			acl.seq = rc;
			acl.type = cs2->type;
			memmove(&acl.addr1, &cs2->addr1, sizeof(acl.addr1));
			acladdacl(gp, &acl);

			/* Copy rule */
			*cs = *cs2;
			++cs;
			break;

		case ATYPE_BLOCKHOSTHOST:
			/* Install rule in appliance */
			what = "cforce_drophosthost";
			rc = cforce_drophosthost(cs2->dir, &cs2->addr1,
			    &cs2->addr2);
			if (rc < 0) {
				lg(LOG_ERR, "cforcelogin: cforce_drophosthost"
				    " failed: %s", val2str(cforceerr2str, rc));
				exit(EX_SOFTWARE);
			}

			/* Sanity */
			if (rc >= NUMRULES) {
				lg(LOG_ERR, "cforcelogin: drop_hosthosthost"
				    " seq number too boku %d", rc);
				exit(EX_SOFTWARE);
			}
			cs = cforceseq + rc;
			if (cs->type != ATYPE_UNKNOWN) {
				lg(LOG_ERR, "cforcelogin: %s failed:"
				    " cforceseq seq %d already in use",
				    what, rc);
				exit(EX_SOFTWARE);
			}

			/* Update our data structure */
			gp = aclfindgroupbyaddr(&cs2->addr1);
			if (gp == NULL) {
				lg(LOG_ERR, "can't find aclgroup for %s",
				    addr2str(&cs2->addr1));
				exit(EX_SOFTWARE);
			}
			memset(&acl, 0, sizeof(acl));
			acl.seq = rc;
			acl.type = cs2->type;
			memmove(&acl.addr1, &cs2->addr1, sizeof(acl.addr1));
			memmove(&acl.addr2, &cs2->addr2, sizeof(acl.addr2));
			acladdacl(gp, &acl);

			/* Copy rule */
			*cs = *cs2;
			++cs;
			break;

		default:
			lg(LOG_ERR, "unhandled type #%d", cs2->type);
			break;
		}
	}
	free(cforceseq2);

	/* Need to sort since the cForce appliance picks the sequence numbers */
	aclsortacls();

	setastate(ASTATE_LOGGEDIN);
	timerreset(&state.t_login);
}

static void
cforcemapfile(const char *fn)
{
	char ch;
	struct stat sbuf;

	if (fn == NULL) {
		lg(LOG_ERR, "cforcemapfile: missing cforce filename");
		exit(EX_SOFTWARE);
	}

	md = open(fn, O_RDWR | O_CREAT, 0600);
	if (md < 0) {
		lg(LOG_ERR, "cforcemapfile(%s): open: %s",
		    fn, strerror(errno));
		exit(EX_SOFTWARE);
	}

	memset(&sbuf, 0, sizeof(sbuf));
	if (fstat(md, &sbuf) < 0) {
		lg(LOG_ERR, "cforcemapfile(%s): fstat: %s",
		    fn, strerror(errno));
		exit(EX_SOFTWARE);
	}

	if (sbuf.st_size == 0) {
		lg(LOG_ERR,
		    "cforcemapfile(%s): initialize empty mmap file", fn);
		if (lseek(md, cforceseqsize - 1, SEEK_SET) < 0) {
			lg(LOG_ERR, "cforcemapfile(%s): lseek: %s",
			    fn, strerror(errno));
			exit(EX_SOFTWARE);
		}
		ch = 0;
		if (write(md, &ch, 1) < 0) {
			lg(LOG_ERR, "cforcemapfile(%s): write: %s",
			    fn, strerror(errno));
			exit(EX_OSERR);
		}
		if (fsync(md) < 0) {
			lg(LOG_ERR, "cforcemapfile(%s): fsync: %s",
			    fn, strerror(errno));
			exit(EX_OSERR);
		}
	} else if (sbuf.st_size != cforceseqsize) {
		lg(LOG_ERR, "cforcemapfile(%s): bad size: %ld != %ld",
		    fn, (u_long)sbuf.st_size, (u_long)cforceseqsize);
		exit(EX_SOFTWARE);
	}

	cforceseq = mmap(0, cforceseqsize, PROT_READ | PROT_WRITE, MAP_SHARED,
	    md, 0);
	if ((int)cforceseq == -1) {
		lg(LOG_ERR, "cforcemapfile(%s): mmap: %s",
		    fn, strerror(errno));
		exit(EX_OSERR);
	}
	if (close(md) < 0) {
		lg(LOG_ERR, "cforcemapfile(%s): close: %s",
		    fn, strerror(errno));
		exit(EX_OSERR);
	}
	cforcemsync();
}

static void
cforcemsync(void)
{
	if (msync(cforceseq, cforceseqsize, MS_SYNC) < 0) {
		lg(LOG_ERR, "cforcemsync(): msync: %s", strerror(errno));
		exit(EX_OSERR);
	}
}

static void
cforcenullzero(struct client *cl, struct req *rp)
{

	clientsenderr(cl, rp, "cForce doesn't support null zero routes");
}

static void
cforcequerynullzero(struct client *cl, struct req *rp)
{
	clientsenderr(cl, rp, "cForce doesn't support null zero routes");
}

static void
cforcesend(const char *fmt, ...)
{
	abort();
}

static void
cforcesendattr(void)
{
	++state.sentattrs;
}

/* Assume if we can get the appliance counters that it's alive */
static void
cforceserverayt(void)
{
	int rc;
	cforce_counters_t counters;

	rc = get_counters(&counters);
	if (rc < 0) {
		lg(LOG_DEBUG, "cforceserverayt: get_counters failed: %s",
		    val2str(cforceerr2str, rc));
		cforcekill();
	}
}

static void
cforceservercompact(void)
{
	lg(LOG_DEBUG, "cforceservercompact: noop");
}

static void
cforceserversync(void)
{
	lg(LOG_DEBUG, "cforceserversync: noop");
}

static void
cforceunmapfile(void)
{
	if (cforceseq == NULL)
		return;
	if (munmap(cforceseq, cforceseqsize) < 0) {
		lg(LOG_ERR, "cforceunmapfile(): munmap: %s", strerror(errno));
		exit(EX_SOFTWARE);
	}
	cforceseq = NULL;
}
