/*
 * Copyright (c) 2002, 2003, 2004, 2005, 2006, 2007, 2008, 2009, 2010, 2011, 2012, 2013, 2014, 2016, 2018, 2019, 2020, 2021, 2022, 2024
 * The Regents of the University of California. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *     * Redistributions of source code must retain the above copyright
 *       notice, this list of conditions and the following disclaimer.
 *     * Redistributions in binary form must reproduce the above copyright
 *       notice, this list of conditions and the following disclaimer in the
 *       documentation and/or other materials provided with the distribution.
 *     * Neither the name of the University nor the names of its contributors
 *       may be used to endorse or promote products derived from this software
 *       without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#ifndef lint
static const char rcsid[] __attribute__ ((unused)) =
    "@(#) $Id$ (LBL)";
#endif

#include <sys/types.h>
#include <sys/time.h>

#include <errno.h>
#include <signal.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sysexits.h>
#include <unistd.h>

#include "acld.h"
#include "cf.h"
#include "child.h"
#include "prefix.h"

/* Forwards */
static int childresponse(char **);

/* Fix up things if updating the router failed */
void
childaclcleanup(struct req *rp)
{
	struct aclgroup *gp;
	const char *errmsg;

	switch (rp->type) {

	case REQ_DROP:
	case REQ_BLOCKHOSTHOST:
		/* Need to delete ACL entry since we failed */
		gp = aclfindgroupbyaddr(&rp->acl.addr1);
		if (gp == NULL) {
			lg(LOG_ERR, "childaclcleanup: NULL %s", rp->cmd);
			exit(EX_SOFTWARE);
		}
		if (!acldeleteacl(gp, &rp->acl))
			lg(LOG_ERR, "childaclcleanup: acldeleteacl failed");
		break;

	case REQ_BLOCKMAC:
		/* Need to delete ACL entry since we failed */
		gp = aclgetmacgroup();
		if (!acldeleteacl(gp, &rp->acl)) {
			lg(LOG_ERR, "clientdroprestore: acldeleteacl failed");
			exit(EX_SOFTWARE);
		}
		break;

	case REQ_RESTOREMAC:
		/* Need to restore ACL entry since we failed */
		gp = aclgetmacgroup();
		acladdacl(gp, &rp->acl);
		break;

	case REQ_PERMITUDPDSTHOSTPORT:
	case REQ_PERMITTCPDSTHOSTPORT:
	case REQ_DROPTCPDSTHOSTPORT:
		/* Need to delete ACL entry since we failed */
		gp = aclfindgroupdefault(&rp->acl.addr1);
		if (gp == NULL) {
			lg(LOG_ERR, "childaclcleanup: NULL %s", rp->cmd);
			exit(EX_SOFTWARE);
		}
		if (!acldeleteacl(gp, &rp->acl))
			lg(LOG_ERR, "childaclcleanup: acldeleteacl failed");
		break;

	case REQ_RESTORE:
	case REQ_RESTOREHOSTHOST:
		/* Need to restore ACL entry since we failed */
		gp = aclfindgroupbyaddr(&rp->acl.addr1);
		if (gp == NULL) {
			lg(LOG_ERR, "childaclcleanup: NULL %s", rp->cmd);
			exit(EX_SOFTWARE);
		}
		acladdacl(gp, &rp->acl);
		break;

	case REQ_UNPERMITUDPDSTHOSTPORT:
	case REQ_UNPERMITTCPDSTHOSTPORT:
	case REQ_RESTORETCPDSTHOSTPORT:
		/* Need to restore ACL entry since we failed */
		gp = aclfindgroupdefault(&rp->acl.addr1);
		if (gp == NULL) {
			lg(LOG_ERR, "childaclcleanup: NULL %s", rp->cmd);
			exit(EX_SOFTWARE);
		}
		acladdacl(gp, &rp->acl);
		break;

	case REQ_DROPUDPPORT:
	case REQ_DROPTCPPORT:
	case REQ_DROPTCPSYNPORT:
		/* Need to delete/restore ACL entry since we failed */
		gp = aclfindgroupbyname(rp->aclname);
		if (gp == NULL) {
			lg(LOG_ERR, "childaclcleanup: NULL %s", rp->cmd);
			exit(EX_SOFTWARE);
		}
		if (!acldeleteacl(gp, &rp->acl))
			lg(LOG_ERR, "childaclcleanup: acldeleteacl failed");
		break;

	case REQ_RESTOREUDPPORT:
	case REQ_RESTORETCPPORT:
	case REQ_RESTORETCPSYNPORT:
		/* Need to delete/restore ACL entry since we failed */
		gp = aclfindgroupbyname(rp->aclname);
		if (gp == NULL) {
			lg(LOG_ERR, "childaclcleanup: NULL %s", rp->cmd);
			exit(EX_SOFTWARE);
		}
		acladdacl(gp, &rp->acl);
		break;

	case REQ_NULLZERO:
		/* Need to delete nullzero route since we failed */
		if (!routedelete(&rp->nullzero))
			lg(LOG_ERR, "childaclcleanup: routedelete failed");
		break;

	case REQ_NONULLZERO:
		/* Need to restore nullzero route since we failed */
		routeadd(&rp->nullzero);
		break;

	case REQ_FILTER:
		/* Nothing to do */
		break;

	case REQ_NOFILTER:
		/* Nothing to do */
		break;

	case REQ_PREFIX:
		/* Need to delete prefix since we failed */
		errmsg = prefixdeleteaddr(rp->prefixname, &rp->prefix);
		if (errmsg != NULL)
			lg(LOG_ERR, "childaclcleanup: prefix: %s", errmsg);
		break;

	case REQ_NOPREFIX:
		/* Need to restore prefix since we failed */
		errmsg = prefixaddaddr(rp->prefixname, &rp->prefix);
		if (errmsg != NULL)
			lg(LOG_ERR, "childaclcleanup: noprefix: %s", errmsg);
		break;

	case REQ_COMPACT:
		/* Need to reacquire all sequence numbers for all ACL groups */
		aclgroupsfree();
		break;

	default:
		lg(LOG_DEBUG, "childaclcleanup: request type %s (%d) unhandled",
		    val2str(cmd2req, rp->type), rp->type);
		break;
	}
}

int
childbusy(void)
{
	struct client *cl;
	struct req *rp;

	for (cl = state.clients; cl != NULL; cl = cl->next)
		for (rp = cl->req; rp != NULL; rp = rp->next)
			if (rp->state == RSTATE_CHILD)
				return (1);
	return (0);
}

/* We support everything clientdroprestore does */
const char *
childcheckreq(struct req *rp)
{
	return (NULL);
}

/* Process input from the child, return 1 if done */
int
childinput(void)
{
	int an, flags, i, haveline;
	char *cp;
	struct aclgroup *gp, *gp2;
	struct array *dp;
	struct client *cl;
	struct req *rp, *rp2;
	struct iobuf *ip;
	char **av;
	static char *p1, *p2;
	static char prompt[] = "expect>";
	struct state *sp;

	sp = &state;
	ip = &sp->rbuf;
	switch (sp->state) {

	case ASTATE_READRESPONSE:
		if (childresponse(&p1)) {
			setastate(ASTATE_LOGGEDIN);
			sp->cmd = NULL;
			/* We're logged in now */
			timerreset(&sp->t_login);
			if (p1 != NULL) {
				trimws(p1, strlen(p1));
				lg(LOG_DEBUG, "childinput response: \"%s\"",
				    pretty(p1));
				free(p1);
				p1 = NULL;
			}
		}
		return (0);

	case ASTATE_READERROR:
		if (childresponse(&p2)) {
			setastate(ASTATE_CONNECTED);
			sp->cmd = NULL;
			/* Wait a bit before trying again */
			timerset(&sp->t_login, cf->c_login_secs);
			if (p2 != NULL) {
				trimws(p2, strlen(p2));
				lg(LOG_DEBUG, "childinput error: \"%s\"",
				    pretty(p2));
				free(p2);
				p2 = NULL;
			}
		}
		return (0);

	case ASTATE_READACL:
		while ((haveline = iohaveline(ip)) &&
		    (cp = iogetstr(ip)) != NULL) {
			/* Find the last ACL list that hasn't been updated */
			gp2 = cf->c_aclgroup;
			gp = NULL;
			dp = &cf->c_aclgroup_array;
			for (i = 0; i < dp->len; ++i) {
				if (!gp2->sentlistacl)
					break;
				gp = gp2++;
			}
			if (gp == NULL) {
				lg(LOG_ERR, "childinput: can't happen");
				exit(EX_SOFTWARE);
			}

			/* Remove trailing newline for possible error message */
			trimws(cp, strlen(cp));
			if (aclstraddacl(gp, cp)) {
				setastate(ASTATE_LOGGEDIN);
				timerreset(&sp->t_login);
				sp->cmd = NULL;
				break;
			}
		}
		/* We're not done if there wasn't a newline in the buffer */
		return (!haveline);

	case ASTATE_READROUTE:
		while ((haveline = iohaveline(ip)) &&
		    (cp = iogetstr(ip)) != NULL) {
			/* Remove trailing newline for possible error message */
			trimws(cp, strlen(cp));
			if (routestradd(cp)) {
				setastate(ASTATE_LOGGEDIN);
				timerreset(&sp->t_login);
				sp->cmd = NULL;
				break;
			}
		}
		/* We're not done if there wasn't a newline in the buffer */
		return (!haveline);

	default:
		/* Find server or client request */
		rp = sp->req;
		cl = NULL;
		if (rp == NULL) {
			for (cl = sp->clients; cl != NULL; cl = cl->next) {
				rp2 = cl->req;
				for (; rp2 != NULL; rp2 = rp2->next) {
					if (rp2->state == RSTATE_CHILD ||
					    rp2->state == RSTATE_READRESPONSE) {
						rp = rp2;
						break;
					}
				}
				if (rp != NULL)
					break;
			}
		}

		if (rp != NULL && rp->state == RSTATE_READRESPONSE) {
			while ((cp = iogetstr(ip)) != NULL) {
				/* Strip trailing "\r\n" */
				trimws(cp, strlen(cp));
				if (strcmp(cp, ".") != 0) {
					lg(LOG_DEBUG,
					    "childinput response: \"%s\"",
					    pretty(cp));
					if (cl != NULL)
						ioappendline(&rp->payload, cp);
					continue;
				}
				if (cl != NULL)
					clientsend(cl, rp);
				else {
					rp->state = RSTATE_DONE;
					getts(&rp->cts);
				}
				timerset(&sp->t_ayt, cf->c_ayt_secs);

				/* Log for NETS */
				if (cl != NULL)
					nets_log(cl, rp);

				/* Fix up things if the router command failed */
				if ((rp->flags & RFLAG_FAILED) != 0)
					childaclcleanup(rp);
				break;
			}
			return (0);
		}
		break;
	}

	/* Get input */
	cp = iogetstr(ip);
	if (strncmp(cp, prompt, sizeof(prompt) - 1) == 0) {
		if (sp->state == ASTATE_NOTCONNECTED) {
			setastate(ASTATE_CONNECTED);
			sp->cmd = NULL;
		}
		if (debug)
			fprintf(log_lf.f, "%s childinput: (ready)\n", tsstr());
		cp += sizeof(prompt) - 1;
		if (*cp == '\0')
			return (0);
	}

	/* Break into arguments */
	av = NULL;
	an = makeargv(cp, &av);
	if (an <= 0) {
		freeargv(av);
		return (0);
	}

	if (debug)
		fprintf(log_lf.f, "%s childinput: %s\n",
		    tsstr(), fmtargv(an, av));

	/* Child appears to be alive */
	timerset(&sp->t_ayt, cf->c_ayt_secs);

	switch (sp->state) {

	case ASTATE_SENTLISTACL:
	case ASTATE_SENTLISTROUTE:
	case ASTATE_SENTLOGIN:
	case ASTATE_SENTATTR:
		flags = 0;
		if (failed(sp->cmd, av[0]))
			flags |= RFLAG_FAILED;
		else if (strcmp(sp->cmd, av[0]) != 0) {
			lg(LOG_ERR,
			    "childinput: unexpected response %s waiting for %s",
			    av[0], sp->cmd);
			freeargv(av);
			return (0);
		}
		if (an > 1 && strcmp(av[1], "-") == 0)
			flags |= RFLAG_CONTINUE;

		switch (sp->state) {

		case ASTATE_SENTLOGIN:
		case ASTATE_SENTATTR:
			if ((flags & RFLAG_FAILED) == 0) {
				if ((flags & RFLAG_CONTINUE) == 0) {
					setastate(ASTATE_LOGGEDIN);
					timerreset(&sp->t_login);
				} else
					setastate(ASTATE_READRESPONSE);
			} else {
				if ((flags & RFLAG_CONTINUE) == 0) {
					setastate(ASTATE_CONNECTED);
					/* Wait a bit before trying again */
					timerset(&sp->t_login,
					    cf->c_login_secs);
				} else
					setastate(ASTATE_READERROR);
			}
			break;

		case ASTATE_SENTLISTACL:
			if ((flags & RFLAG_FAILED) == 0) {
				if ((flags & RFLAG_CONTINUE) == 0) {
					setastate(ASTATE_LOGGEDIN);
					timerreset(&sp->t_login);
				} else
					setastate(ASTATE_READACL);
			} else {
				if ((flags & RFLAG_CONTINUE) == 0) {
					setastate(ASTATE_LOGGEDIN);
					childkill();
				} else
					setastate(ASTATE_READERROR);
			}
			break;

		case ASTATE_SENTLISTROUTE:
			if ((flags & RFLAG_FAILED) == 0) {
				if ((flags & RFLAG_CONTINUE) == 0) {
					setastate(ASTATE_LOGGEDIN);
					timerreset(&sp->t_login);
				} else
					setastate(ASTATE_READROUTE);
			} else {
				if ((flags & RFLAG_CONTINUE) == 0) {
					setastate(ASTATE_LOGGEDIN);
					timerreset(&sp->t_login);
					childkill();
				} else
					setastate(ASTATE_READERROR);
			}
			break;

		default:
			break;
		}

		freeargv(av);
		return (0);

	default:
		break;
	}

	if (rp == NULL) {
		lg(LOG_ERR, "childinput: no server/client request to process");
		freeargv(av);
		return (0);
	}

	if (rp->state != RSTATE_CHILD) {
		lg(LOG_ERR, "childinput: not waiting for child to talk!");
		freeargv(av);
		return (0);
	}

	rp->flags = 0;
	if (failed(rp->cmd, av[0]))
		rp->flags = RFLAG_FAILED;
	else if (strcmp(rp->cmd, av[0]) != 0) {
		lg(LOG_ERR, "childinput: unexpected response %s waiting for %s",
		    av[0], rp->cmd);
		freeargv(av);
		return (0);
	}

	if (an > 1 && strcmp(av[1], "-") == 0) {
		rp->flags |= RFLAG_CONTINUE;
		rp->state = RSTATE_READRESPONSE;
		freeargv(av);
		/* Need to collect response before we're done */
		return (0);
	} else {
		rp->state = RSTATE_DONE;
		getts(&rp->cts);

		/* Log for NETS */
		if (cl != NULL)
			nets_log(cl, rp);
	}

	/* Done with cracked arguments */
	freeargv(av);

	/* Wrap up this request */
	if (cl != NULL)
		clientsend(cl, rp);

#ifdef notdef
	/* Fix up things if the router command failed */
	if ((rp->flags & RFLAG_FAILED) != 0) {
		abort();
		// childaclcleanup(rp);
	}
#endif

	return (0);
}

void
childkill(void)
{
	struct state *sp;

	sp = &state;
	if (sp->wfd >= 0 && sp->state == ASTATE_LOGGEDIN)
		childsend("logout");
	setastate(ASTATE_NOTCONNECTED);
	sp->cmd = NULL;

	errno = 0;
	if (sp->pid >= 0) {
		if (kill(sp->pid, SIGTERM) < 0 && errno != ESRCH)
			lg(LOG_ERR, "childkill: kill %d: %s",
			    (int)sp->pid, strerror(errno));

		/* Try harder */
		errno = 0;
		if (kill(sp->pid, 0) >= 0 && errno == 0) {
			lg(LOG_ERR, "childkill: sending SIGKILL to %d",
			    (int)sp->pid);
			if (kill(sp->pid, SIGKILL) < 0)
				lg(LOG_ERR, "childkill: kill %d: %s",
				    (int)sp->pid, strerror(errno));
		}
	}

	sp->pid = -1;
	if (sp->rfd >= 0) {
		(void)close(sp->rfd);
		/* In case they are the same fd (e.g. libssh2) */
		if (sp->rfd == sp->wfd)
			sp->wfd = -1;
		sp->rfd = -1;
	}

	if (sp->wfd >= 0) {
		(void)close(sp->wfd);
		sp->wfd = -1;
	}

	iofree(&sp->rbuf);

	timerreset(&sp->t_ayt);
	timerreset(&sp->t_login);
	timerreset(&sp->t_sync);
}

void
childlistacl(void)
{
	int i;
	struct aclgroup *gp;
	struct array *dp;
	struct state *sp;

	sp = &state;
	dp = &cf->c_aclgroup_array;
	for (i = 0, gp = cf->c_aclgroup; i < dp->len; ++i, ++gp)
		if (!gp->sentlistacl) {
			setastate(ASTATE_SENTLISTACL);
			sp->cmd = "listacl";
			childsend("%s %s {%s}",
			    sp->cmd, gp->name, gp->intlist->i_name);
			++gp->sentlistacl;
			return;
		}

	/* If we got here, we've listed all the ACLs; sort 'em */
	++sp->listedallacls;
	aclsortacls();

	/* Write out configured ACL files */
	for (i = 0, gp = cf->c_aclgroup; i < dp->len; ++i, ++gp)
		if (gp->fn != NULL)
			aclupdatefile("childlistacl", gp);
}

void
childlistroute(void)
{
	struct state *sp;

	sp = &state;
	routelistsfree();
	setastate(ASTATE_SENTLISTROUTE);
	sp->cmd = "listroute";
	childsend("%s", sp->cmd);
	++sp->listedroutes;
}

void
childlogin(void)
{
	int odebug;
	struct state *sp;

	/* Free ACL lists so we'll re-aquire them from the router */
	aclgroupsfree();

	/* Free routes so we'll re-aquire them from the router */
	routelistsfree();

	setastate(ASTATE_SENTLOGIN);
	sp = &state;
	sp->cmd = "login";
	odebug = debug;
	if (odebug) {
		/* Avoid exposing router passwords in debugging output */
		debug = 0;
		fprintf(log_lf.f, "%s childsend: \"%s %s %s %s %s %s %s %s\"\n",
		    tsstr(),
		    sp->cmd,
		    cf->c_router,
		    cf->c_cuser,
		    "?",
		    "?",
		    cf->c_euser,
		    "?",
		    "?");
	}
	childsend("%s %s %s %s %s %s %s %s",
	    sp->cmd,
	    cf->c_router,
	    cf->c_cuser,
	    cf->c_cpass1,
	    cf->c_cpass2,
	    cf->c_euser,
	    cf->c_epass1,
	    cf->c_epass2);
	debug = odebug;
	timerreset(&sp->t_ayt);
	timerreset(&sp->t_sync);
}

void
childrequest(struct aclgroup *gp, struct req *rp)
{
	char buf[132];

	switch (rp->acl.type) {

	case ATYPE_BLOCKHOST:
	case ATYPE_BLOCKNET:
		childsend("%s %s %s %d",
		    rp->cmd, addr2str(&rp->acl.addr1), gp->name, rp->acl.seq);
		break;

	case ATYPE_BLOCKHOSTHOST:
		(void)snprintf(buf, sizeof(buf), "%s",
		    addr2str(&rp->acl.addr2));
		childsend("%s %s %s %s %d",
		    rp->cmd, addr2str(&rp->acl.addr1), buf,
		    gp->name, rp->acl.seq);
		break;

	case ATYPE_BLOCKMAC:
		childsend("%s %s %s %d",
		    rp->cmd, mac2str(&rp->acl.mac), gp->name, rp->acl.seq);
		break;

	case ATYPE_BLOCKUDPPORT:
	case ATYPE_BLOCKTCPPORT:
	case ATYPE_BLOCKTCPSYNPORT:
		childsend("%s %s %s %d",
		    rp->cmd, rp->acl.port1, gp->name, rp->acl.seq);
		break;

	case ATYPE_PERMITUDPDSTHOSTPORT:
	case ATYPE_PERMITTCPDSTHOSTPORT:
	case ATYPE_BLOCKTCPDSTHOSTPORT:
		childsend("%s %s %s %s %d",
		    rp->cmd, addr2str(&rp->acl.addr1), rp->acl.port1,
		    gp->name, rp->acl.seq);
		break;

	default:
		lg(LOG_ERR, "childrequest: bad atype %s",
		    val2str(str2acl, rp->acl.type));
		exit(EX_SOFTWARE);
	}

	/* Update request state */
	rp->state = RSTATE_CHILD;
}

static int
childresponse(char **pp)
{
	char *cp;
	struct iobuf *ip;

	/* Strip possible leading dot and then look for end of message */
	ip = &state.rbuf;
	while ((cp = iogetstr(ip)) != NULL) {
		if (*cp == '.') {
			++cp;
			if (*cp == '\0' || *cp == '\n')
				return (1);
		}
		strappend(pp, cp);
	}
	return (0);
}

void
childsend(const char *fmt, ...)
{
	char *cp;
	size_t n;
	ssize_t cc;
	char buf[1024];
	va_list ap;

	va_start(ap, fmt);
	(void)vsnprintf(buf, sizeof(buf) - 1, fmt, ap);
	va_end(ap);
	n = strlen(buf);
	cp = buf + n - 1;

	if (debug)
		fprintf(log_lf.f, "%s childsend: \"%s\"\n", tsstr(), buf);

	if (cp >= buf) {
		*++cp = '\n';
		*++cp = '\0';
		++n;
	}
	cc = write(state.wfd, buf, n);
	if (cc < 0) {
		/* XXX kill off child here? */
		lg(LOG_ERR, "childsend: write: %s", strerror(errno));
		return;
	}
	if (cc != n) {
		lg(LOG_ERR, "childsend: short write (%d != %d)",
		    (int)cc, (int)n);
		return;
	}
}

void
childsendattr(void)
{
	int i, didany;
	const char *attr;
	struct aclgroup *gp;
	struct array *dp;
	struct state *sp;

	sp = &state;
	didany = 0;
	dp = &cf->c_aclgroup_array;
	for (i = 0, gp = cf->c_aclgroup; i < dp->len; ++i, ++gp) {
		if (gp->sentattr)
			continue;
		if (gp->ismac)
			attr = "mac";
		else if (ACLGROUP_ISINET6(gp))
			attr = "ipv6";
		else
			continue;

		setastate(ASTATE_SENTATTR);
		sp->cmd = "attr";
		childsend("%s %s %s", sp->cmd, gp->name, attr);
		++gp->sentattr;
		++didany;
		break;
	}
	if (!didany)
		++sp->sentattrs;
}
