/*
 * Copyright (c) 2018, 2019, 2020, 2024
 * The Regents of the University of California. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *     * Redistributions of source code must retain the above copyright
 *       notice, this list of conditions and the following disclaimer.
 *     * Redistributions in binary form must reproduce the above copyright
 *       notice, this list of conditions and the following disclaimer in the
 *       documentation and/or other materials provided with the distribution.
 *     * Neither the name of the University nor the names of its contributors
 *       may be used to endorse or promote products derived from this software
 *       without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#ifndef lint
static const char rcsid[] __attribute__ ((unused)) =
    "@(#) $Id$ (LBL)";
#endif

#include <sys/types.h>
#include <sys/time.h>

#include <errno.h>
#include <signal.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sysexits.h>
#include <time.h>
#include <unistd.h>

#include "acld.h"
#include "child2.h"

/* child2 version of REQUESTDONE() */
#define REQUESTDONE2(rp) \
    ((rp) != NULL && \
    (rp)->state == RSTATE_DONE && \
    (rp)->child2notdone == 0)

/* Return 1 if we did anything */
void
child2connect(fd_set *rfp, fd_set *wfp)
{
	int i;
	struct child *chp;
	struct array *dp;
	struct state *sp;

	sp = &state;
	dp = &sp->chp2_array;
	for (i = 0, chp = sp->chp2; i < dp->len; ++i, ++chp) {
// lg(LOG_DEBUG, "# %s: child2connect: entry", chp->name);
		if (chp->state == ASTATE_CONNECTING &&
		    chp->f_childconnect != NULL &&
		    ((chp->rfd >= 0 && FD_ISSET(chp->rfd, rfp)) ||
		    (chp->wfd >= 0 && FD_ISSET(chp->wfd, wfp))))
			(chp->f_childconnect)(chp);
	}
}

void child2freereq(struct req *rp)
{
	int i;
	struct child *chp;
	struct array *dp;
	struct state *sp;

	sp = &state;
	dp = &sp->chp2_array;
	for (i = 0, chp = sp->chp2; i < dp->len; ++i, ++chp)
		if (chp->curreq == rp) {
			chp->curreq = NULL;
			chp->eatresponse = 1;
		}
}

void
child2input(void)
{
	int i;
	struct child *chp;
	struct req *rp;
	struct array *dp;
	struct state *sp;

	sp = &state;
	dp = &sp->chp2_array;
	for (i = 0, chp = sp->chp2; i < dp->len; ++i, ++chp) {
		if (chp->f_childinput == NULL)
			continue;
		while (IOBUF_LEN(&chp->rbuf) > 0 && !(chp->f_childinput)(chp))
			continue;
		rp = chp->req;
		// XXX more than one request could be done here
		if (REQUESTDONE2(rp)) {
			/* Remove from request list */
			chp->req = rp->next;
			rp->next = NULL;
			freereq(rp);
			return;
		}
	}
}

void
child2poll(fd_set *rfp, fd_set *wfp)
{
	int i;
	ssize_t cc;
	struct req *rp;
	struct child *chp;
	struct array *dp;
	struct state *sp;

	sp = &state;
	dp = &sp->chp2_array;
	for (i = 0, chp = sp->chp2; i < dp->len; ++i, ++chp) {
		/* Remove server request from list when completed */
		rp = chp->req;
		if (REQUESTDONE2(rp)) {
			/* Remove from request list */
			chp->req = rp->next;
			rp->next = NULL;
			freereq(rp);
		}

		switch (chp->state) {

		case ASTATE_NOTCONNECTED:
			/* Did the child fall asleep? */
			if (chp->f_childlogin != NULL &&
			    timerdue(&chp->t_login)) {
				(chp->f_childlogin)(chp);
				return;
			}

			/* Reset timers if not pending */
			if (chp->f_childlogin != NULL &&
			    timercheck(&chp->t_login) < 0)
				timerset(&chp->t_login, cf->c_login_secs2);
			break;

		case ASTATE_CONNECTING:
			/* Don't interfer with connect */
			return;

		case ASTATE_CONNECTED:
			if (chp->f_childbusy != NULL && (chp->f_childbusy)(chp))
				break;
			if (chp->f_childayt != NULL) {
				(chp->f_childayt)(chp);
				child2setastate(chp, ASTATE_SENTLOGIN);
				return;
			}
			break;

		case ASTATE_LOGGEDIN:
			/* Poke the child */
			if (cf->c_ayt_secs2 > 0 && chp->f_childayt != NULL) {
				if (chp->f_childbusy != NULL &&
				    (chp->f_childbusy)(chp))
					break;

				if (timercheck(&chp->t_ayt) == 0)
					(chp->f_childayt)(chp);
			}
			break;

		default:
			break;
		}

		/* Read */
		if (chp->f_childread != NULL &&
		    chp->rfd >= 0 && FD_ISSET(chp->rfd, rfp)) {
			cc = (chp->f_childread)(chp);
			if (cc <= 0) {
				if (cc == 0)
					lg(LOG_DEBUG, "%s: child disconnect",
					    chp->name);
				else
					lg(LOG_ERR, "%s: child read: %s",
					    chp->name, strerror(errno));
				if (chp->state != ASTATE_NOTCONNECTED)
					(chp->f_childkill)(chp);
				return;
			}
		}

		/* Write */
		if (chp->f_childwrite != NULL &&
		    chp->wfd >= 0 && FD_ISSET(chp->wfd, wfp)) {
			cc = (chp->f_childwrite)(chp);
			if (cc <= 0) {
				if (cc == 0)
					lg(LOG_DEBUG, "%s: child disconnect",
					    chp->name);
				else
					lg(LOG_ERR,
					    "%s: second child write: %s",
					    chp->name, strerror(errno));
				if (chp->state != ASTATE_NOTCONNECTED)
					(chp->f_childkill)(chp);
				return;
			}
		}
	}
}
void
child2clientsend(struct child *chp, struct client *cl, struct req *rp)
{
	/* This child/request is done */
	rp->child2pending &= ~CHILD2BIT(chp);
	rp->child2notdone &= ~CHILD2BIT(chp);
	chp->curreq = NULL;

	/* Wait for other children to finish */
	if (rp->child2notdone == 0)
		clientsend(cl, rp);
}

void
child2clientsenderr(struct child *chp, struct client *cl, struct req *rp,
    const char *fmt, ...)
{
	va_list ap;

	// iofree(&rp->payload);
	va_start(ap, fmt);
	ioappendvfmt(&rp->payload, fmt, ap);
	va_end(ap);
	rp->flags |= (RFLAG_FAILED | RFLAG_CONTINUE);
	child2clientsend(chp, cl, rp);
}

void
child2clientsendfmt(struct child *chp, struct client *cl, struct req *rp,
    const char *fmt, ...)
{
	va_list ap;

	va_start(ap, fmt);
	ioappendvfmt(&rp->payload, fmt, ap);
	va_end(ap);
	rp->flags |= RFLAG_CONTINUE;
	child2clientsend(chp, cl, rp);
}

void
child2setastate(struct child *chp, enum acldstate astate2)
{
	/* Kill login timer when we login */
	if (!CHILD2LOGGEDIN(chp->state) && CHILD2LOGGEDIN(astate2))
		timerreset(&chp->t_login);

	/* Track logged in vs. not logged in */
	if (CHILD2LOGGEDIN(chp->state) ^ CHILD2LOGGEDIN(astate2))
		chp->loginout_ts = time(NULL);
	chp->state = astate2;
}

/* Return a possibly lower number of select seconds */
int
child2settimeout(int secs)
{
	int i, t;
	struct child *chp;
	struct array *dp;
	struct state *sp;

	sp = &state;
	dp = &sp->chp2_array;
	for (i = 0, chp = sp->chp2; i < dp->len; ++i, ++chp) {
		if (chp->f_childlogin != NULL) {
			t = timercheck(&chp->t_login);
			if (t >= 0 && secs > t)
				secs = t;
		}
		if (cf->c_ayt_secs2 > 0 && chp->f_childayt != NULL) {
			t = timercheck(&chp->t_ayt);
			if (t >= 0 && secs > t)
				secs = t;
		}
	}
	return (secs);
}

void
child2state(struct req *rp)
{
	int i;
	const char *p;
	struct child *chp;
	char buf[256];
	struct array *dp;
	struct state *sp;

	sp = &state;
	dp = &sp->chp2_array;
	for (i = 0, chp = sp->chp2; i < dp->len; ++i, ++chp) {
		p = val2str(astate2str, chp->state);
		if (p == NULL) {
			(void)snprintf(buf, sizeof(buf), "#%d", chp->state);
			p = buf;
		}
		ioappendfmt(&rp->payload, "# %s state %s", chp->name, p);
	}
}

/* Returns 1 if at least one child is logged in */
int
child2ready(void)
{
	int i;
	struct child *chp;
	struct array *dp;
	struct state *sp;

	sp = &state;
	dp = &sp->chp2_array;
	for (i = 0, chp = sp->chp2; i < dp->len; ++i, ++chp)
		if (CHILD2LOGGEDIN(chp->state))
			return (1);
	return (0);
}

void
child2reload(void)
{
	int i;
	struct child *chp;
	struct array *dp;
	struct state *sp;

	sp = &state;
	dp = &sp->chp2_array;
	for (i = 0, chp = sp->chp2; i < dp->len; ++i, ++chp) {
		if (chp->f_childkill != NULL && chp->wfd >= 0 &&
		    CHILD2LOGGEDIN(chp->state))
			(chp->f_childkill)(chp);
	}
}

/* Freshen up things and retry, return 1 if successful */
int
child2retry(struct child *chp, struct client *cl, struct req *rp,
    const char *errmsg)
{
	struct client *cl2;
	struct state *sp;
	const char *detail;
	int retries;
	char buf[32];

	sp = &state;
	retries = cf->c_retries * sp->chp2_array.len;
	if (rp->retries >= retries)
		return (0);

	++rp->retries;
	rp->state = RSTATE_PENDING;
	rp->child2pending |= CHILD2BIT(chp);
	rp->child2notdone |= CHILD2BIT(chp);
	child2setastate(chp, ASTATE_LOGGEDIN);
	chp->curreq = NULL;
	if (cl == NULL) {
		detail = "system";
	} else {
		(void)snprintf(buf, sizeof(buf), " client #%u", cl->n);
		detail = buf;
	}
	lg(LOG_ERR, "%s: corsaparse:%s %sretry %d: %s",
	    chp->name, detail, (rp->retries >= retries) ? "final " : "",
	    rp->retries, errmsg);

	/* Insure we're first and defeat client round robin */
	cl2 = sp->clients;
	if (cl2 != cl && cl2->next != NULL) {
		while (cl2->next != NULL && cl2->next != cl)
			cl2 = cl2->next;

		/* Make sure we found the guy before us */
		if (cl2->next != cl) {
			lg(LOG_ERR, "child2retry: couldn't find child #%d ",
			    cl->n);
			abort();
		}
		cl2->next = cl->next;
		cl->next = sp->clients;
		sp->clients = cl;
	}
	return (1);
}

int
child2update_fd_set(fd_set *rfp, fd_set *wfp, int nfds)
{
	int i;
	struct child *chp;
	struct array *dp;
	struct state *sp;

	sp = &state;
	dp = &sp->chp2_array;
	for (i = 0, chp = sp->chp2; i < dp->len; ++i, ++chp) {
		if (chp->rfd >= 0) {
			FD_SET(chp->rfd, rfp);
			nfds = MAX(nfds, chp->rfd);
		}

		if (chp->wfd >= 0) {
			FD_SET(chp->wfd, wfp);
			nfds = MAX(nfds, chp->wfd);
		}
	}
	return (nfds);
}
