/*
 * Copyright (c) 2002, 2003, 2004, 2006, 2007, 2008, 2009, 2010, 2011, 2012, 2013, 2014, 2015, 2016, 2017, 2018, 2019, 2020, 2021, 2022, 2023, 2024
 * The Regents of the University of California. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *     * Redistributions of source code must retain the above copyright
 *       notice, this list of conditions and the following disclaimer.
 *     * Redistributions in binary form must reproduce the above copyright
 *       notice, this list of conditions and the following disclaimer in the
 *       documentation and/or other materials provided with the distribution.
 *     * Neither the name of the University nor the names of its contributors
 *       may be used to endorse or promote products derived from this software
 *       without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#ifndef lint
static const char rcsid[] __attribute__ ((unused)) =
    "@(#) $Id$ (LBL)";
#endif

#include <sys/types.h>
#include <sys/time.h>

#include <arpa/inet.h>

#include <ctype.h>
#include <errno.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sysexits.h>
#include <unistd.h>

#include "acld.h"
#include "cf.h"
#include "child.h"
#include "child2.h"
#include "client.h"
#ifdef __FreeBSD__
#include "freebsd.h"
#endif
#ifdef HAVE_JUNOSCRIPT
#include "junoscript.h"
#endif
#include "prefix.h"

/* Globals */
struct s2v cmd2req[] = {
	{ "addmacwhitelist",	REQ_ADDMACWHITELIST },
	{ "addwhitelist",	REQ_ADDWHITELIST },
	{ "blockhosthost",	REQ_BLOCKHOSTHOST },
	{ "blockmac",		REQ_BLOCKMAC },
	{ "compact",		REQ_COMPACT },
	{ "drop",		REQ_DROP },
	{ "droptcpdsthostport", REQ_DROPTCPDSTHOSTPORT },
	{ "droptcpport",	REQ_DROPTCPPORT },
	{ "droptcpsynport",	REQ_DROPTCPSYNPORT },
	{ "dropudpport",	REQ_DROPUDPPORT },
	{ "exit",		REQ_EXIT },
	{ "filter",		REQ_FILTER },
	{ "help",		REQ_HELP },
	{ "listacl",		REQ_LISTACL },
	{ "listmac",		REQ_LISTMAC },
	{ "listnullzero",	REQ_LISTNULLZERO },
	{ "listprefix",		REQ_LISTPREFIX },
	{ "listroute",		REQ_LISTROUTE },
	{ "macwhitelist",	REQ_MACWHITELIST },
	{ "nofilter",		REQ_NOFILTER },
	{ "nonullzero",		REQ_NONULLZERO },
	{ "noprefix",		REQ_NOPREFIX },
	{ "nullzero",		REQ_NULLZERO },
	{ "permittcpdsthostport", REQ_PERMITTCPDSTHOSTPORT },
	{ "permitudpdsthostport", REQ_PERMITUDPDSTHOSTPORT },
	{ "prefix",		REQ_PREFIX },
	{ "query",		REQ_QUERY },
	{ "queryfilter",	REQ_QUERYFILTER },
	{ "querymac",		REQ_QUERYMAC },
	{ "querymacwhitelist",	REQ_QUERYMACWHITELIST },
	{ "querynullzero",	REQ_QUERYNULLZERO },
	{ "querywhitelist",	REQ_QUERYWHITELIST },
	{ "reload",		REQ_RELOAD },
	{ "remmacwhitelist",	REQ_REMMACWHITELIST },
	{ "remwhitelist",	REQ_REMWHITELIST },
	{ "replicant",		REQ_REPLICANT },
	{ "restore",		REQ_RESTORE },
	{ "restorehosthost",	REQ_RESTOREHOSTHOST },
	{ "restoremac",		REQ_RESTOREMAC },
	{ "restoretcpdsthostport", REQ_RESTORETCPDSTHOSTPORT },
	{ "restoretcpport",	REQ_RESTORETCPPORT },
	{ "restoretcpsynport",	REQ_RESTORETCPSYNPORT },
	{ "restoreudpport",	REQ_RESTOREUDPPORT },
	{ "state",		REQ_STATE },
	{ "test",		REQ_TEST },		/* undocumented */
	{ "unknown",		REQ_UNKNOWN },
	{ "unpermittcpdsthostport", REQ_UNPERMITTCPDSTHOSTPORT },
	{ "unpermitudpdsthostport", REQ_UNPERMITUDPDSTHOSTPORT },
	{ "whitelist",		REQ_WHITELIST },
	{ NULL,			REQ_UNKNOWN }
};

struct s2v ctype2str[] = {
	{ "priv",		CTYPE_PRIV },
	{ "ro",			CTYPE_RO },
	{ "web",		CTYPE_WEB },
	{ NULL,			0 }
};

/* Locals */
struct mac maczero;

/* Forwards */
static int clientallowed(struct client *, struct req *);
static void clientcfstate(struct req *);
static void clienthelp(struct client *, struct req *);
static void clientlistacl(struct client *, struct req *);
static void clientmacwhitelist(struct client *, struct req *);
static int clientnumargchk(struct client *, struct req *, int, int);
static void clientquery(struct client *, struct req *);
static void clientquerymacwhitelist(struct client *, struct req *);
static void clientquerymac(struct client *, struct req *);
static void clientquerywhitelist(struct client *, struct req *);
static void clientstate(struct client *, struct req *);
static void clientwhitelist(struct client *, struct req *);
static void clientwhitelistaddrem(struct client *, struct req *);
static const char *inconsistentresultdetail(u_int32_t, int);

static enum reqtype ro_ok[] = {
	REQ_EXIT,
	REQ_HELP,
	REQ_LISTACL,
	REQ_LISTMAC,
	REQ_LISTNULLZERO,
	REQ_LISTPREFIX,
	REQ_LISTROUTE,
	REQ_QUERY,
	REQ_QUERYFILTER,
	REQ_QUERYNULLZERO,
	REQ_QUERYWHITELIST,
	REQ_STATE,
	REQ_TEST,
	REQ_UNKNOWN,
	REQ_WHITELIST,
};

/* Check if a client is allowed to execute this command */
static int
clientallowed(struct client *cl, struct req *rp)
{
	int i, n;
	const char *port;

	if (cl->type == CTYPE_PRIV)
		return (1);

	/* Handle readonly guys */
	for (i = 0, n = sizeof(ro_ok) / sizeof(ro_ok[0]); i < n; ++i)
		if (ro_ok[i] == rp->type)
			return (1);

	/* The only other client type is web registration */
	if (cl->type != CTYPE_WEB)
		return (0);

	/* These are the only additional allowed requests */
	if (rp->type != REQ_PERMITTCPDSTHOSTPORT &&
	    rp->type != REQ_UNPERMITTCPDSTHOSTPORT)
		return (0);

	/* Look for right number of arguments */
	if (rp->an != 4)
		return (0);

	/* Only two ports are allowed */
	port = rp->av[3];
	if (strcmp(port, "80") == 0 || strcmp(port, "443") == 0)
		return (1);

	/* Deny anything we don't understand */
	return (0);
}

/* Helper routine for clientstate() */
static void
clientcfstate(struct req *rp)
{
	int i, j;
	char *cp;
	size_t size, len;
	struct acllimit *alp;
	struct aclgroup *gp;
	struct intlist *iip;
	struct array *dp, *dp2;
	struct aroute *rtp;
	struct iobuf *ip;
	struct addr *a;
	struct state *sp;
	char buf[132];
#ifdef HAVE_CORSA
	struct corsa *csp;
#endif

	sp = &state;
	ip = &rp->payload;
	if (cf->c_router != NULL)
		ioappendfmt(ip, "router \"%s\"", cf->c_router);
	if (cf->c_expect != NULL)
		ioappendfmt(ip, "expect \"%s\"", cf->c_expect);
	if (cf->c_expectlog != NULL)
		ioappendfmt(ip, "expectlog \"%s\"", cf->c_expectlog);
	if (cf->c_script != NULL)
		ioappendfmt(ip, "script \"%s\"", cf->c_script);
#ifdef HAVE_JUNOSCRIPT
	if (cf->c_junoscript != 0)
		ioappendfmt(ip, "junoscript %d", cf->c_junoscript);
	if (cf->c_portssh != 0)
		ioappendfmt(ip, "portssh %d", (int)cf->c_portssh);
	if (cf->c_sshkeyfn != NULL)
		ioappendfmt(ip, "sshkeyfn \"%s\"", cf->c_sshkeyfn);
	if (cf->c_sshpubfn != NULL)
		ioappendfmt(ip, "sshpubfn \"%s\"", cf->c_sshpubfn);
	if (cf->c_junoscriptlog != NULL)
		ioappendfmt(ip, "junoscriptlog \"%s\"", cf->c_junoscriptlog);
#endif
#ifdef HAVE_CORSA
	if (cf->c_corsa != 0)
		ioappendfmt(ip, "corsa %d", cf->c_corsa);
	if (cf->c_corsalog != NULL)
		ioappendfmt(ip, "corsalog \"%s\"", cf->c_corsalog);
	dp = &cf->c_corsaservers_array;
	for (i = 0, csp = cf->c_corsaservers; i < dp->len; ++i, ++csp)
		ioappendfmt(ip, "corsaserver %s %s", csp->saddr, csp->tokenfn);
	if (cf->c_corsacacert != NULL)
		ioappendfmt(ip, "corsacacert \"%s\"", cf->c_corsacacert);
	dp = &cf->c_corsalocalnets_array;
	for (i = 0, a = cf->c_corsalocalnets; i < dp->len; ++i, ++a)
		ioappendfmt(ip, "corsalocalnet %s", addr2str(a));
#endif
#ifdef HAVE_CFORCE
	if (cf->c_cforceaddr != NULL)
		ioappendfmt(ip, "cforceaddr \"%s\"", cf->c_cforceaddr);
	if (cf->c_cforcedata != NULL)
		ioappendfmt(ip, "cforcedata \"%s\"", cf->c_cforcedata);
	if (cf->c_portcforce != 0)
		ioappendfmt(ip, "portcforce %d", (int)cf->c_portcforce);
#endif
	if (cf->c_whitelistfn != NULL)
		ioappendfmt(ip, "whitelist \"%s\"", cf->c_whitelistfn);
	if (cf->c_macwhitelistfn != NULL)
		ioappendfmt(ip, "macwhitelist \"%s\"", cf->c_macwhitelistfn);
	if (cf->c_bindaddr.family != 0)
		ioappendfmt(ip, "bindaddr %s", addr2str(&cf->c_bindaddr));
	if (cf->c_bindaddr6.family != 0)
		ioappendfmt(ip, "bindaddr6 %s", addr2str(&cf->c_bindaddr6));
	ioappendfmt(ip, "port %d", (int)cf->c_port);
	if (cf->c_portro != 0)
		ioappendfmt(ip, "portro %d", (int)cf->c_portro);
	if (cf->c_portweb != 0)
		ioappendfmt(ip, "portweb %d", (int)cf->c_portweb);
	if (cf->c_replicantaddr.family != 0)
		ioappendfmt(ip, "replicantaddr %s",
		    addr2str(&cf->c_replicantaddr));
	if (cf->c_portreplicant != 0) {
		ioappendfmt(ip, "portreplicant %d", (int)cf->c_portreplicant);
		ioappendfmt(ip, "# replicantdepth %d", sp->replicantdepth);
	}
	if (cf->c_replicate_acl +
	    cf->c_replicate_nullzero +
	    cf->c_replicate_prefix +
	    cf->c_replicate_whitelist > 0) {
		size = sizeof(buf);
		cp = buf;
		(void)strlcpy(cp, "replicate", size);
		len = strlen(cp);
		cp += len;
		size -= len;
		if (cf->c_replicate_acl != 0) {
			(void)strlcpy(cp, " acl", size);
			len = strlen(cp);
			cp += len;
			size -= len;
		}
		if (cf->c_replicate_nullzero != 0) {
			(void)strlcpy(cp, " nullzero", size);
			len = strlen(cp);
			cp += len;
			size -= len;
		}
		if (cf->c_replicate_prefix != 0) {
			(void)strlcpy(cp, " prefix", size);
			len = strlen(cp);
			cp += len;
			size -= len;
		}
		if (cf->c_replicate_primary) {
			(void)strlcpy(cp, " primary", size);
			len = strlen(cp);
			cp += len;
			size -= len;
		}
		if (cf->c_replicate_whitelist != 0) {
			(void)strlcpy(cp, " whitelist", size);
			len = strlen(cp);
			cp += len;
			// size -= len;
		}
		*cp = '\0';
		ioappendfmt(ip, "%s", buf);
	}
	if (cf->c_netsfac != 0)
		ioappendfmt(ip, "netsfac %s",
		    val2str(syslog2str, cf->c_netsfac));
	if (cf->c_id != NULL)
		ioappendfmt(ip, "id \"%s\"", cf->c_id);
	if (cf->c_device != NULL)
		ioappendfmt(ip, "device \"%s\"", cf->c_device);
	if (cf->c_login_secs != LOGIN_SECS)
		ioappendfmt(ip, "login_secs %d", cf->c_login_secs);
	if (cf->c_login_secs2 != LOGIN_SECS2)
		ioappendfmt(ip, "login_secs2 %d", cf->c_login_secs2);
	if (cf->c_ayt_secs != AYT_SECS)
		ioappendfmt(ip, "ayt_secs %d", cf->c_ayt_secs);
	if (cf->c_ayt_secs2 != 0)
		ioappendfmt(ip, "ayt_secs2 %d", cf->c_ayt_secs2);
	if (cf->c_sync_secs != SYNC_SECS)
		ioappendfmt(ip, "sync_secs %d", cf->c_sync_secs);
	if (cf->c_select_secs != SELECT_SECS)
		ioappendfmt(ip, "select_secs %d", cf->c_select_secs);
	ioappendfmt(ip, "incrseq %d", cf->c_incrseq);

	ioappendfmt(ip, "ipv4_maxwidth %d", cf->c_ipv4_maxwidth);
	ioappendfmt(ip, "ipv6_maxwidth %d", cf->c_ipv6_maxwidth);

	ioappendfmt(ip, "ipv4_maxnullzerowidth %d",
	    cf->c_ipv4_maxnullzerowidth);
	ioappendfmt(ip, "ipv6_maxnullzerowidth %d",
	    cf->c_ipv6_maxnullzerowidth);

	if (cf->c_lowseq >= 0 || cf->c_highseq >= 0)
		ioappendfmt(ip, "seqrange %d %d", cf->c_lowseq, cf->c_highseq);
	if (cf->c_maxseq != 0)
		ioappendfmt(ip, "maxseq %d", cf->c_maxseq);

	if (cf->c_retries != DEFAULT_RETRIES)
		ioappendfmt(ip, "retries %d", cf->c_retries);

	if (cf->c_lowportseq >= 0 || cf->c_highportseq >= 0)
		ioappendfmt(ip, "portseqrange %d %d",
		    cf->c_lowportseq, cf->c_highportseq);

	if (cf->c_lowpermithostportseq >= 0 || cf->c_highpermithostportseq >= 0)
		ioappendfmt(ip, "permithostportseqrange %d %d",
		    cf->c_lowpermithostportseq, cf->c_highpermithostportseq);

#ifdef __FreeBSD__
	if (cf->c_freebsdroutes != 0)
		ioappendfmt(ip, "freebsdroutes %d", cf->c_freebsdroutes);
	if (cf->c_nullzeroif != NULL)
		ioappendfmt(ip, "nullzeroif \"%s\"", cf->c_nullzeroif);
	if (cf->c_nullzeroaddr.family != 0)
		ioappendfmt(ip, "nullzeroaddr %s",
		    addr2str(&cf->c_nullzeroaddr));
	if (cf->c_nullzeroaddr6.family != 0)
		ioappendfmt(ip, "nullzeroaddr6 %s",
		    addr2str(&cf->c_nullzeroaddr6));
#endif

	if (cf->c_prefixes != NULL)
		prefixstate(ip);

#ifdef HAVE_CFORCE
	/* Don't display lastseq when using the cForce */
	if (cf->c_cforceaddr == NULL) {
#endif
#ifdef HAVE_JUNOSCRIPT
	/* Don't display lastseq when using junoscript */
	if (cf->c_junoscript == 0) {
#endif
		dp = &cf->c_aclgroup_array;
		for (i = 0, gp = cf->c_aclgroup; i < dp->len; ++i, ++gp) {
			ioappendfmt(ip, "# lastseq for ACL %s is %d",
			    gp->name, gp->lastseq);
			if (cf->c_lowpermithostportseq > 0)
			    ioappendfmt(ip,
				"# lastpermithostportseq for ACL %s is %d",
				gp->name, gp->lastpermithostportseq);
			if (cf->c_lowportseq > 0)
			    ioappendfmt(ip, "# lastportseq for ACL %s is %d",
				gp->name, gp->lastportseq);
		}
#ifdef HAVE_CFORCE
	}
#endif
#ifdef HAVE_JUNOSCRIPT
	}
#endif

	dp = &cf->c_aclgroup_array;
	for (i = 0, gp = cf->c_aclgroup; i < dp->len; ++i, ++gp)
		ioappendfmt(ip, "# acllen for ACL %s is %d",
		    gp->name, (int)gp->acltotallen);
	dp = &cf->c_acllimit_array;
	for (i = 0, alp = cf->c_acllimit; i < dp->len; ++i, ++alp)
		ioappendfmt(ip, "# limit %d/%d %s", alp->a_len,
		    alp->a_maxseq, fmtargv(alp->a_nacls, alp->a_acls));
	dp = &cf->c_aclgroup_array;
	for (i = 0, gp = cf->c_aclgroup; i < dp->len; ++i, ++gp)
		ioappendfmt(ip, "# hiwater for ACL %s is %d",
		    gp->name, stats_gethiwater(&gp->stats));
#ifdef HAVE_CFORCE
	if (cf->c_cforceaddr != NULL)
#endif
		ioappendfmt(ip, "# hiwater for nullzero routes is %d",
		    stats_gethiwater(&sp->nullzerostats));
	dp = &cf->c_aclgroup_array;
	for (i = 0, gp = cf->c_aclgroup; i < dp->len; ++i, ++gp)
		ioappendfmt(ip, "# rate for ACL %s is %d acl/min",
		    gp->name, stats_getrate(&gp->stats, gp->name));
#ifdef HAVE_CFORCE
	if (cf->c_cforceaddr != NULL)
#endif
		ioappendfmt(ip, "# rate for nullzero routes is %d routes/min",
		    stats_getrate(&sp->nullzerostats, "nullzero"));
#ifdef HAVE_CORSA
	if (cf->c_corsa)
		ioappendfmt(ip, "# rate for filter addresses is %d per min",
		    stats_getrate(&sp->filterstats, "filter"));
#endif
	dp = &cf->c_aclgroup_array;
	for (i = 0, gp = cf->c_aclgroup; i < dp->len; ++i, ++gp) {
		if (gp->compactseq > 0)
			ioappendfmt(ip, "# compactseq for ACL %s is %d",
			    gp->name, gp->compactseq);
		if (gp->compactportseq > 0)
			ioappendfmt(ip, "# compactportseq for ACL %s is %d",
			    gp->name, gp->compactportseq);
		if (gp->compactpermithostportseq > 0)
			ioappendfmt(ip,
			    "# compactpermithostportseq for ACL %s is %d",
			    gp->name, gp->compactpermithostportseq);
	}
	dp = &cf->c_whitelist_array;
	if (dp->len != 0)
		ioappendfmt(ip, "# whitelistlen %d", (int)dp->len);

	dp = &cf->c_acllimit_array;
	for (i = 0, alp = cf->c_acllimit; i < dp->len; ++i, ++alp)
		ioappendfmt(ip, "limit %d %s",
		    alp->a_maxseq, fmtargv(alp->a_nacls, alp->a_acls));
	dp = &cf->c_aclgroup_array;
	for (i = 0, gp = cf->c_aclgroup; i < dp->len; ++i, ++gp) {
		dp2 = &gp->addr_array;
		for (j = 0, a = gp->addr; j < dp2->len; ++j, ++a)
			ioappendfmt(ip, "acl %s %s", gp->name, addr2str(a));
	}
	gp = aclgetmacgroup();
	if (gp != NULL)
		ioappendfmt(ip, "acl %s %s", gp->name, mac2str(&maczero));
	dp = &cf->c_intlist_array;
	for (i = 0, iip = cf->c_intlist; i < dp->len; ++i, ++iip)
		ioappendfmt(ip, "interface %s %s", iip->i_name, iip->i_acl);
	dp = &cf->c_nullzeronets_array;
	for (i = 0, rtp = cf->c_nullzeronets; i < dp->len; ++i, ++rtp)
		ioappendfmt(ip, "nullzeronet %s", addr2str(&rtp->dst));
	if (cf->c_nullzeromax != 0)
		ioappendfmt(ip, "nullzeromax %d", cf->c_nullzeromax);
}

void
clientcompact(struct client *cl, struct req *rp)
{
	struct aclgroup *gp;
	struct req *rp2;

	gp = aclfindgroupbyname(rp->aclname);
	if (gp == NULL) {
		clientsenderr(cl, rp, "ACL %s unknown", rp->aclname);
		return;
	}

	if (gp->compactseq > 0 ||
	    gp->compactportseq > 0 ||
	    gp->compactpermithostportseq > 0) {
		rp->flags |= RFLAG_IGNORE;
		clientsendfmt(cl, rp,
		    "compaction already in progress for ACL %s", rp->aclname);
		return;
	}

	syslog(LOG_INFO,
	    "clientcompact: started compacting ACL %s (lastseq %d)",
	    gp->name, gp->lastseq);

	/* Compaction only makes sense if we have a range */
	if (cf->c_lowseq != 0 || cf->c_highseq != 0) {
		if (cf->c_incrseq)
			gp->compactseq = cf->c_lowseq;
		else
			gp->compactseq = cf->c_highseq;
	}
	if (cf->c_lowportseq != 0 || cf->c_highportseq != 0)
		gp->compactportseq = cf->c_lowportseq;
	if (cf->c_lowpermithostportseq != 0 ||
	    cf->c_highpermithostportseq != 0)
		gp->compactpermithostportseq = cf->c_lowpermithostportseq;
	gp->compactclient = cl;

	/*
	 * We want to unlink this request but we can't do it here.
	 * So allocate a new one, copy stuff over and mark this one done.
	 */
	rp2 = newreq("clientcompact req");
	*rp2 = *rp;
	rp->rereq = NULL;
	rp->aclname = NULL;
	rp->comment = NULL;
	rp->payload.buf = NULL;
	IOBUF_LEN(&rp->payload) = 0;
	IOBUF_SIZE(&rp->payload) = 0;
	rp->av = NULL;
	rp->state = RSTATE_DONE;

	rp2->state = RSTATE_PENDING;
	rp2->next = NULL;
	gp->compactreq = rp2;
	timerset(&state.t_compact, 0);
}

/* XXX unfortunately, this duplicates a lot of serverdroprestore() */
void
clientdroprestore(struct client *cl, struct req *rp)
{
	const char *p, *errmsg;
	struct aclgroup *gp;
	struct acl *al;
	struct addr *wp, *tp;
	struct mac *ma;
	char buf[132];
	struct state *sp;

	sp = &state;
	errmsg = (sp->f_childcheckreq)(rp);
	if (errmsg != NULL) {
		clientsenderr(cl, rp, "%s", errmsg);
		return;
	}

	/* Find ACL list */
	switch (rp->acl.type) {

	case ATYPE_PERMITHOST:
	case ATYPE_BLOCKHOST:
	case ATYPE_PERMITNET:
	case ATYPE_BLOCKNET:
	case ATYPE_BLOCKHOSTHOST:
	case ATYPE_BLOCKTCPDSTHOSTPORT:
		/* By address */
		gp = aclfindgroupbyaddr(&rp->acl.addr1);
		if (gp == NULL) {
			clientsenderr(cl, rp, "can't find aclgroup for %s"
			    " (is the default ACL missing?)",
			    addr2str(&rp->acl.addr1));
			return;
		}
		rp->aclgroup = gp;
		break;

	case ATYPE_BLOCKUDPPORT:
	case ATYPE_BLOCKTCPPORT:
	case ATYPE_BLOCKTCPSYNPORT:
		/* By ACL name */
		gp = aclfindgroupbyname(rp->aclname);
		if (gp == NULL) {
			clientsenderr(cl, rp, "can't find ACL %s", rp->aclname);
			return;
		}
		rp->aclgroup = gp;
		break;

	case ATYPE_PERMITUDPDSTHOSTPORT:
	case ATYPE_PERMITTCPDSTHOSTPORT:
		/* Use default ACL list for permit guys */
		gp = aclfindgroupdefault(&rp->acl.addr1);
		if (gp == NULL) {
			clientsenderr(cl, rp,
			    "can't find aclgroup for %s (is the default ACL"
			    " for its address type missing?)",
			    addr2str(&rp->acl.addr1));
			return;
		}
		rp->aclgroup = gp;
		break;

	case ATYPE_BLOCKMAC:
		gp = aclgetmacgroup();
		if (gp == NULL) {
			clientsenderr(cl, rp,
			    "can't find mac address aclgroup");
			return;
		}
		rp->aclgroup = gp;
		break;

	default:
		clientsenderr(cl, rp, "%s not implemented",
		    val2str(str2acl, rp->acl.type));
		return;
	}

	/* Check if ACL should or should not already exist */
	if (gp != NULL)
		al = aclfindacl(gp, &rp->acl, 0);
	else
		al = NULL;
	switch (rp->type) {

	case REQ_DROP:
		tp = &rp->acl.addr1;
		wp = whitelist(tp);
		if (wp != NULL) {
			snprintf(buf, sizeof(buf), "%s", addr2str(tp));
			clientsenderr(cl, rp, "%s is on the whitelist (%s)",
			    buf, addr2str(wp));

			/* Log for NETS */
			nets_log(cl, rp);
			return;
		}

		if (al != NULL) {
			rp->flags |= RFLAG_IGNORE;
			clientsendfmt(cl, rp,
			    "Note: %s is already blocked in ACL %s",
			    addr2str(tp), gp->name);

			/* Log for NETS */
			nets_log(cl, rp);
			return;
		}
		stats_setrate(&gp->stats);
		break;

	case REQ_BLOCKMAC:
		ma = &rp->acl.mac;
		if (macwhitelist(ma) != NULL) {
			clientsenderr(cl, rp, "%s is on the whitelist",
			    mac2str(ma));

			/* Log for NETS */
			nets_log(cl, rp);
			return;
		}

		if (al != NULL) {
			rp->flags |= RFLAG_IGNORE;
			clientsendfmt(cl, rp,
			    "Note: %s is already blocked in ACL %s",
			    mac2str(ma), gp->name);

			/* Log for NETS */
			nets_log(cl, rp);
			return;
		}
		break;

	case REQ_RESTOREMAC:
		if (al == NULL) {
			rp->flags |= RFLAG_IGNORE;
			clientsenderr(cl, rp, "%s not found in ACL %s",
			    mac2str(&rp->acl.mac), gp->name);

			/* Log for NETS */
			nets_log(cl, rp);
			return;
		}
		break;

	case REQ_BLOCKHOSTHOST:
		tp = &rp->acl.addr1;
		wp = whitelist(tp);
		if (wp == NULL) {
			/* Check second address too */
			tp = &rp->acl.addr2;
			wp = whitelist(tp);
		}
		if (wp != NULL) {
			snprintf(buf, sizeof(buf), "%s", addr2str(tp));
			clientsenderr(cl, rp, "%s is on the whitelist (%s)",
			    buf, addr2str(wp));

			/* Log for NETS */
			nets_log(cl, rp);
			return;
		}

		if (al != NULL) {
			rp->flags |= RFLAG_IGNORE;
			snprintf(buf, sizeof(buf), "%s", addr2str(tp));
			clientsendfmt(cl, rp,
			    "Note: %s/%s is already blocked in ACL %s",
			    buf, addr2str(&rp->acl.addr2), gp->name);

			/* Log for NETS */
			nets_log(cl, rp);
			return;
		}
		stats_setrate(&gp->stats);
		break;

	case REQ_DROPUDPPORT:
	case REQ_DROPTCPPORT:
	case REQ_DROPTCPSYNPORT:
		if (al != NULL) {
			rp->flags |= RFLAG_IGNORE;
			clientsendfmt(cl, rp,
			    "Note: %s/%s is already blocked in ACL %s",
			    tcporudpreqstr(rp->type), rp->acl.port1, gp->name);

			/* Log for NETS */
			nets_log(cl, rp);
			return;
		}
		stats_setrate(&gp->stats);
		break;

	case REQ_PERMITUDPDSTHOSTPORT:
	case REQ_PERMITTCPDSTHOSTPORT:
		if (al != NULL) {
			rp->flags |= RFLAG_IGNORE;
			clientsendfmt(cl, rp,
			    "Note: %s %s/%s is already permitted in ACL %s",
			    addr2str(&rp->acl.addr1), tcporudpreqstr(rp->type),
			    rp->acl.port1, gp->name);

			/* Log for NETS */
			nets_log(cl, rp);
			return;
		}
		break;

	case REQ_DROPTCPDSTHOSTPORT:
		if (al != NULL) {
			rp->flags |= RFLAG_IGNORE;
			clientsendfmt(cl, rp,
			    "Note: %s %s/%s is already blocked in ACL %s",
			    addr2str(&rp->acl.addr1), tcporudpreqstr(rp->type),
			    rp->acl.port1, gp->name);

			/* Log for NETS */
			nets_log(cl, rp);
			return;
		}
		break;

	case REQ_RESTORE:
	case REQ_RESTOREHOSTHOST:
		if (al == NULL) {
			rp->flags |= RFLAG_IGNORE;
			clientsenderr(cl, rp, "%s not found in ACL %s",
			    addr2str(&rp->acl.addr1), gp->name);

			/* Log for NETS */
			nets_log(cl, rp);
			return;
		}
		break;

	case REQ_RESTOREUDPPORT:
	case REQ_RESTORETCPPORT:
	case REQ_RESTORETCPSYNPORT:
		if (al == NULL) {
			rp->flags |= RFLAG_IGNORE;
			clientsenderr(cl, rp, "%s/%s not found in ACL %s",
			    tcporudpreqstr(rp->type), rp->acl.port1, gp->name);

			/* Log for NETS */
			nets_log(cl, rp);
			return;
		}
		break;

	case REQ_UNPERMITUDPDSTHOSTPORT:
	case REQ_UNPERMITTCPDSTHOSTPORT:
	case REQ_RESTORETCPDSTHOSTPORT:
		if (al == NULL) {
			rp->flags |= RFLAG_IGNORE;
			clientsenderr(cl, rp, "%s:%s/%s not found in ACL %s",
			    addr2str(&rp->acl.addr1), tcporudpreqstr(rp->type),
			    rp->acl.port1, gp->name);

			/* Log for NETS */
			nets_log(cl, rp);
			return;
		}
		break;

	default:
		lg(LOG_ERR, "clientdroprestore: bad type %d (1)", rp->type);
		exit(EX_SOFTWARE);
	}

	/* If removing an ACL, check that its seq number is in the range */
#ifdef HAVE_JUNOSCRIPT
	if (cf->c_junoscript == 0) {
#endif
	switch (rp->type) {

	case REQ_RESTORE:
	case REQ_RESTOREHOSTHOST:
		/* Host sequence range */
		if (!IN_SEQRANGE(al->seq)) {
			rp->flags |= RFLAG_IGNORE;
			clientsenderr(cl, rp,
			    "seq %d in ACL %s not in host seq range (%d-%d)",
			    al->seq, gp->name, cf->c_lowseq, cf->c_highseq);

			/* Log for NETS */
			nets_log(cl, rp);
			return;
		}
		break;

	case REQ_RESTOREUDPPORT:
	case REQ_RESTORETCPPORT:
	case REQ_RESTORETCPSYNPORT:
		/* Port sequence range */
		if (!IN_PORTSEQRANGE(al->seq)) {
			rp->flags |= RFLAG_IGNORE;
			clientsenderr(cl, rp,
			    "seq %d in ACL %s not in port seq range (%d-%d)",
			    al->seq, gp->name,
			    cf->c_lowportseq, cf->c_highportseq);
			/* Log for NETS */
			nets_log(cl, rp);
			return;
		}
		break;

	case REQ_UNPERMITUDPDSTHOSTPORT:
	case REQ_UNPERMITTCPDSTHOSTPORT:
	case REQ_RESTORETCPDSTHOSTPORT:
		/* Permit host/port sequence range */
		if (!IN_PERMITHOSTPORTSEQRANGE(al->seq)) {
			rp->flags |= RFLAG_IGNORE;
			clientsenderr(cl, rp,
			    "seq %d in ACL %s not in port seq range (%d-%d)",
			    al->seq, gp->name,
			    cf->c_lowpermithostportseq,
			    cf->c_highpermithostportseq);

			/* Log for NETS */
			nets_log(cl, rp);
			return;
		}
		break;

	default:
		break;
	}
#ifdef HAVE_JUNOSCRIPT
	}
#endif

	/* Add or remove the ACL */
	switch (rp->type) {

	case REQ_DROP:
	case REQ_BLOCKHOSTHOST:
		/* Set correct lastseq when the range is empty */
		if (!IN_SEQRANGE(gp->lastseq)) {
			if (cf->c_incrseq)
				gp->lastseq = cf->c_lowseq - 1;
			else
				gp->lastseq = cf->c_highseq + 1;
		}

		/* Are we moving up or down? */
		if (cf->c_incrseq) {
			if (cf->c_highseq >= 0 &&
			    gp->lastseq >= cf->c_highseq) {
				(void)snprintf(buf, sizeof(buf),
				    "Last sequence number in ACL %s"
				    " already in use (%d)",
				    gp->name, cf->c_highseq);
				clientsenderr(cl, rp, "%s", buf);
				syslog(LOG_INFO, "%s", buf);
				return;
			}
		} else {
			if (cf->c_lowseq >= 0 &&
			    gp->lastseq <= cf->c_lowseq) {
				(void)snprintf(buf, sizeof(buf),
				    "First sequence number in ACL %s"
				    " already in use (%d)",
				    gp->name, cf->c_lowseq);
				clientsenderr(cl, rp, "%s", buf);
				syslog(LOG_INFO, "%s", buf);
				return;
			}
		}

		/* Check cap and limits */
		p = checklimits(gp);
		if (p != NULL) {
			clientsenderr(cl, rp, "%s", p);
			syslog(LOG_INFO, "%s", p);
			return;
		}

		if (cf->c_incrseq)
			rp->acl.seq = ++gp->lastseq;
		else
			rp->acl.seq = --gp->lastseq;

		/* Add here to avoid the race (remove later if it fails) */
		acladdacl(gp, &rp->acl);
		break;

	case REQ_BLOCKMAC:
		/* Add here to avoid the race (remove later if it fails) */
		acladdacl(gp, &rp->acl);
		break;

	case REQ_RESTOREMAC:
		/* Remove here to avoid the race (restore later if it fails) */
		if (!acldeleteacl(gp, &rp->acl)) {
			lg(LOG_ERR, "clientdroprestore: acldeleteacl failed");
			exit(EX_SOFTWARE);
		}
		break;

	case REQ_DROPUDPPORT:
	case REQ_DROPTCPPORT:
	case REQ_DROPTCPSYNPORT:
		/* Set correct lastportseq when the range is empty */
		if (!IN_PORTSEQRANGE(gp->lastportseq))
			gp->lastportseq = cf->c_lowportseq - 1;

		if (cf->c_highportseq >= 0 &&
		    gp->lastportseq >= cf->c_highportseq) {
			clientsenderr(cl, rp,
			    "Last port sequence number in ACL %s"
			    " already in use (%d)",
			    gp->name, cf->c_highportseq);
			return;
		}

		/* Check cap and limits */
		p = checklimits(gp);
		if (p != NULL) {
			clientsenderr(cl, rp, "%s", p);
			syslog(LOG_INFO, "%s", p);
			return;
		}

		rp->acl.seq = ++gp->lastportseq;

		/* Add here to avoid the race (remove later if it fails) */
		acladdacl(gp, &rp->acl);
		break;

	case REQ_PERMITUDPDSTHOSTPORT:
	case REQ_PERMITTCPDSTHOSTPORT:
	case REQ_DROPTCPDSTHOSTPORT:
		/* Set correct lastpermithostportseq when the range is empty */
		if (!IN_PERMITHOSTPORTSEQRANGE(gp->lastpermithostportseq))
			gp->lastpermithostportseq =
			    cf->c_lowpermithostportseq - 1;

		if (cf->c_highpermithostportseq >= 0 &&
		    gp->lastpermithostportseq >=
		    cf->c_highpermithostportseq) {
			clientsenderr(cl, rp,
			    "Last permit host/port sequence number in ACL %s"
			    " already in use (%d)",
			    gp->name, cf->c_highpermithostportseq);
			return;
		}

		/* Check cap and limits */
		p = checklimits(gp);
		if (p != NULL) {
			clientsenderr(cl, rp, "%s", p);
			syslog(LOG_INFO, "%s", p);
			return;
		}

		rp->acl.seq = ++gp->lastpermithostportseq;

		/* Add here to avoid the race (remove later if it fails) */
		acladdacl(gp, &rp->acl);
		break;

	case REQ_RESTORE:
	case REQ_RESTOREHOSTHOST:
	case REQ_RESTOREUDPPORT:
	case REQ_RESTORETCPPORT:
	case REQ_RESTORETCPSYNPORT:
	case REQ_UNPERMITUDPDSTHOSTPORT:
	case REQ_UNPERMITTCPDSTHOSTPORT:
	case REQ_RESTORETCPDSTHOSTPORT:
		rp->acl.seq = al->seq;

		/* Remove here to avoid the race (restore later if it fails) */
		if (!acldeleteacl(gp, &rp->acl)) {
			lg(LOG_ERR, "clientdroprestore: acldeleteacl failed");
			exit(EX_SOFTWARE);
		}
		break;

	default:
		lg(LOG_ERR, "clientdroprestore: bad type %d (2)", rp->type);
		exit(EX_SOFTWARE);
	}

	/* Send the request to the child */
	(sp->f_childrequest)(gp, rp);

	/* Schedule a sync */
	if (cf->c_sync_secs > 0 && timercheck(&sp->t_sync) < 0)
		timerset(&sp->t_sync, cf->c_sync_secs);
}

void
clientfinishreq(struct client *cl, struct req *rp)
{
	int w;
	long v;
	char *cp, *ep;
	const char *p;
	struct addr *a;
	struct aclgroup *gp;

	/* Only say something back if there was a command */
	if (rp->an < 1) {
		freereq(rp);
		return;
	}

	/* Parse command */
	rp->type = str2val(cmd2req, rp->av[0]);
	rp->cmd = val2str(cmd2req, rp->type);
	if (rp->type == REQ_UNKNOWN) {
		clientsenderr(cl, rp, "unknown command \"%s\"", rp->av[0]);
		freereq(rp);
		return;
	}

	/* We need a minimum of a command and a cookie */
	if (rp->an < 2) {
		clientsenderr(cl, rp, "missing cookie");
		freereq(rp);
		return;
	}

	/* Parse cookie */
	rp->cookie = strtol(rp->av[1], &ep, 10);
	if (*ep != '\0') {
		clientsenderr(cl, rp, "non-numeric cookie");
		freereq(rp);
		return;
	}

	/* Enforce client privileges */
	if (!clientallowed(cl, rp)) {
		lg(LOG_ERR,
		    "clientinput: client #%d: %s request denied for %s client",
		    cl->n, val2str(cmd2req, rp->type),
		    val2str(ctype2str, cl->type));
		clientsenderr(cl, rp, "Permission denied");
		freereq(rp);
/* XXX need to add this? cl->treq = NULL  */
		return;
	}

	/* Verify parameters and build request */
	switch (rp->type) {

	case REQ_EXIT:
	case REQ_HELP:
	case REQ_LISTNULLZERO:
	case REQ_LISTROUTE:
	case REQ_RELOAD:
	case REQ_STATE:
	case REQ_TEST:
	case REQ_WHITELIST:
		if (!clientnumargchk(cl, rp, 2, 0))
			return;
		break;

	case REQ_REPLICANT:
		if (!clientnumargchk(cl, rp, 2, 3))
			return;
		break;

	case REQ_DROP:
	case REQ_RESTORE:
	case REQ_QUERY:
		if (!clientnumargchk(cl, rp, 3, 4))
			return;

		/* Extract host or network addr */
		if (rp->an == 4)
			cp = rp->av[3];
		else
			cp = NULL;
		p = extractaddr(rp->av[2], cp, &rp->acl.addr1);
		if (p != NULL) {
			clientsenderr(cl, rp, "extractaddr: %s", p);
			freereq(rp);
			return;
		}

		p = checkmaskwidth(&rp->acl.addr1);
		if (p != NULL) {
			clientsenderr(cl, rp, "%s", p);
			freereq(rp);
			return;
		}

		if (ISHOST(&rp->acl.addr1))
			rp->acl.type = ATYPE_BLOCKHOST;
		else
			rp->acl.type = ATYPE_BLOCKNET;
		break;

	case REQ_BLOCKMAC:
	case REQ_RESTOREMAC:
	case REQ_QUERYMAC:
		if (!clientnumargchk(cl, rp, 3, 0))
			return;

		/* Extract mac addr */
		p = extractmac(rp->av[2], &rp->acl.mac);
		if (p != NULL) {
			clientsenderr(cl, rp, "extractmac: %s", p);
			freereq(rp);
			return;
		}
		rp->acl.type = ATYPE_BLOCKMAC;

		/* XXX could check for/disallow multicast or broadcast */
		break;

	case REQ_BLOCKHOSTHOST:
	case REQ_RESTOREHOSTHOST:
		if (!clientnumargchk(cl, rp, 4, 0))
			return;

		/* Extract host addrs */
		if ((p = extracthost(rp->av[2], NULL,
		    &rp->acl.addr1)) != NULL ||
		    (p = extracthost(rp->av[3], NULL,
		    &rp->acl.addr2)) != NULL) {
			clientsenderr(cl, rp, "extracthost: %s", p);
			freereq(rp);
			return;
		}

		rp->acl.type = ATYPE_BLOCKHOSTHOST;
		break;

	case REQ_DROPUDPPORT:
	case REQ_RESTOREUDPPORT:
	case REQ_DROPTCPPORT:
	case REQ_RESTORETCPPORT:
	case REQ_DROPTCPSYNPORT:
	case REQ_RESTORETCPSYNPORT:
		if (!clientnumargchk(cl, rp, 4, 0))
			return;

		if (
#ifdef HAVE_JUNOSCRIPT
		    cf->c_junoscript == 0 &&
#endif
		    (cf->c_lowportseq < 0 || cf->c_highportseq < 0)) {
			clientsenderr(cl, rp,
			    "portseqrange are not configured");
			freereq(rp);
			return;
		}

		rp->acl.port1 = strsave(rp->av[2]);
		rp->aclname = strsave(rp->av[3]);
		switch (rp->type) {

		case REQ_DROPUDPPORT:
		case REQ_RESTOREUDPPORT:
			rp->acl.type = ATYPE_BLOCKUDPPORT;
			break;

		case REQ_DROPTCPPORT:
		case REQ_RESTORETCPPORT:
			rp->acl.type = ATYPE_BLOCKTCPPORT;
			break;

		case REQ_DROPTCPSYNPORT:
		case REQ_RESTORETCPSYNPORT:
			rp->acl.type = ATYPE_BLOCKTCPSYNPORT;
			break;

		default:
			abort();
		}
		break;

	case REQ_PERMITUDPDSTHOSTPORT:
	case REQ_UNPERMITUDPDSTHOSTPORT:
		if (!clientnumargchk(cl, rp, 4, 0))
			return;

		if (
#ifdef HAVE_JUNOSCRIPT
		    cf->c_junoscript == 0 &&
#endif
		    (cf->c_lowpermithostportseq == 0 ||
		    cf->c_highpermithostportseq == 0)) {
			clientsenderr(cl, rp,
			    "permithostportseqrange are not configured");
			freereq(rp);
			return;
		}

		/* Extract host addr */
		p = extracthost(rp->av[2], NULL, &rp->acl.addr1);
		if (p != NULL) {
			clientsenderr(cl, rp, "extractaddr: %s", p);
			freereq(rp);
			return;
		}

		rp->acl.port1 = strsave(rp->av[3]);
		rp->acl.type = ATYPE_PERMITUDPDSTHOSTPORT;
		break;

	case REQ_PERMITTCPDSTHOSTPORT:
	case REQ_UNPERMITTCPDSTHOSTPORT:
		if (!clientnumargchk(cl, rp, 4, 0))
			return;

		if (
#ifdef HAVE_JUNOSCRIPT
		    cf->c_junoscript == 0 &&
#endif
		    (cf->c_lowpermithostportseq == 0 ||
		    cf->c_highpermithostportseq == 0)) {
			clientsenderr(cl, rp,
			    "permithostportseqrange are not configured");
			freereq(rp);
			return;
		}

		/* Extract host addr */
		p = extracthost(rp->av[2], NULL, &rp->acl.addr1);
		if (p != NULL) {
			clientsenderr(cl, rp, "extractaddr: %s", p);
			freereq(rp);
			return;
		}

		rp->acl.port1 = strsave(rp->av[3]);
		rp->acl.type = ATYPE_PERMITTCPDSTHOSTPORT;
		break;

	case REQ_DROPTCPDSTHOSTPORT:
	case REQ_RESTORETCPDSTHOSTPORT:
		if (!clientnumargchk(cl, rp, 4, 0))
			return;

		/* Extract host addr */
		p = extracthost(rp->av[2], NULL, &rp->acl.addr1);
		if (p != NULL) {
			clientsenderr(cl, rp, "extractaddr: %s", p);
			freereq(rp);
			return;
		}

		rp->acl.port1 = strsave(rp->av[3]);
		rp->acl.type = ATYPE_BLOCKTCPDSTHOSTPORT;
		break;

	case REQ_NULLZERO:
	case REQ_NONULLZERO:
	case REQ_QUERYNULLZERO:
		if (!clientnumargchk(cl, rp, 3, 0))
			return;

		/* Extract host addr */
		a = &rp->nullzero.dst;
		p = extractaddr(rp->av[2], NULL, a);
		if (p != NULL) {
			clientsenderr(cl, rp, "extractaddr: %s", p);
			freereq(rp);
			return;
		}

		switch (a->family) {

		case AF_INET:
			w = cf->c_ipv4_maxnullzerowidth;
			break;

		case AF_INET6:
			w = cf->c_ipv6_maxnullzerowidth;
			break;

		default:
			abort();
		}
		if (maskwidth(a) < w) {
			clientsenderr(cl, rp,
			    "nullzero cidr wider than /%d not supported", w);
			freereq(rp);
			return;
		}

		rp->nullzero.type = ROUTE_NULLZERO;
		break;

	case REQ_FILTER:
	case REQ_NOFILTER:
	case REQ_QUERYFILTER:
		if (!clientnumargchk(cl, rp, 3, 0))
			return;

		/* Extract host addr */
		p = extracthost(rp->av[2], NULL, &rp->filter);
		if (p != NULL) {
			clientsenderr(cl, rp, "extractaddr: %s", p);
			freereq(rp);
			return;
		}
		break;

	case REQ_PREFIX:
	case REQ_NOPREFIX:
		if (!clientnumargchk(cl, rp, 4, 0))
			return;

		/* Extract addr or cidr */
		p = extractaddr(rp->av[2], NULL, &rp->prefix);
		if (p != NULL) {
			clientsenderr(cl, rp, "extractaddr: %s", p);
			freereq(rp);
			return;
		}
		rp->prefixname = strsave(rp->av[3]);
		break;

	case REQ_LISTPREFIX:
		if (!clientnumargchk(cl, rp, 3, 0))
			return;
		rp->prefixname = strsave(rp->av[2]);
		break;

	case REQ_ADDMACWHITELIST:
	case REQ_QUERYMACWHITELIST:
	case REQ_REMMACWHITELIST:
		if (!clientnumargchk(cl, rp, 3, 0))
			return;

		/* Extract mac */
		p = extractmac(rp->av[2], &rp->acl.mac);
		if (p != NULL) {
			clientsenderr(cl, rp, "extractmac: %s", p);
			freereq(rp);
			return;
		}

		if (cf->c_macwhitelistfn == NULL) {
			clientsenderr(cl, rp, "mac whitelist not configured");
			freereq(rp);
			return;
		}
		break;

	case REQ_ADDWHITELIST:
	case REQ_QUERYWHITELIST:
	case REQ_REMWHITELIST:
		if (!clientnumargchk(cl, rp, 3, 4))
			return;

		/* Extract host or network addr */
		if (rp->an == 4)
			cp = rp->av[3];
		else
			cp = NULL;

		/* Extract host or network addr */
		p = extractaddr(rp->av[2], cp, &rp->whitelist);
		if (p != NULL) {
			clientsenderr(cl, rp, "extractaddr: %s", p);
			freereq(rp);
			return;
		}

		/* Loopback is immune from mask width restrictions */
		if (!isloopback(&rp->whitelist)) {
			p = checkmaskwidth(&rp->whitelist);
			if (p != NULL) {
				clientsenderr(cl, rp, "%s", p);
				freereq(rp);
				return;
			}
		}

		if (cf->c_whitelistfn == NULL) {
			clientsenderr(cl, rp, "whitelist not configured");
			freereq(rp);
			return;
		}
		break;

	case REQ_LISTACL:
	case REQ_COMPACT:
		if (!clientnumargchk(cl, rp, 3, 0))
			return;
		rp->aclname = strsave(rp->av[2]);
		break;

	case REQ_LISTMAC:
		if (!clientnumargchk(cl, rp, 2, 0))
			return;
		gp = aclgetmacgroup();
		if (gp == NULL) {
			clientsenderr(cl, rp,
			    "not configured for mac address blocking");
			freereq(rp);
			return;
		}
		rp->aclname = strsave(gp->name);

	default:
		break;
	}

	/* Validate the port values */
	cp = rp->acl.port1;
	if (cp != NULL && isdigit(*cp)) {
		v = strtol(cp, &ep, 10);
		if (*ep != '\0' || !VALID_PORT(v)) {
			clientsenderr(cl, rp, "bad port valid \"%s\"",
			    cp);
			freereq(rp);
			return;
		}
	}
	cp = rp->acl.port2;
	if (cp != NULL && isdigit(*cp)) {
		v = strtol(cp, &ep, 10);
		if (*ep != '\0' || !VALID_PORT(v)) {
			clientsenderr(cl, rp, "bad port valid \"%s\"",
			    cp);
			freereq(rp);
			return;
		}
	}

	/* Append the new request to client request queue */
	appendreq(&cl->req, rp);
}

char helpstr[] =
	"drop cookie addr [-]\r\n"
#ifdef notdef
	"drop cookie net/width [-]\r\n"
	"drop cookie net netmask [-]\r\n"
#endif
	"blockhosthost cookie addr addr [-]\r\n"
	"blockmac cookie mac [-]\r\n"
	"dropudpport cookie port acl [-]\r\n"
	"droptcpport cookie port acl [-]\r\n"
	"droptcpdsthostport cookie addr port [-]\r\n"
	"permitudpdsthostport cookie addr port [-]\r\n"
	"permittcpdsthostport cookie addr port [-]\r\n"
	"restore cookie addr [-]\r\n"
#ifdef notdef
	"restore cookie net/width [-]\r\n"
	"restore cookie net netmask [-]\r\n"
#endif
	"restorehosthost cookie addr addr [-]\r\n"
	"restoremac cookie mac [-]\r\n"
	"restoretcpdsthostport cookie addr port [-]\r\n"
	"restoretcpport cookie port acl [-]\r\n"
	"restoreudpport cookie port acl [-]\r\n"
	"unpermitudpdsthostport cookie addr port [-]\r\n"
	"unpermittcpdsthostport cookie addr port [-]\r\n"
	"filter cookie addr [-]\r\n"
	"nofilter cookie addr [-]\r\n"
	"nullzero cookie addr [-]\r\n"
	"nonullzero cookie addr [-]\r\n"
	"prefix cookie addr name [-]\r\n"
	"noprefix cookie addr name [-]\r\n"
	"addwhitelist cookie addr [-]\r\n"
	"remwhitelist cookie addr [-]\r\n"
	"state cookie\r\n"
	"listacl cookie acl\r\n"
	"listmac cookie\r\n"
	"listprefix cookie name [-]\r\n"
	"listroute cookie\r\n"
	"whitelist cookie\r\n"
	"query cookie addr\r\n"
	"querymac cookie mac [-]\r\n"
	"queryfilter cookie addr\r\n"
	"querynullzero cookie addr\r\n"
	"querywhitelist cookie addr\r\n"
	"compact cookie acl\r\n"
	"reload cookie\r\n"
	"exit cookie\r\n"
	"help cookie\r\n";

static void
clienthelp(struct client *cl, struct req *rp)
{
	rp->payload.buf = helpstr;
	IOBUF_LEN(&rp->payload) = sizeof(helpstr) - 1;
	IOBUF_SIZE(&rp->payload) = 0;		/* don't free() */
	rp->flags |= RFLAG_CONTINUE;
	clientsend(cl, rp);
}

/* Read a command from the client */
void
clientinput(struct client *cl)
{
	int i;
	char *cp;
	struct req *rp;

	rp = cl->treq;
	if (rp == NULL) {
		/* New request */
		rp = newreq("clientinput req");
		cl->treq = rp;
		rp->state = RSTATE_PENDING;

		/* Arrival timestamp */
		getts(&rp->ats);

		cp = iogetstr(&cl->rbuf);
		rp->an = makeargv(cp, &rp->av);

		/* Do we have a comment? */
		if (rp->an > 2 && strcmp(rp->av[rp->an - 1], "-") == 0) {
			--rp->an;
			rp->state = RSTATE_READCOMMENT;

			/* That's all we can do until we collect the comment */
			return;
		}
		/* Fall through and do syntax checks */
	} else {
		/* Request in progress */
		if (rp->state != RSTATE_READCOMMENT) {
			lg(LOG_ERR, "clientinput: unexpected input");
			exit(EX_SOFTWARE);
		}

		/* Collect comment lines */
		if (iocomment(&rp->comment, &cl->rbuf))
			return;

		/* Done reading the comment */
		if (rp->comment != NULL) {
			trimws(rp->comment, strlen(rp->comment));
			lg(LOG_DEBUG, "client #%d comment: \"%s\"",
			    cl->n, pretty(rp->comment));
		}
		rp->state = RSTATE_PENDING;
	}

	/* Partial request is now complete */
	cl->treq = NULL;

	if (debug) {
		fprintf(log_lf.f, "%s clientinput: client #%d: \"%s",
		    tsstr(), (int)cl->n, rp->an > 0 ? rp->av[0] : "");
		for (i = 1; i < rp->an; ++i)
			fprintf(log_lf.f, " %s", rp->av[i]);
		fprintf(log_lf.f, "\"\n");
	}

	clientfinishreq(cl, rp);
}

static void
clientlistacl(struct client *cl, struct req *rp)
{
	int i, j;
	struct acl *al;
	struct aclgroup *gp;
	struct acllist *ap;
	struct array *dp, *dp2;

	/* We only know about configured ACLs */
	gp = aclfindgroupbyname(rp->aclname);
	if (gp == NULL) {
		clientsenderr(cl, rp, "ACL %s unknown", rp->aclname);
		return;
	}

	dp = &gp->acllist_array;
	for (i = 0, ap = gp->acllist; i < dp->len; ++i, ++ap) {
		dp2 = &ap->acl_array;
		for (j = 0, al = ap->acl; j < dp2->len; ++j, ++al)
			ioappendfmt(&rp->payload, "%s", aclformat(al));
	}
	rp->flags |= RFLAG_CONTINUE;
	clientsend(cl, rp);
}

void
clientlistroute(struct client *cl, struct req *rp, int nullzeroonly)
{
	int i;
	struct aroute *rtp;
	struct array *dp;
	struct state *sp;

	sp = &state;
	dp = &sp->routes_array;
	for (i = 0, rtp = sp->routes; i < dp->len; ++i, ++rtp) {
		if (nullzeroonly) {
			if (rtp->type == ROUTE_NULLZERO)
				ioappendfmt(&rp->payload, "%s",
				    addr2str(&rtp->dst));
		} else
			ioappendfmt(&rp->payload, "%s", routeformat(rtp));
	}
	rp->flags |= RFLAG_CONTINUE;
	clientsend(cl, rp);
}

void
clientnullzero(struct client *cl, struct req *rp)
{
	struct aroute *rtp;
	struct addr *wp;
	const char *p;
	char buf[64];
	struct state *sp;

	sp = &state;
	if (rp->nullzero.type != ROUTE_NULLZERO) {
		lg(LOG_ERR, "clientnullzero: bad routetype %d",
		    rp->nullzero.type);
		exit(EX_SOFTWARE);
	}

	/*
	 * Everything is ok (subject to whitelist check) if no
	 * configured nullzero nets
	 */
	if (cf->c_nullzeronets != NULL &&
	    !goodnullzero(&rp->nullzero.dst)) {
		clientsenderr(cl, rp, "%s not part of a configured nullzeronet",
		    addr2str(&rp->nullzero.dst));
		return;
	}

	rtp = routefind(&rp->nullzero);

	switch (rp->type) {

	case REQ_NULLZERO:
		wp = whitelist(&rp->nullzero.dst);
		if (wp != NULL) {
			snprintf(buf, sizeof(buf), "%s",
			    addr2str(&rp->nullzero.dst));
			clientsenderr(cl, rp, "%s is on the whitelist (%s)",
			    buf, addr2str(wp));

			/* Log for NETS */
			nets_log(cl, rp);
			return;
		}
		if (rtp != NULL) {
			rp->flags |= RFLAG_IGNORE;
			clientsendfmt(cl, rp,
			    "Note: %s is already nullzero routed",
			    addr2str(&rp->nullzero.dst));

			/* Log for NETS */
			nets_log(cl, rp);
			return;
		}

		p = checkmaskwidth(&rp->nullzero.dst);
		if (p != NULL) {
			clientsenderr(cl, rp, "%s", p);
			return;
		}

		if (cf->c_nullzeromax != 0 &&
		    sp->nullzerolen >= cf->c_nullzeromax) {
			rp->flags |= RFLAG_IGNORE;
			clientsenderr(cl, rp,
			    "Too many nullzero routes (%d in use, %d allowed)",
			    sp->nullzerolen, cf->c_nullzeromax);

			/* Log for NETS */
			nets_log(cl, rp);
			return;
		}
		stats_setrate(&sp->nullzerostats);

		/* Add here to avoid the race (remove later if it fails) */
		routeadd(&rp->nullzero);

		/* Queue up the command */
		childsend("%s %s", rp->cmd, addr2str(&rp->nullzero.dst));
		break;

	case REQ_NONULLZERO:
		if (rtp == NULL) {
			rp->flags |= RFLAG_IGNORE;
			clientsenderr(cl, rp, "%s not found",
			    addr2str(&rp->nullzero.dst));
			return;
		}

		p = checkmaskwidth(&rp->nullzero.dst);
		if (p != NULL) {
			clientsenderr(cl, rp, "%s", p);
			return;
		}

		/* Remove here to avoid the race (restore later if it fails) */
		if (!routedelete(&rp->nullzero)) {
			lg(LOG_ERR, "clientnullzero: deleteroute failed");
			exit(EX_SOFTWARE);
		}

		/* Queue up the command */
		childsend("%s %s", rp->cmd, addr2str(&rp->nullzero.dst));
		break;

	default:
		lg(LOG_ERR, "clientnullzero: bad type %d", rp->type);
		exit(EX_SOFTWARE);
	}

	/* Update request state */
	rp->state = RSTATE_CHILD;

	/* Schedule a sync */
	if (cf->c_sync_secs > 0 && timercheck(&sp->t_sync) < 0)
		timerset(&sp->t_sync, cf->c_sync_secs);
}

void
clientprefix(struct client *cl, struct req *rp)
{
	clientsenderr(cl, rp, "prefix not supported by this configuration");
}

/* When busy is true, the child is already busy */
// XXX need busy2?
void
clientprocess(struct client *cl, struct req *rp, int busy)
{
	int i;
	const char *errmsg;
	struct state *sp;

	/* First handle requests that never block on talking to the child */
	sp = &state;
	switch (rp->type) {

	case REQ_UNKNOWN:
		clientsenderr(cl, rp, "unknown command \"%s\"", rp->cmd);
		return;

	case REQ_STATE:
		clientstate(cl, rp);
		return;

	case REQ_EXIT:
		clientsend(cl, rp);
		lg(LOG_DEBUG, "client #%d exit", cl->n);
		/* Flag that this client is finished */
		++cl->close;
		return;

	case REQ_TEST:
		/* Generate lots of output for testing purposes */
		for (i = 1; i <= rp->cookie; ++i)
			ioappendfmt(&rp->payload, "%032d", i);
		rp->flags |= RFLAG_CONTINUE;
		clientsend(cl, rp);
		return;

	case REQ_HELP:
		clienthelp(cl, rp);
		return;

	case REQ_REPLICANT:
		cl->replicant = 1;
		if (rp->an == 3) {
			if (strcmp(rp->av[2], "primary") != 0) {
				clientsenderr(cl, rp,
				    "unknown replicant keyword \"%s\"",
				    rp->av[2]);
				return;
			}
			if (cf->c_replicate_primary) {
				/* We both think we're the primary */
				errmsg = "replicant primary conflict!";
				lg(LOG_ERR, "%s", errmsg);
				clientsenderr(cl, rp, "%s", errmsg);
				exit(EX_CONFIG);
			}
			/* The acld is the primary, bump persistent clients */
			replicantbump();
		}
		clientsend(cl, rp);

		/* Reset connect timer if we're not connected */
		if (cf->c_portreplicant != 0 && sp->refd < 0 &&
		    timercheck(&sp->t_re_connect) > 1) {
			sp->re_secs = 1;
			timerset(&sp->t_re_connect, sp->re_secs);
		}
		return;

	case REQ_MACWHITELIST:
		clientmacwhitelist(cl, rp);
		return;

	case REQ_ADDMACWHITELIST:
	case REQ_REMMACWHITELIST:
		clientwhitelistaddrem(cl, rp);
		return;

	case REQ_ADDWHITELIST:
	case REQ_REMWHITELIST:
		clientwhitelistaddrem(cl, rp);
		return;

	case REQ_QUERYMAC:
		clientquerymac(cl, rp);
		return;

	case REQ_QUERYMACWHITELIST:
		clientquerymacwhitelist(cl, rp);
		return;

	case REQ_QUERYWHITELIST:
		clientquerywhitelist(cl, rp);
		return;

	case REQ_WHITELIST:
		clientwhitelist(cl, rp);
		return;

	case REQ_FILTER:
	case REQ_NOFILTER:
#ifdef HAVE_CORSA
		/* Always an error if we are not configured */
		if (!cf->c_corsa) {
			clientsenderr(cl, rp,
			    "Not configured for filter blocks");
			return;
		}
#endif
		/* These handled elsewhere */
		return;

	case REQ_QUERYFILTER:
#ifdef HAVE_CORSA
		/* Always false if we are not configured */
		if (!cf->c_corsa) {
			clientsenderr(cl, rp, "%s not found",
			    addr2str(&rp->filter));
			return;
		}
#endif
		/* This handled elsewhere */
		return;

	default:
		/* Fall through */
		break;
	}

#ifdef __FreeBSD__
	/* Handle FreeBSD nullzero requests that don't block */
	if (cf->c_freebsdroutes) {
		switch (rp->type) {

		case REQ_LISTNULLZERO:
			/* XXX need to list corsa routes too */
			(sp->f_clientlistroute)(cl, rp, 1);
			// XXX sp->f_clientlistroute2()
			return;

		case REQ_LISTROUTE:
			/* XXX need to list corsa routes too */
			(sp->f_clientlistroute)(cl, rp, 0);
			// XXX sp->f_clientlistroute2()
			return;

		case REQ_NULLZERO:
		case REQ_NONULLZERO:
			/* XXX need to pick freebsd or corsa */
			(sp->f_clientnullzero)(cl, rp);
			return;

		case REQ_QUERYNULLZERO:
			(sp->f_clientquerynullzero)(cl, rp);
			return;

		default:
			/* Fall through */
			break;
		}
	}
#endif

	/* Bail if we're not ready to handle general requests yet */
	if (!ACLDREADY())
		return;

	/* Handle requests where we must be logged in and it's ok to be busy */
	switch (rp->type) {

	case REQ_LISTACL:
	case REQ_LISTMAC:
		clientlistacl(cl, rp);
		return;

	case REQ_LISTNULLZERO:
		(sp->f_clientlistroute)(cl, rp, 1);
		return;

	case REQ_LISTROUTE:
		(sp->f_clientlistroute)(cl, rp, 0);
		return;

	case REQ_QUERY:
		clientquery(cl, rp);
		return;

	case REQ_QUERYNULLZERO:
		(sp->f_clientquerynullzero)(cl, rp);
		return;

	case REQ_PREFIX:
	case REQ_NOPREFIX:
	case REQ_LISTPREFIX:
		(sp->f_clientprefix)(cl, rp);
		return;

	default:
		/* Fall through */
		break;
	}

	/* Bail if the child is busy */
	if (sp->f_childready != NULL && !(sp->f_childready)())
		return;

	/* Bail if we're already busy (unless multiball) */
	if (busy && !sp->multiball)
		return;

	/* Handle potentially blocking requests */
	switch (rp->type) {

	case REQ_RELOAD:
		++reload;
		clientsend(cl, rp);
		return;

	case REQ_BLOCKHOSTHOST:
	case REQ_BLOCKMAC:
	case REQ_DROP:
	case REQ_DROPTCPDSTHOSTPORT:
	case REQ_DROPTCPPORT:
	case REQ_DROPTCPSYNPORT:
	case REQ_DROPUDPPORT:
	case REQ_PERMITTCPDSTHOSTPORT:
	case REQ_PERMITUDPDSTHOSTPORT:
	case REQ_RESTORE:
	case REQ_RESTOREHOSTHOST:
	case REQ_RESTOREMAC:
	case REQ_RESTORETCPDSTHOSTPORT:
	case REQ_RESTORETCPPORT:
	case REQ_RESTORETCPSYNPORT:
	case REQ_RESTOREUDPPORT:
	case REQ_UNPERMITTCPDSTHOSTPORT:
	case REQ_UNPERMITUDPDSTHOSTPORT:
		(sp->f_clientdroprestore)(cl, rp);
		return;

	case REQ_NULLZERO:
	case REQ_NONULLZERO:
		(sp->f_clientnullzero)(cl, rp);
		return;

	case REQ_COMPACT:
		(sp->f_clientcompact)(cl, rp);
		return;

	default:
		lg(LOG_ERR, "clientprocess: unhandled request type %s (%d)",
		    val2str(cmd2req, rp->type), rp->type);
		exit(EX_SOFTWARE);
	}
}

static void
clientmacwhitelist(struct client *cl, struct req *rp)
{
	if (macwhitelistdump(&rp->payload) > 0)
		rp->flags |= RFLAG_CONTINUE;

	clientsend(cl, rp);
}

/*
 * Check for n == x when y is zero otherwise check for x <= n <= y
 * Returns 1 when OK
 */
static int
clientnumargchk(struct client *cl, struct req *rp, int x, int y)
{
	int fail;

	fail = 0;
	if (y == 0) {
		if (rp->an != x)
			++fail;

	} else if (rp->an < x || rp->an > y) {
		++fail;
	}

	if (fail) {
		clientsenderr(cl, rp, "wrong number of arguments");
		freereq(rp);
		return (0);
	}
	return (1);
}

static void
clientquery(struct client *cl, struct req *rp)
{
	struct aclgroup *gp;

	switch (rp->acl.type) {

	case ATYPE_BLOCKHOST:
	case ATYPE_BLOCKNET:
		break;

	default:
		clientsenderr(cl, rp,
		    "don't know how to handle ACL request type %d",
		    rp->acl.type);
		return;
	}

	gp = aclfindgroupbyaddr(&rp->acl.addr1);
	if (gp == NULL) {
		clientsenderr(cl, rp,
		    "can't find aclgroup for %s (is the default ACL missing?)",
		    addr2str(&rp->acl.addr1));
		return;
	}

	if (!aclquery(gp, rp)) {
		clientsenderr(cl, rp, "%s not found in ACL %s",
		    addr2str(&rp->acl.addr1), gp->name);
		return;
	}

	rp->flags |= RFLAG_CONTINUE;
	clientsend(cl, rp);
}

static void
clientquerymacwhitelist(struct client *cl, struct req *rp)
{
	struct mac *ma;

	ma = &rp->acl.mac;
	if (macwhitelist(ma) == NULL) {
		clientsenderr(cl, rp, "%s not found", mac2str(ma));
		return;
	}

	rp->flags |= RFLAG_CONTINUE;
	clientsend(cl, rp);
}

static void
clientquerymac(struct client *cl, struct req *rp)
{
	int i, present;
	struct aclgroup *gp;
	struct acllist *ap;
	struct acl *al;
	struct array *dp;

	present = 0;
	gp = aclgetmacgroup();
	if (gp != NULL) {
		ap = gp->acllist;
		dp = &ap->acl_array;
		for (i = 0, al = ap->acl; i < dp->len; ++i, ++al) {
			if (al->type == ATYPE_BLOCKMAC &&
			    ITEMEQ(&al->mac, &rp->acl.mac)) {
				++present;
				break;
			}
		}
	}

	if (present)
		clientsendfmt(cl, rp, "blockmac");
	else
		clientsenderr(cl, rp, "%s not found", mac2str(&rp->acl.mac));
}

void
clientquerynullzero(struct client *cl, struct req *rp)
{
	struct aroute *rtp;

	if (rp->nullzero.type != ROUTE_NULLZERO) {
		clientsenderr(cl, rp,
		    "don't know how to handle nullzero request type %d",
		    rp->nullzero.type);
		return;
	}

	rtp = routefind(&rp->nullzero);
	if (rtp == NULL) {
		clientsenderr(cl, rp, "%s not found",
		    addr2str(&rp->nullzero.dst));
		return;
	}

	clientsendfmt(cl, rp, "%s", routeformat(rtp));
}

static void
clientquerywhitelist(struct client *cl, struct req *rp)
{
	struct addr *a;

	a = &rp->whitelist;
	if (!whitelistquery(a, &rp->payload)) {
		clientsenderr(cl, rp, "%s not found", addr2str(a));
		return;
	}

	rp->flags |= RFLAG_CONTINUE;
	clientsend(cl, rp);
}

/* This shouldn't be called until the response payload has been collected */
void
clientsend(struct client *cl, struct req *rp)
{
	int n, failed, refailed;
	u_int32_t cookie, results;
	const char *failedp, *detail, *cmd, *p;
	struct addr *a;
	struct timeval *tvp;
	struct rereq *rep;
	struct iobuf *ip;
	struct timeval delta;
	char buf[132];
	const char inconsistent_fmt[] = "# inconsistent state %s %s";

	/* The local part of request is complete */
	rp->state = RSTATE_DONE;

	/* Wait for the replicant to finish */
	rep = rp->rereq;
	if (rep != NULL && rep->state != RESTATE_DONE)
		return;

	/* Completion timestamp */
	tvp = &rp->cts;
	if (tvp->tv_sec == 0 && tvp->tv_usec == 0)
		getts(tvp);

	cmd = rp->cmd;
	cookie = rp->cookie;

	/* Report how long retries take */
	if (rp->retries > 0) {
		/* Only filter/nofilter use retries */
		switch (rp->type) {

		case REQ_FILTER:
		case REQ_NOFILTER:
			(void)snprintf(buf, sizeof(buf), "%s %s",
			    cmd, addr2str(&rp->filter));
			detail = buf;
			break;

		default:
			detail = cmd;
			break;
		}
		timersub(&rp->cts, &rp->ats, &delta);
		lg(LOG_DEBUG, "client #%d %d %s for %s took %ld.%06ld",
		    cl->n, rp->retries,
		    (rp->retries == 1) ? "retry" : "retries",
		    detail, delta.tv_sec, delta.tv_usec);
	}

	/* Insure there is an allocated buf */
	ip = &rp->payload;
	if (ip->buf == NULL)
		ioappendbuf(ip, "");

	/* Handle replicant logic */

	if (rep != NULL) {
		/* First look for inconsistent results */
		failed = ((rp->flags & RFLAG_FAILED) != 0);
		refailed = ((rep->reflags & REFLAG_FAILED) != 0);
		if (failed == refailed) {
			/* Add the replicant response if it's different */
			// XXX not sure we need this?
			if (IOBUF_LEN(ip) == 0 &&
			    rep->response != NULL &&
			    strcmp(ip->buf, rep->response) != 0)
				    ioappendline(ip, rep->response);
		} else {
			/* We know there will be a payload */
			rp->flags |= RFLAG_CONTINUE;

			/* Insure payload for both local and replicant */
			results = 0;
			if (refailed) {
				/* Insure local payload */
				if (IOBUF_LEN(ip) == 0)
					ioappendline(ip, "# local: success");

				/* Flag presense */
				if (cf->c_replicate_primary)
					n = 1;
				else
					n = 2;
				results = (1 << n);

				/* Insure replicant payload */
				if (rep->response != NULL)
					p = rep->response;
				else
					p = "# replicant: failed";
			} else {
				/* Insure local payload */
				if (IOBUF_LEN(ip) == 0)
					ioappendline(ip, "# local: failure");

				/* Flag presense */
				if (cf->c_replicate_primary)
					n = 2;
				else
					n = 1;
				results = (1 << n);

				/* Insure replicant payload */
				if (rep->response != NULL)
					p = rep->response;
				else
					p = "# replicant: success";
			}

			/* Replicant payload last if we're primary */
			if (cf->c_replicate_primary)
				ioappendbuf(ip, p);
			else
				ioappendbuf(ip, p);

			/* Format the address */
			switch (rp->type) {

			case REQ_QUERYNULLZERO:
				a = &rp->nullzero.dst;
				break;

			case REQ_QUERYWHITELIST:
				a = &rp->whitelist;
				break;

			default:
				a = NULL;
				break;
			}

			/* Only address types we can format are supported */
			if (a != NULL) {
				/* Make this the first payload line */
				(void)snprintf(buf, sizeof(buf),
				    inconsistent_fmt, addr2str(a),
				    inconsistentresultdetail(results, 2));
				ioinsertline(&rp->payload, buf);
			}
		}
	}

	/* Consolidate payload messages if necessary */
	if (IOBUF_LEN(ip) > 0 && (rp->flags & RFLAG_CONSOLIDATE) != 0 &&
	    !ioconsolidate(ip) && rp->child2num > 0) {
		/* Format the address (only queryfilter uses this) */
		if (rp->type == REQ_QUERYFILTER)
			p = addr2str(&rp->filter);
		else
			p = "?";

		/* Make this the first payload line */
		(void)snprintf(buf, sizeof(buf), inconsistent_fmt, p,
		    inconsistentresultdetail(rp->child2results, rp->child2num));
		ioinsertline(&rp->payload, buf);
	}

	/* Report failure or success */
	if ((rp->flags & RFLAG_FAILED) != 0 ||
	    (rep != NULL && (rep->reflags & REFLAG_FAILED) != 0))
		failedp = "-failed";
	else
		failedp = "";

	(void)snprintf(buf, sizeof(buf), "%ld.%06ld %u %s%s%s\r\n",
	    tvp->tv_sec,
	    tvp->tv_usec,
	    cookie,
	    cmd,
	    failedp,
	    IOBUF_LEN(ip) > 0 ? " -" : "");
	ioappendbuf(&cl->wbuf, buf);

	if (IOBUF_LEN(ip) > 0) {
		ioappendbuf(&cl->wbuf, ip->buf);
		IOBUF_LEN(ip) = 0;
		ioappendline(&cl->wbuf, ".");
	}
}

/* Post results to all requests in state RSTATE_CHILD */
void
clientsendall(void)
{
	struct client *cl;
	struct req *rp;
	struct state *sp;

	sp = &state;
	for (cl = sp->clients; cl != NULL; cl = cl->next) {
		for (rp = cl->req; rp != NULL; rp = rp->next) {
			if (rp->state != RSTATE_CHILD)
				continue;
			if (sp->multiball && !rp->multiball)
				continue;
			/* Wrap up this request */
			clientsend(cl, rp);

			/* Log for NETS */
			nets_log(cl, rp);
		}
	}
}

void
clientsenderr(struct client *cl, struct req *rp, const char *fmt, ...)
{
	va_list ap;

	// iofree(&rp->payload);
	va_start(ap, fmt);
	ioappendvfmt(&rp->payload, fmt, ap);
	va_end(ap);
	rp->flags |= (RFLAG_FAILED | RFLAG_CONTINUE);
	clientsend(cl, rp);
}

void
clientsendallerr(const char *fmt, ...)
{
	struct client *cl;
	struct req *rp;
	va_list ap;
	struct state *sp;

	sp = &state;
	for (cl = sp->clients; cl != NULL; cl = cl->next) {
		for (rp = cl->req; rp != NULL; rp = rp->next) {
			if (rp->state != RSTATE_CHILD)
				continue;
			if (sp->multiball && !rp->multiball)
				continue;
			// iofree(&rp->payload);
			va_start(ap, fmt);
			ioappendvfmt(&rp->payload, fmt, ap);
			va_end(ap);
			rp->flags |= (RFLAG_FAILED | RFLAG_CONTINUE);
			clientsend(cl, rp);

			/* Revert changes */
			childaclcleanup(rp);

			/* Log for NETS */
			nets_log(cl, rp);
		}
	}
}

void
clientsendfmt(struct client *cl, struct req *rp, const char *fmt, ...)
{
	va_list ap;

	// iofree(&rp->payload);
	va_start(ap, fmt);
	ioappendvfmt(&rp->payload, fmt, ap);
	va_end(ap);
	rp->flags |= RFLAG_CONTINUE;
	clientsend(cl, rp);
}

static void
clientstate(struct client *cl, struct req *rp)
{
	const char *p;
	char buf[256];
	struct state *sp;

	sp = &state;
	p = val2str(astate2str, sp->state);
	if (p == NULL) {
		(void)snprintf(buf, sizeof(buf), "#%d", sp->state);
		p = buf;
	}
	ioappendfmt(&rp->payload, "# state %s", p);
	if (sp->chp2 != NULL)
		child2state(rp);
	ioappendfmt(&rp->payload, "# version %s", version);
	if (cf->c_portreplicant != 0)
		ioappendfmt(&rp->payload, "# replicant %sconnected",
		    sp->refd >= 0 ? "" : "not ");
	ioappendfmt(&rp->payload, "# host %s", hostname);
	ioappendfmt(&rp->payload, "# config \"%s\"", configfn);
	if (verbose)
		ioappendfmt(&rp->payload, "# verbose %d", verbose);
	if (debug)
		ioappendfmt(&rp->payload, "# debug %d", debug);
	ioappendfmt(&rp->payload, "# epoc %ld", sp->epoc_ts);
	if (cf != NULL)
		clientcfstate(rp);
	if (sp->nullzerolen != 0)
		ioappendfmt(&rp->payload, "# nullzerolen is %d",
		    sp->nullzerolen);
	rp->flags |= RFLAG_CONTINUE;
	clientsend(cl, rp);
}

static void
clientwhitelist(struct client *cl, struct req *rp)
{
	if (whitelistdump(&rp->payload) > 0)
		rp->flags |= RFLAG_CONTINUE;

	clientsend(cl, rp);
}

static void
clientwhitelistaddrem(struct client *cl, struct req *rp)
{
	const char *errmsg;
	struct addr *a;
	struct mac *ma;

	a = &rp->whitelist;
	ma = &rp->acl.mac;
	switch (rp->type) {

	case REQ_ADDMACWHITELIST:
		if (macwhitelist(ma) != NULL) {
			clientsendfmt(cl, rp,
			    "Note: %s is already on the whitelist",
			    mac2str(ma));

			/* Log for NETS */
			nets_log(cl, rp);
			return;
		}
		macwhitelistadd(ma);
		errmsg = macwhitelistupdate();
		if (errmsg != NULL) {
			clientsenderr(cl, rp, "%s failed:\n%s",
			    mac2str(ma), errmsg);

			/* Log for NETS */
			nets_log(cl, rp);
			return;
		}
		clientsend(cl, rp);

		/* Log for NETS */
		nets_log(cl, rp);
		break;

	case REQ_ADDWHITELIST:
		if (whitelistsearch(a) != NULL) {
			clientsendfmt(cl, rp,
			    "Note: %s is already on the whitelist",
			    addr2str(a));

			/* Log for NETS */
			nets_log(cl, rp);
			return;
		}
		whitelistadd(a);
		errmsg = whitelistupdate();
		if (errmsg != NULL) {
			clientsenderr(cl, rp, "%s failed:\n%s",
			    addr2str(a), errmsg);

			/* Log for NETS */
			nets_log(cl, rp);
			return;
		}
		clientsend(cl, rp);

		/* Log for NETS */
		nets_log(cl, rp);
		break;

	case REQ_REMMACWHITELIST:
		if (macwhitelist(ma) == NULL) {
			clientsenderr(cl, rp, "%s not found", mac2str(ma));

			/* Log for NETS */
			nets_log(cl, rp);
			return;
		}
		if (!macwhitelistrem(ma)) {
			clientsenderr(cl, rp, "%s failed", mac2str(ma));

			/* Log for NETS */
			nets_log(cl, rp);
			return;
		}
		errmsg = macwhitelistupdate();
		if (errmsg != NULL) {
			clientsenderr(cl, rp, "%s failed:\n%s",
			    mac2str(ma), errmsg);

			/* Log for NETS */
			nets_log(cl, rp);
			return;
		}
		clientsend(cl, rp);

		/* Log for NETS */
		nets_log(cl, rp);
		break;

	case REQ_REMWHITELIST:
		if (whitelistsearch(a) == NULL) {
			clientsenderr(cl, rp, "%s not found", addr2str(a));

			/* Log for NETS */
			nets_log(cl, rp);
			return;
		}
		if (!whitelistrem(a)) {
			clientsenderr(cl, rp, "%s failed", addr2str(a));

			/* Log for NETS */
			nets_log(cl, rp);
			return;
		}
		errmsg = whitelistupdate();
		if (errmsg != NULL) {
			clientsenderr(cl, rp, "%s failed:\n%s",
			    addr2str(a), errmsg);

			/* Log for NETS */
			nets_log(cl, rp);
			return;
		}
		clientsend(cl, rp);

		/* Log for NETS */
		nets_log(cl, rp);
		break;

	default:
		lg(LOG_ERR, "clientwhitelist: bad type %d", rp->type);
		exit(EX_SOFTWARE);
	}
}

void
freeclient(struct client *cl)
{
	int i;
	struct client *cl2;
	struct req *rp, *rp2;
	struct aclgroup *gp;
	struct array *dp;
	struct state *sp;

	sp = &state;
	if (cl->c >= 0) {
		(void)close(cl->c);
		cl->c = -1;
	}
	iofree(&cl->rbuf);
	iofree(&cl->wbuf);

	rp = cl->req;
	while (rp != NULL) {
		rp2 = rp->next;
		rp->next = NULL;
		freereq(rp);
		rp = rp2;
	}
	if (cl->treq != NULL)
		freereq(cl->treq);

	if (sp->clients == cl) {
		sp->clients = sp->clients->next;
	} else {
		for (cl2 = sp->clients; cl2 != NULL; cl2 = cl2->next)
			if (cl2->next == cl)
				break;
		if (cl2 == NULL) {
			lg(LOG_ERR, "freeclient: client #%d: can't happen",
			    cl->n);
			exit(EX_SOFTWARE);
		}
		cl2->next = cl->next;
	}
	cl->next = NULL;
	free(cl);

	/* Clean up compaction state */
	dp = &cf->c_aclgroup_array;
	for (i = 0, gp = cf->c_aclgroup; i < dp->len; ++i, ++gp)
		if (gp->compactclient == cl) {
			gp->compactclient = NULL;
			if (gp->compactreq != NULL) {
				freereq(gp->compactreq);
				gp->compactreq = NULL;
			}
			break;
		}
}

static const char *
inconsistentresultdetail(u_int32_t inconsistentresults, int n)
{
	int i;
	char *cp, ch;
	static char buf[(sizeof(inconsistentresults) * 8) + 1];

	if (n > sizeof(buf) - 1)
		n = sizeof(buf) - 1;
	cp = buf;
	/* First server/child is 1 */
	for (i = 1; i <= n; ++i) {
		if ((inconsistentresults & (1 << i)) != 0)
			ch = '+';
		else
			ch = '-';
		*cp++ = ch;
	}

	*cp = '\0';
	return (buf);
}

struct client *
newclient(int c)
{
	struct client *cl, *cl2;
	struct state *sp;

	sp = &state;
	cl = new(1, sizeof(*cl), "newclient client");
	IOBUF_INIT(&cl->rbuf);
	IOBUF_INIT(&cl->wbuf);
	cl->c = c;
	cl->n = sp->nclients++;
	cl->connect_ts = time(NULL);

	/* Append to end of client list */
	if (sp->clients == NULL) {
		sp->clients = cl;
	} else {
		for (cl2 = sp->clients; cl2 != NULL; cl2 = cl2->next)
			if (cl2->next == NULL)
				break;
		cl2->next = cl;
	}
	return (cl);
}
