/*
 * Copyright (c) 2018, 2019, 2020, 2024
 * The Regents of the University of California. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *     * Redistributions of source code must retain the above copyright
 *       notice, this list of conditions and the following disclaimer.
 *     * Redistributions in binary form must reproduce the above copyright
 *       notice, this list of conditions and the following disclaimer in the
 *       documentation and/or other materials provided with the distribution.
 *     * Neither the name of the University nor the names of its contributors
 *       may be used to endorse or promote products derived from this software
 *       without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#ifndef lint
static const char rcsid[] __attribute__ ((unused)) =
    "@(#) $Id$ (LBL)";
#endif

#include <sys/types.h>
#include <sys/mman.h>
#include <sys/time.h>

#include <netinet/tcp.h>

#include <arpa/inet.h>

#include <ctype.h>
#include <errno.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sysexits.h>
#include <unistd.h>

#include <json-c/json.h>
#include <openssl/ssl.h>
#include <openssl/err.h>
#include <openssl/x509_vfy.h>

#include "acld.h"
#include "child2.h"
#include "corsa.h"
#include "sslutil.h"
#include "timer.h"

/* Reset the connection when we hit this number of requests sent */
#define MAXREQUESTS 100

#define CORSABUSY(p) (IOBUF_LEN(&(p)->rbuf) > 0 || IOBUF_LEN(&(p)->wbuf) > 0)

/* Locals */
static const char gigafilter_uri[] = "/api/v1/gigafilter";
static const char http11[] = "HTTP/1.1";
static int chp2all;			/* value with all child bits set */
static struct log log_c;

/* XXX move to cf? */
static u_short port = 443;

/* Forwards */
static void corsaayt(struct child *);
static int corsabusy(struct child *);
static void corsaconnect(struct child *);
static int corsainput(struct child *);
static void corsakill(struct child *);
static void corsalimitcheck(struct child *);
static void corsalog(struct child *, const char *, int,
    const char *, const char *);
static void
corsalogpretty(const char *);
static void corsalogin(struct child *);
static int corsaparse(struct child *, struct client *, struct req *,
    const char *, int);
static void corsaparseuptime(struct child *, json_object *);
static void corsaprocess(void);
static int corsareadwrite(struct child *);
static void corsasend(struct child *, const char *, const char *, const char *);
static int corsavalidate(struct child *, struct client *, struct req *);
static int goodfilter(struct addr *);

int
corsaaddnet(const char *str)
{
	const char *errmsg;
	struct addr *a;
	struct array *dp;

	dp = &cf->c_corsalocalnets_array;
	DYNARRAY(dp, cf->c_corsalocalnets, dp->len + 1, "corsa nets");
	a = cf->c_corsalocalnets + dp->len;
	errmsg = extractaddr(str, NULL, a);
	if (errmsg != NULL) {
		lg(LOG_ERR, "corsaaddnet: extractaddr: %s", errmsg);
		return (0);
	}

	if (a->family != AF_INET) {
		lg(LOG_ERR, "corsaaddnet: Not an IPv4 network: %s", str);
		return (0);
	}

	++dp->len;
	return (1);
}

static void
corsaayt(struct child *chp)
{
	struct req *rp;

	rp = newreq("corsaserverayt req");
	rp->state = RSTATE_PENDING;
	rp->type = REQ_AYT;
	rp->cmd = "ayt";

	/* Arrival timestamp */
	getts(&rp->ats);

	/* Append the new request to end of the list */
	appendreq(&chp->req, rp);

	/* Keep track of the request */
	chp->curreq = rp;

	/* Add the request to the iobuf */
	corsasend(chp, "/api/v1/equipment/cpu", "GET", NULL);
}

static int
corsabusy(struct child *chp)
{
	return (CORSABUSY(chp));
}

static void
corsaconnect(struct child *chp)
{
	int r, err;
	struct sslclient *sup;

	sup = (struct sslclient *)chp->opaque;
	if (chp->rfd >= 0) {
		ERR_clear_error();
		r = SSL_connect(sup->ssl);
		if (r == 0) {
			err = SSL_get_error(sup->ssl, r);
			lg(LOG_DEBUG, "%s: connection to %s failed: %s",
			    chp->name, sup->remote,
			    ERR_reason_error_string(err));
			corsakill(chp);
			return;
		}

		if (r < 0) {
			if (!sslhandleerr(sup, r, "corsaconnect"))
				corsakill(chp);
			return;
		}
	}

	/* We're connected now */
	lg(LOG_DEBUG, "%s: connected to %s", chp->name, sup->remote);
	child2setastate(chp, ASTATE_CONNECTED);
	chp->numsends = 0;

	/* We only need to select() on read */
	chp->rfd = sup->s;
	chp->wfd = -1;
}

void
corsainit(void)
{
	int i;
	struct child *chp;
	struct array *dp, *dp2;
	struct corsa *csp;
	struct sslclient *sup;
	char buf[64];
	struct state *sp;

	/* Openssl */
	sp = &state;
#if OPENSSL_VERSION_NUMBER < 0x10100000L
	(void)SSL_library_init();
#else
	OPENSSL_init_ssl(0, NULL);
#endif
	SSL_load_error_strings();
	OpenSSL_add_all_algorithms();

	/* Corsa log */
	if (cf->c_corsalog != NULL) {
		log_c.fn = cf->c_corsalog;
		checklog(&log_c);
	}

	/* Stats */
	stats_init(&sp->filterstats, 10, 60, "corsainit: stats");

	dp = &cf->c_corsaservers_array;
	csp = cf->c_corsaservers;
	dp2 = &sp->chp2_array;
	/* Allocate once and avoid realloc() to allow pointers to rfd and wfd */
	DYNARRAY(dp2, sp->chp2, dp2->len + dp->len, "child2");
	dp2->len += dp->len;
	chp = sp->chp2;
	for (i = 1; i <= dp->len; ++i, ++csp, ++chp) {
		chp->n = i;
		snprintf(buf, sizeof(buf), "corsa%d", chp->n);
		chp->name = strsave(buf);
		chp->loginout_ts = time(NULL);

		chp2all |= CHILD2BIT(chp);

		/* Static initialization */
		IOBUF_INIT(&chp->rbuf);
		IOBUF_INIT(&chp->wbuf);

		/* SSL */
		sup = new(1, sizeof(*sup), "corsa SSL");
		chp->opaque = (void *)sup;
		sup->s = -1;
		sup->name = chp->name;
		sup->rfdp = &chp->rfd;
		sup->wfdp = &chp->wfd;
		sup->doingwrite = 0;
		sup->opaque = (void *)csp;

		chp->rfd = -1;
		chp->wfd = -1;
		chp->pid = -1;

		/* Global child2 hook */
		sp->f_child2process = corsaprocess;

		/* Child two hooks */
		chp->f_childayt = corsaayt;
		chp->f_childbusy = corsabusy;
		chp->f_childconnect = corsaconnect;
		chp->f_childinput = corsainput;
		chp->f_childkill = corsakill;
		chp->f_childlogin = corsalogin;
		chp->f_childread = corsareadwrite;
		chp->f_childwrite = corsareadwrite;

		/* Set login timer for first attempt */
		child2setastate(chp, ASTATE_NOTCONNECTED);
		timerset(&chp->t_login, 1);
	}
}

/* Process input from the second child, return 1 if done */
static int
corsainput(struct child *chp)
{
	int ret;
	long v;
	char *cp, *ep;
	const char *p;
	struct iobuf *ip;
	struct client *cl;
	struct req *rp, *rp2;
	/* XXX */
	struct state *sp;

	/* Find server or client request */
	sp = &state;
	rp = chp->req;
	cl = NULL;
	ip = &chp->rbuf;
	if (rp == NULL) {
		rp = NULL;
		for (cl = sp->clients; cl != NULL; cl = cl->next) {
			for (rp2 = cl->req; rp2 != NULL; rp2 = rp2->next) {
				/* Be careful in case chp->curreq was freed */
				if ((rp2->state == RSTATE_CHILD ||
				    rp2->state == RSTATE_DONE) &&
				    rp2 == chp->curreq) {
					rp = rp2;
					break;
				}
			}
			if (rp != NULL)
				break;
		}
	}

	if (rp == NULL) {
		if (chp->eatresponse) {
			chp->eatresponse = 0;
			lg(LOG_DEBUG,
			    "%s: corsainput: Eat response for aborted request",
			    chp->name);
			while ((cp = iogetstr(ip)) != NULL)
				continue;
			child2setastate(chp, ASTATE_LOGGEDIN);
			return (1);
		}
		lg(LOG_DEBUG, "%s: corsainput: input without request:",
		    chp->name);
		while ((cp = iogetstr(ip)) != NULL)
			lg(LOG_DEBUG, "corsainput: eat %s: \"%s\"",
			    chp->name, pretty(cp));
		child2setastate(chp, ASTATE_LOGGEDIN);
		return (1);
	}

	/* Likely unnecessary */
	chp->eatresponse = 0;

	/* Parse the response headers */
	ret = -1;
	while ((cp = iogetstr(ip)) != NULL) {
		if (*cp == '\r' && strcmp(cp, "\r\n") == 0)
			break;
		if (strncasecmp(cp, http11, sizeof(http11) - 1) == 0) {
			if (debug > 1)
				lg(LOG_DEBUG, "corsainput: hdr1 %s: \"%s\"",
				    chp->name, pretty(cp));
			cp = strchr(cp, ' ');
			if (cp != NULL) {
				++cp;
				v = strtol(cp, &ep, 10);
				if (*ep == ' ')
					ret = v;
			}
			continue;
		}
		if (debug > 1)
			lg(LOG_DEBUG, "corsainput: hdr2 %s: \"%s\"",
			    chp->name, pretty(cp));
	}

	if (IOBUF_LEN(ip) == 0) {
		p = "corsainput: missing json payload";
		if (cl != NULL) {
			child2clientsenderr(chp, cl, rp, "%s", p);

			/* Log for NETS */
			if (rp->child2notdone == 0)
				nets_log(cl, rp);
		} else {
			/* Can't use child2clientsend() with a client */
			chp->curreq = NULL;
			rp->state = RSTATE_DONE;
			rp->child2pending &= ~CHILD2BIT(chp);
			rp->child2notdone &= ~CHILD2BIT(chp);
			// XXX clean up server req here?
			getts(&rp->cts);
		}
		lg(LOG_ERR, "corsainput: %s %s: %s",
		    chp->name, val2str(cmd2req, rp->type), p);
		return (1);
	}
	cp = iogetstr(ip);
	// ostate = rp->state;
	rp->state = RSTATE_DONE;

	/* This child request is no longer pending */
	rp->child2pending &= ~CHILD2BIT(chp);
	rp->child2notdone &= ~CHILD2BIT(chp);

	ret = corsaparse(chp, cl, rp, cp, ret);
	if (rp->state == RSTATE_DONE && rp->child2notdone == 0)
		chp->curreq = NULL;
	return (ret);
}

static void
corsakill(struct child *chp)
{
	struct iobuf *ip;
	struct sslclient *sup;
	int delta;

	if (chp->state == ASTATE_LOGGEDIN) {
		delta = time(NULL) - chp->loginout_ts;
		lg(LOG_DEBUG,
		    "%s: corsakill: logged in for %d secs (%d requests sent)",
		    chp->name, delta, chp->numsends);
	}

	sup = (struct sslclient *)chp->opaque;
	if (sup->ssl != NULL) {
		ERR_clear_error();
		if (SSL_shutdown(sup->ssl) == 0)
			(void)SSL_shutdown(sup->ssl);
		ERR_clear_error();
		/* Also frees BIO */
		SSL_free(sup->ssl);
		sup->ssl = NULL;
		SSL_CTX_free(sup->ssl_ctx);
		sup->ssl_ctx = NULL;
	}
	if (sup->s >= 0)
		(void)close(sup->s);
	chp->rfd = -1;
	chp->wfd = -1;
	sup->s = -1;

	ip = &chp->rbuf;
	if (IOBUF_LEN(ip) != 0)
		iofree(ip);
	IOBUF_INIT(ip);
	ip = &chp->wbuf;
	if (IOBUF_LEN(ip) != 0)
		iofree(ip);
	IOBUF_INIT(ip);
	child2setastate(chp, ASTATE_NOTCONNECTED);
	// timerset(&chp->t_login, 1);
	timerreset(&chp->t_login);
	timerreset(&chp->t_ayt);
}

/*
 * The corsa closes the connection after N requests,
 * reset the connection before we hit this limit
 */
static void
corsalimitcheck(struct child *chp)
{
	if (chp->numsends >= MAXREQUESTS - 1 && chp->state == ASTATE_LOGGEDIN) {
		corsakill(chp);
		/* Immediately re-login */
		corsalogin(chp);
	}
}

static void
corsalog(struct child *chp, const char *what, int code,
    const char *p1, const char *p2)
{
	const char *tssp;

	if (log_c.fn != NULL) {
		checklog(&log_c);
		tssp = tsstr();
		fprintf(log_c.f, "%s %s%d:", tssp, what, chp->n);
		if (code > 0)
			fprintf(log_c.f, " %d", code);
		if (p1 != NULL && *p1 != '\0') {
			fputc(' ', log_c.f);
			corsalogpretty(p1);
		}
		if (p2 != NULL && *p2 != '\0') {
			fputc(' ', log_c.f);
			corsalogpretty(p2);
		}
		fputc('\n', log_c.f);
		fflush(log_c.f);
	}
}

/* Print a string to the corsalog, escaping embedded \r and \n characters */
static void
corsalogpretty(const char *p)
{
	int len;
	const char *p2;

	p2 = p;
	while (*p2 != '\0') {
		if (*p2 != '\r' && *p2 != '\n' && *p2 != '\0') {
			++p2;
			continue;
		}
		len = p2 - p;
		fprintf(log_c.f, "%.*s", len, p);
		while (*p2 == '\r' || *p2 == '\n') {
			/* Only print escaped version if there's more text */
			if (p2[1] != '\0')
			    fprintf(log_c.f, "\\%c",
				(*p2 == '\r') ?  'r' : 'n');
			++p2;
		}
		p = p2;
	}
	/* Any remaining text */
	if (*p != '\0')
		fputs(p, log_c.f);
}

static void
corsalogin(struct child *chp)
{
	const char *errmsg;
	struct addr *a;
	struct addr addr;
	struct sockaddr_storage sa;
	struct sockaddr *sap;
	struct iobuf iobuf2;
	struct iobuf *ip2;
	BIO *bio;
	socklen_t salen;
	struct sslclient *sup;
	struct corsa *csp;

	sup = (struct sslclient *)chp->opaque;
	if (sup->s >= 0)
		(void)close(sup->s);
	chp->wfd = -1;
	chp->rfd = -1;
	sup->s = -1;

	csp = (struct corsa *)sup->opaque;

	a = &addr;
	memset(a, 0, sizeof(*a));
	/* Clobber remote server */
	strlcpy(sup->remote, "?", sizeof(sup->remote));

	ip2 = &iobuf2;
	memset(ip2, 0, sizeof(*ip2));
	IOBUF_INIT(ip2);

	errmsg = extractaddr(csp->saddr, NULL, a);
	if (errmsg != NULL) {
		lg(LOG_ERR, "%s: corsalogin: %s: %s",
		    chp->name, csp->saddr, errmsg);
		exit(EX_SOFTWARE);
	}

	sap = (struct sockaddr *)&sa;
	salen = addr2sa(a, port, sap);
	(void)snprintf(sup->remote, sizeof(sup->remote), "%s.%d",
	    addr2str(a), (int)port);

	sup->s = socket(a->family, SOCK_STREAM, 0);
	if (sup->s < 0) {
		lg(LOG_ERR, "%s: corsainit: %s: %s",
		    chp->name, sup->remote, strerror(errno));
		exit(EX_SOFTWARE);
	}

	/* Non-blocking */
	if (blocking(sup->s, 0) < 0)
		exit(EX_OSERR);
	errno = 0;
	if (connect(sup->s, sap, salen) < 0) {
		if (errno != EINPROGRESS) {
			lg(LOG_ERR, "%s: connect to %s: %s",
			    chp->name, sup->remote, strerror(errno));

			/* Clean up */
			(void)close(sup->s);
			sup->s = -1;
			return;
		}
		lg(LOG_DEBUG, "%s: connecting to %s", chp->name, sup->remote);
	}

	if (sup->ssl != NULL) {
		ERR_clear_error();
		if (SSL_shutdown(sup->ssl) == 0)
			(void)SSL_shutdown(sup->ssl);
		ERR_clear_error();
		/* Also frees BIO */
		SSL_free(sup->ssl);
		sup->ssl = NULL;
	}
	if (sup->ssl_ctx != NULL) {
		SSL_CTX_free(sup->ssl_ctx);
		sup->ssl_ctx = NULL;
	}

	sup->ssl_ctx = SSL_CTX_new(SSLv23_client_method());
	if (sup->ssl_ctx == NULL) {
		lg(LOG_ERR, "%s: corsalogin: SSL_CTX_new failed: %s",
		    chp->name, ERR_reason_error_string(ERR_get_error()));
		exit(EX_SOFTWARE);
	}

	/* Good options (lifted from libfetch) */
	(void)SSL_CTX_set_options(sup->ssl_ctx,
	    SSL_OP_ALL|SSL_OP_NO_SSLv2|SSL_OP_NO_SSLv3|SSL_OP_NO_TICKET);

	sup->ssl = SSL_new(sup->ssl_ctx);
	SSL_set_connect_state(sup->ssl);
	bio = BIO_new(BIO_f_ssl());
	(void)BIO_set_nbio(bio, 1);
	SSL_set_bio(sup->ssl, bio, bio);

	SSL_set_verify(sup->ssl,
	    SSL_VERIFY_PEER | SSL_VERIFY_FAIL_IF_NO_PEER_CERT, NULL);

	if (!SSL_CTX_load_verify_locations(sup->ssl_ctx, cf->c_corsacacert,
	    NULL)) {
		lg(LOG_ERR, "%s: corsalogin: "
		    "SSL_CTX_load_verify_locations failed: %s: %s",
		    chp->name, cf->c_corsacacert,
		    ERR_reason_error_string(ERR_get_error()));
		exit(EX_SOFTWARE);
	}

	SSL_set_fd(sup->ssl, sup->s);

	chp->rfd = sup->s;

	/* Try to connect (will likely fail) */
	child2setastate(chp, ASTATE_CONNECTING);
	corsaconnect(chp);

	/* Set login retry in case this attempt fails */
	// timerset(&chp->t_login, cf->c_login_secs2);
	timerreset(&chp->t_login);
}

/* It's important to update the child state when returning */
static int
corsaparse(struct child *chp, struct client *cl, struct req *rp,
    const char *jstr, int ret)
{
	int present;
	const char *what, *info, *errmsg;
	json_object *jparp, *valop;

	corsalog(chp, "recv", ret, jstr, NULL);
	if (chp->state != ASTATE_READRESPONSE &&
	    chp->state != ASTATE_LOGGEDIN &&
	    chp->state != ASTATE_CONNECTED) {
		lg(LOG_ERR, "corsaparse: %s unexpected state %s",
		    chp->name, val2str(astate2str, chp->state));
		abort();
	}

	jparp = json_tokener_parse(jstr);
	if (jparp == NULL) {
		lg(LOG_ERR, "%s: Cannot parse json \"%s\"",
		    chp->name, pretty(jstr));
		if (cl != NULL) {
			child2clientsenderr(chp, cl, rp, "%s",
			    "Cannot parse json from server");

			/* Log for NETS */
			if (rp->child2notdone == 0)
				nets_log(cl, rp);
		}
		child2setastate(chp, ASTATE_LOGGEDIN);
		timerreset(&chp->t_login);
		corsalimitcheck(chp);
		return (1);
	}

	/* Check for errors */
	switch (ret) {

	case 200:
		/* Success */
		break;

	case 400:
		child2setastate(chp, ASTATE_LOGGEDIN);
		timerreset(&chp->t_login);
		corsalimitcheck(chp);
		what = "error";
		errmsg = "unknown";
		if (json_object_object_get_ex(jparp, what, &valop)) {
			errmsg = json_object_get_string(valop);
			/* "A GigaFilter export operation is in progress." */
			if (strstr(errmsg, "retry your command") != NULL &&
			    child2retry(chp, cl, rp, errmsg)) {
				json_object_put(jparp);
				return (1);
			    }
		}
		lg(LOG_ERR, "%s: corsaparse: %s", chp->name, errmsg);

		if (cl != NULL) {
			child2clientsenderr(chp, cl, rp,
			    "corsa error: %s", errmsg);

			/* Log for NETS */
			if (rp->child2notdone == 0)
				nets_log(cl, rp);
		}
		json_object_put(jparp);
		return (1);

	case 401:
		/* Authentication error */
		corsakill(chp);
		what = "error";
		if (json_object_object_get_ex(jparp, what, &valop)) {
			errmsg = json_object_get_string(valop);
			lg(LOG_ERR, "%s: corsaparse: %s", chp->name, errmsg);
		} else
			errmsg = "unknown";
		if (cl != NULL) {
			child2clientsenderr(chp, cl, rp,
			    "corsa auth error: %s", errmsg);
			/* Log for NETS */
			if (rp->child2notdone == 0)
				nets_log(cl, rp);
		}
		// XXX deal with requests for this client?
		json_object_put(jparp);
		return (1);

	case 404:
		/* Only queryfilter should return 404 */
		if (rp->type == REQ_QUERYFILTER)
			break;
		/* fall through */

	default:
		corsakill(chp);
		lg(LOG_ERR, "corsaparse: %s: %s: unknown http ret %d",
		    chp->name, val2str(cmd2req, rp->type), ret);
		json_object_put(jparp);
		return (1);
	}

	/* Assume we're logged in if we got here */
	child2setastate(chp, ASTATE_LOGGEDIN);
	timerreset(&chp->t_login);
	corsalimitcheck(chp);

	switch (rp->type) {

	case REQ_AYT:
		corsaparseuptime(chp, jparp);
		rp->state = RSTATE_DONE;
		getts(&rp->cts);
		json_object_put(jparp);
		return (1);

	case REQ_FILTER:
		if (ret != 200) {
			lg(LOG_ERR, "corsaparse: %s: %s: unknown http ret %d",
			    chp->name, val2str(cmd2req, rp->type), ret);
			if (cl != NULL)
				child2clientsenderr(chp, cl, rp,
				    "unknown http ret %d", ret);
			json_object_put(jparp);
			return (1);
		}

		what = "info";
		if (json_object_object_get_ex(jparp, what, &valop)) {
			info = json_object_get_string(valop);
			if (strstr(info, "already") != 0) {
				rp->flags |= RFLAG_IGNORE;
				if (cl != NULL)
					child2clientsendfmt(chp, cl, rp,
					    "Note: %s is already filtered",
					    addr2str(&rp->filter));
				break;
			}
		}
		if (cl != NULL)
			child2clientsend(chp, cl, rp);
		break;

	case REQ_NOFILTER:
		if (ret != 200) {
			lg(LOG_ERR, "corsaparse: %s: %s: unknown http ret %d",
			    chp->name, val2str(cmd2req, rp->type), ret);
			if (cl != NULL)
				child2clientsenderr(chp, cl, rp,
				    "unknown http ret %d", ret);
			json_object_put(jparp);
			return (1);
		}

		what = "info";
		if (json_object_object_get_ex(jparp, what, &valop)) {
			info = json_object_get_string(valop);
			if (strstr(info, "does not exist") != 0) {
				if (cl != NULL) {
					child2clientsenderr(chp, cl, rp,
					    "%s not found",
					    addr2str(&rp->filter));

					/* Log for NETS */
					if (rp->child2notdone == 0)
						nets_log(cl, rp);
				}
				json_object_put(jparp);
				return (1);
			}
		}
		if (cl != NULL)
			child2clientsend(chp, cl, rp);
		break;

	case REQ_QUERYFILTER:
		if (ret == 404) {
			if (cl != NULL)
				child2clientsenderr(chp, cl, rp,
				    "%s not found", addr2str(&rp->filter));
			json_object_put(jparp);
			return (1);
		}

		what = "present";
		if (json_object_object_get_ex(jparp, what, &valop)) {
			present = json_object_get_int64(valop);
			if (!present) {
				if (cl != NULL)
					child2clientsenderr(chp, cl, rp,
					    "%s not present?",
					    addr2str(&rp->filter));
				json_object_put(jparp);
				return (1);
			}
		}

		/* Mark that the address was present for this child2 */
		rp->child2results |= CHILD2BIT(chp);

		if (cl != NULL)
			child2clientsendfmt(chp, cl, rp, "%s",
			    addr2str(&rp->filter));
		json_object_put(jparp);
		return (1);

	default:
		lg(LOG_ERR, "%s: corsaparse: bad type %d", chp->name, rp->type);
		exit(EX_SOFTWARE);
	}

	/* Log for NETS */
	/* XXX include child name somehow? */
	if (cl != NULL && rp->child2notdone == 0)
		nets_log(cl, rp);
	json_object_put(jparp);
	return (1);
}

static void
corsaparseuptime(struct child *chp, json_object *jparp)
{
	int i, n;
	const char *p, *what;
	json_object *valop2;
	double avenrun[3];

	memset(avenrun, 0, sizeof(avenrun));
	n = 0;
	json_object_object_foreach(jparp, key, valop) {
		p = strchr(key, ' ');
		if (p == NULL)
			continue;
		++p;
		switch (atoi(p)) {

		case 1:
			i = 0;
			break;

		case 5:
			i = 1;
			break;

		case 15:
			i = 2;
			break;

		default:
			continue;
		}
		what = "load-avg";
		if (i >= 0 && json_object_object_get_ex(valop, what, &valop2)) {
			avenrun[i] = json_object_get_double(valop2);
			++n;
		}
	}
	if (n < 3) {
		lg(LOG_ERR, "corsaparseuptime: didn't find any load averages!");
		return;
	}
	lg(LOG_INFO, "%s: load averages: %.2f, %.2f, %.2f",
	    chp->name, avenrun[0], avenrun[1], avenrun[2]);
}

/* Process as many requests we can */
static void
corsaprocess(void)
{
	int i, ready, didany, skip, delta;
	const char *jstr, *what;
	struct addr *a;
	struct array *dp;
	struct child *chp;
	struct client *cl;
	struct req *rp;
	json_object *reqo, *itemop;
	struct state *sp;
	char path[128];
	char uri[128];

	didany = 0;

	/* Validate and handle requests that don't require child2 interaction */
	sp = &state;
	dp = &sp->chp2_array;
	for (i = 0, chp = sp->chp2; i < dp->len; ++i, ++chp) {
		if (CORSABUSY(chp))
			continue;
		if (chp->curreq != NULL)
			continue;
		for (cl = sp->clients; cl != NULL; cl = cl->next) {
			for (rp = cl->req; rp != NULL; rp = rp->next) {
				/* Want only filter requests */
				if (rp->type != REQ_FILTER &&
				    rp->type != REQ_NOFILTER &&
				    rp->type != REQ_QUERYFILTER)
					continue;

				if (rp->state != RSTATE_PENDING)
					continue;
				if (rp->child2pending != 0)
					continue;

				/* Check for viable request */
				ready = corsavalidate(chp, cl, rp);

				/*
				 * Set bits here, clear in
				 * child2clientsend() if this request
				 * is still pending
				 */
				if (rp->state != RSTATE_DONE) {
					rp->child2pending = chp2all;
					rp->child2notdone = chp2all;
					rp->flags |= RFLAG_CONSOLIDATE;

					/* Set total number of children */
					rp->child2num = dp->len;
				}

				if (!ready) {
					/* Ignore request this time through */
					rp->state = RSTATE_PENDING2;
					++didany;
					break;
				}

				break;
			}
		}
	}

	/* Handle filter/nofilter requests */
	for (i = 0, chp = sp->chp2; i < dp->len; ++i, ++chp) {
		/* Ready and not already processing a request */
		if (CORSABUSY(chp))
			continue;
		if (chp->curreq != NULL)
			continue;
		reqo = NULL;
		for (cl = sp->clients; cl != NULL; cl = cl->next) {
			for (rp = cl->req; rp != NULL; rp = rp->next) {
				if (rp->state != RSTATE_PENDING &&
				    rp->state != RSTATE_CHILD &&
				    rp->state != RSTATE_DONE)
					continue;

				if (rp->type == REQ_FILTER)
					what = "add";
				else if (rp->type == REQ_NOFILTER)
					what = "remove";
				else
					continue;

				/* Skip if child already completed request */
				if (CHILD2DONE(chp, rp))
					continue;

				/*
				 * Wait for child to login or
				 * complete existing request
				 */
				if (chp->state != ASTATE_LOGGEDIN) {
					delta = time(NULL) - chp->loginout_ts;
					if (delta > CHILD2_FAIL_SECS) {
						child2clientsenderr(chp, cl, rp,
						    "# %s not logged in",
						    chp->name);
						lg(LOG_ERR,
						    "corsaprocess: %s"
						    " not logged in",
						    chp->name);

						/* Log for NETS */
						if (cl != NULL &&
						    rp->child2notdone == 0)
							nets_log(cl, rp);
					}
					continue;
				}

				/* Flag that this child has this request */
				rp->child2pending &= ~CHILD2BIT(chp);

				// XXX should limit max number we can do here
				a = &rp->filter;
				(void)snprintf(path, sizeof(path), "/ipv4/%s",
				    addr2str(a));
				itemop = json_object_new_object();
				json_object_object_add(itemop, "op",
				    json_object_new_string(what));

				json_object_object_add(itemop, "path",
				    json_object_new_string(path));
				if (rp->type == REQ_FILTER)
					json_object_object_add(itemop, "value",
					    NULL);

				if (reqo == NULL)
					reqo = json_object_new_array();
				json_object_array_add(reqo, itemop);

				/* We're still building this request */
				rp->state = RSTATE_CHILD;
				/* XXX can't do multiple yet */
				goto hack;
			}
		}
		/* Send to child if we did any */
		if (reqo != NULL) {
hack:
			jstr = json_object_to_json_string(reqo);
			corsasend(chp, gigafilter_uri, "PATCH", jstr);
			json_object_put(reqo);
			reqo = NULL;

			/* Keep track of the request */
			chp->curreq = rp;
		}
	}

	/* Handle queryfilter */
	for (i = 0, chp = sp->chp2; i < dp->len; ++i, ++chp) {
		if (CORSABUSY(chp))
			continue;
		if (chp->curreq != NULL)
			continue;
		skip = 0;
		for (cl = sp->clients; cl != NULL && !skip; cl = cl->next) {
			for (rp = cl->req; rp != NULL; rp = rp->next) {
				if (rp->state != RSTATE_PENDING &&
				    rp->state != RSTATE_CHILD &&
				    rp->state != RSTATE_DONE)
					continue;

				if (rp->type != REQ_QUERYFILTER)
					continue;

				/* Skip if child already completed request */
				if (CHILD2DONE(chp, rp))
					continue;

				/*
				 * Wait for child to login or
				 * complete existing request
				 */
				if (chp->state != ASTATE_LOGGEDIN) {
					delta = time(NULL) - chp->loginout_ts;
					if (delta > CHILD2_FAIL_SECS) {
						child2clientsenderr(chp, cl, rp,
						    "# %s not logged in",
						    chp->name);
						lg(LOG_ERR,
						    "corsaprocess: %s"
						    " not logged in",
						    chp->name);
					}
					continue;
				}

				/* Flag that this child has this request */
				rp->child2pending &= ~CHILD2BIT(chp);

				a = &rp->filter;
				(void)snprintf(uri, sizeof(uri), "%s?ipv4=%s",
				    gigafilter_uri, addr2str(a));
				corsasend(chp, uri, "GET", NULL);

				/* Keep track of the request */
				chp->curreq = rp;

				/* Update request state */
				rp->state = RSTATE_CHILD;
				++skip;

				/* All we can do with this corsa */
				break;
			}
		}
	}

	/* Fix up request states */
	if (didany) {
		for (cl = sp->clients; cl != NULL; cl = cl->next)
			for (rp = cl->req; rp != NULL; rp = rp->next)
				if (rp->state == RSTATE_PENDING2)
					rp->state = RSTATE_PENDING;
	}
}

/*
 * Read or write from a socket and update the corresponding iobuf
 *
 * Return 1 on success, -1 when the caller should report the errno.
 * Return 0 on EOF and any other other error fatal to the child.
 */
static int
corsareadwrite(struct child *chp)
{
	int r;
	struct iobuf *ip;
	struct sslclient *sup;

	sup = (struct sslclient *)chp->opaque;
	if (sup->doingwrite) {
		ip = &chp->wbuf;
		r = sslwrite(sup->ssl, ip);
		if (r > 0) {
			if (IOBUF_LEN(ip) == 0) {
				chp->rfd = sup->s;
				chp->wfd = -1;
				sup->doingwrite = 0;
			}
			timerset(&chp->t_ayt, cf->c_ayt_secs2);
			return (1);
		}
	} else {
		r = sslread(sup->ssl, &chp->rbuf);
		if (r > 0) {
			if (chp->state == ASTATE_SENTLOGIN) {
				lg(LOG_DEBUG, "%s: logged in to %s",
				    chp->name, sup->remote);
				child2setastate(chp, ASTATE_LOGGEDIN);
				timerreset(&chp->t_login);
			} else if (chp->state == ASTATE_CONNECTED)
				child2setastate(chp, ASTATE_LOGGEDIN);
			timerset(&chp->t_ayt, cf->c_ayt_secs2);
			return (1);
		}
	}

	if (!sslhandleerr(sup, r, "corsareadwrite")) {
		corsakill(chp);
		return (0);
	}
	return (1);
}

static void
corsasend(struct child *chp, const char *uri, const char *method,
    const char *jstr)
{
	char *cp;
	struct iobuf *ip;
	struct sslclient *sup;
	struct corsa *csp;

	/* Check some assertions */
	if (CORSABUSY(chp)) {
		lg(LOG_ERR, "corsasend: can't happen (rbuf %ld, wbuf %ld)",
		    IOBUF_LEN(&(chp)->rbuf), IOBUF_LEN(&(chp)->wbuf));
		abort();
	}

	/* We are also called as part of the login process */
	if (chp->state != ASTATE_LOGGEDIN && chp->state != ASTATE_CONNECTED) {
		lg(LOG_ERR, "corsasend: can't happen (state %s)",
		    val2str(astate2str, chp->state));
		abort();
	}

	sup = (struct sslclient *)chp->opaque;
	ip = &chp->wbuf;

	csp = (struct corsa *)sup->opaque;

	ioappendfmt(ip, "%s %s HTTP/1.1", method, uri);
	/* Use ip address in place of http server name */
	ioappendfmt(ip, "Host: %s", csp->saddr);
	// ioappendline(ip, "Accept-Encoding: identity");
	// ioappendfmt(ip, "User-Agent: %s", version);
	ioappendfmt(ip, "Authorization: %s", csp->token);

	/* Do we have json to send? */
	corsalog(chp, "send", 0, uri, jstr);
	if (jstr != NULL) {
		ioappendline(ip, "Content-Type: application/json");
		ioappendfmt(ip, "Content-Length: %lu", strlen(jstr));
	}

	/* Blank line */
	ioappendline(ip, "");

	if (jstr != NULL) {
		ioappendline(ip, jstr);
		/* Remove trailing /r/n */
		IOBUF_LEN(ip) -= 2;
		cp = ip->buf + IOBUF_LEN(ip);
		*cp = '\0';
	}
	sup->doingwrite = 1;
	chp->rfd = -1;
	chp->wfd = sup->s;
	++chp->numsends;
	child2setastate(chp, ASTATE_READRESPONSE);
}

/* Returns a malloc() buffer with the token string or else NULL */
char *
corsareadtoken(const char *fn)
{
	size_t n;
	FILE *f;
	char *buf;
	char *cp;
	struct stat sbuf;

	f = fopen(fn, "r");
	if (f == NULL) {
		lg(LOG_ERR, "corsareadtoken: fopen %s: %s",
		    fn, strerror(errno));
		return (NULL);
	}
	if (fstat(fileno(f), &sbuf) < 0) {
		lg(LOG_ERR, "corsareadtoken: fstat %s: %s",
		    fn, strerror(errno));
		(void)fclose(f);
		return (NULL);
	}
	n = sbuf.st_size;
	buf = new(1, n + 1, "corsa auth token");
	if (fgets(buf, n + 1, f) == NULL) {
		lg(LOG_ERR, "corsareadtoken: fgets %s: failed", fn);
		(void)fclose(f);
		return (NULL);
	}
	(void)fclose(f);
	cp = buf + n - 1;
	if (cp >= buf && *cp == '\n')
	    *cp = '\0';

	return (buf);
}

/* Return 0 if not ready to process yet */
static int
corsavalidate(struct child *chp, struct client *cl, struct req *rp)
{
	struct addr *wp;
	char buf[64];

	/* We can only handle IPv4 */
	if (rp->filter.family != AF_INET) {
		if (rp->type == REQ_QUERYFILTER)
			child2clientsenderr(chp, cl, rp, "%s",
			    "not found");
		else
			child2clientsenderr(chp, cl, rp, "%s",
			    "Only IPv4 addresses are supported");
		return (1);
	}

	/* Check for required corsalocalnets cidr list */
	if (cf->c_corsalocalnets == NULL) {
		child2clientsenderr(chp, cl, rp, "%s",
		    "corsalocalnets missing from configuration");
		return (1);
	}

	/* Check that the address is not a local filter network */
	if (!goodfilter(&rp->filter)) {
		child2clientsenderr(chp, cl, rp,
		    "%s part of a configured local filternet",
		    addr2str(&rp->filter));
		return (1);
	}

	switch (rp->type) {

	case REQ_FILTER:
		wp = whitelist(&rp->filter);
		if (wp != NULL) {
			snprintf(buf, sizeof(buf), "%s", addr2str(&rp->filter));
			child2clientsenderr(chp, cl, rp,
			    "%s is on the whitelist (%s)", buf, addr2str(wp));

			/* Log for NETS */
			if (cl != NULL && rp->child2notdone == 0)
				nets_log(cl, rp);
			return (1);
		}
		stats_setrate(&state.filterstats);
		break;

	default:
		break;
	}

	/* We've validated this request but the child isn't ready */
	if (!CHILD2LOGGEDIN(chp->state))
		return (0);

	return (1);
}

/* Return 1 if the address is an allowable (non-local) filter candidate */
static int
goodfilter(struct addr *addrp)
{
	int i;
	struct addr *a;
	struct array *dp;

	dp = &cf->c_corsalocalnets_array;
	for (i = 0, a = cf->c_corsalocalnets; i < dp->len; ++i, ++a)
		if (insubnet(a, addrp))
			return (0);

	return (1);
}
