/*
 * Copyright (c) 2018, 2020, 2022, 2024
 * The Regents of the University of California. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *     * Redistributions of source code must retain the above copyright
 *       notice, this list of conditions and the following disclaimer.
 *     * Redistributions in binary form must reproduce the above copyright
 *       notice, this list of conditions and the following disclaimer in the
 *       documentation and/or other materials provided with the distribution.
 *     * Neither the name of the University nor the names of its contributors
 *       may be used to endorse or promote products derived from this software
 *       without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#ifndef lint
static const char rcsid[] __attribute__ ((unused)) =
    "@(#) $Id$ (LBL)";
#endif

#include <sys/types.h>
#include <sys/socket.h>
#include <sys/sysctl.h>

#include <errno.h>
#include <fcntl.h>
#include <limits.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sysexits.h>
#include <unistd.h>

#include "acld.h"
#include "freebsd.h"

/* Forwards */
static void kread(kvm_t *, u_long, void *, size_t, const char *);
static struct sockaddr *my_rt_key(struct rtentry *);
static struct sockaddr *my_rt_mask(struct rtentry *);
static void reportfamily(struct req *, int, int);
static void route_print(struct req *, int, int,
    struct rtentry *);
static void route_tree(struct req *, int, int,
    struct radix_node *);

#define KREAD(a, d) kread(state.kvmd, (u_long)(a), &(d), sizeof(d), "#d")

/* Locals */
static int fibnum, numfibs;
static struct radix_node_head **rt_tables;

static struct nlist nl[] = {
#define N_RTREE 0
	{ .n_name = "_rt_tables"},
	{ .n_name = NULL },
};

void
freebsdinit2(void)
{
	size_t len, size, intsize;
	char errbuf[_POSIX2_LINE_MAX];
	struct state *sp;

	/* Open the kernel handle */
	sp = &state;
	sp->kvmd = kvm_openfiles(NULL, NULL, NULL, O_RDONLY, errbuf);
	if (sp->kvmd == NULL) {
		lg(LOG_ERR, "freebsdinit: kvm_openfiles: %s", errbuf);
		exit(1);
	}

	if (kvm_nlist(sp->kvmd, nl) < 0) {
		lg(LOG_ERR, "freebsdinit: kvm_nlist: %s", kvm_geterr(sp->kvmd));
		exit(1);
	}
	if (nl[N_RTREE].n_type == 0) {
		lg(LOG_ERR, "freebsdinit: no namelist");
		exit(1);
	}

	intsize = sizeof(int);
	if (sysctlbyname("net.my_fibnum", &fibnum, &intsize, NULL, 0) < 0) {
		lg(LOG_ERR, "freebsdinit: sysctlbyname: %s (1)",
		    strerror(errno));
		exit(EX_SOFTWARE);
	}
	if (sysctlbyname("net.fibs", &numfibs, &intsize, NULL, 0) < 0) {
		lg(LOG_ERR, "freebsdinit: sysctlbyname: %s (2)",
		    strerror(errno));
		exit(EX_SOFTWARE);
	}

	len = numfibs * (AF_MAX + 1);
	rt_tables = calloc(len, sizeof(**rt_tables));
	if (rt_tables == NULL) {
		lg(LOG_ERR, "freebsdinit: calloc: %s", strerror(errno));
		exit(EX_OSERR);
	}

	/* Get the routing table addresses */
	size = len * sizeof(**rt_tables);
	kread(sp->kvmd, nl[N_RTREE].n_value, rt_tables, size, "rt_tables");
}

/* XXX maybe we should return a value so we can avoid racing the rt table */
static void
kread(kvm_t *kvmd, u_long addr, void *buf, size_t size, const char *what)
{
	ssize_t cc;

	cc = kvm_read(kvmd, addr, buf, size);
	if (cc < 0) {
		lg(LOG_ERR, "kread: %s: %s", what, kvm_geterr(kvmd));
		exit(EX_OSERR);
	}
	if (cc != size) {
		lg(LOG_ERR, "kread: %s: short read (%d != %d)",
		    what, (int)cc, (int)size);
		exit(EX_SOFTWARE);
	}
}

/* XXX The FreeBSD macro produces strict-aliasing warnings */
struct sockaddr *
my_rt_key(struct rtentry *r)
{
	caddr_t *addr;
	struct sockaddr **sapp;

	addr = &r->rt_nodes->rn_key;
	sapp = (struct sockaddr **)addr;
	return (*sapp);
}

/* XXX The FreeBSD macro produces strict-aliasing warnings */
struct sockaddr *
my_rt_mask(struct rtentry *r)
{
	caddr_t *addr;
	struct sockaddr **sapp;

	addr = &r->rt_nodes->rn_mask;
	sapp = (struct sockaddr **)addr;
	return (*sapp);
}

void
report(struct req *rp, int nullzeroonly)
{
	reportfamily(rp, AF_INET, nullzeroonly);
	reportfamily(rp, AF_INET6, nullzeroonly);
}

static void
reportfamily(struct req *rp, int fam, int nullzeroonly)
{
	struct radix_node_head **rnhp, *rnh;
	struct radix_node_head head;
	struct state *sp;

	sp = &state;
	rnhp = (struct radix_node_head **)*rt_tables;
	rnhp += (fibnum * (AF_MAX + 1)) + fam;
	kread(sp->kvmd, (unsigned long)rnhp, &rnh, sizeof(rnh), "head addr");
	kread(sp->kvmd, (unsigned long)rnh, &head, sizeof(head), "head");
	route_tree(rp, fam, nullzeroonly, head.rnh_treetop);
}

static void
route_print(struct req *rp, int fam, int nullzeroonly, struct rtentry *rt)
{
	u_int16_t port;
	size_t size, len;
	char *cp;
	struct addr *a;
	struct sockaddr *sap, *smp;
	struct sockaddr_dl *sdl;
	struct sockaddr_storage sa, sm;
	struct state *sp;
	struct aroute *rtp;
	struct aroute route;
	char buf[256];

	sp = &state;
	rtp = &route;
	memset(rtp, 0, sizeof(*rtp));
	if ((rt->rt_flags & (RTF_DYNAMIC | RTF_PROTO1 | RTF_PROTO2)) != 0)
		rtp->type = ROUTE_DYNAMIC;
	else if ((rt->rt_flags & (RTF_STATIC | RTF_MULTICAST)) != 0)
		rtp->type = ROUTE_STATIC;

	/* Destination */
	sap = (struct sockaddr *)&sa;
	kread(sp->kvmd, (u_long)my_rt_key(rt), sap, sizeof(*sap), "dst");

	/* Subnet mask */
	smp = NULL;
	if ((rt->rt_flags & RTF_HOST) == 0 && (smp = my_rt_mask(rt)) != NULL) {
		kread(sp->kvmd, (u_long)smp, &sm, sizeof(sm), "mask");
		smp = (struct sockaddr *)&sm;
	}
	a = &rtp->dst;
	sa2addr(sap, smp, a, &port);

	/* Gateway */
	memset(&sa, 0, sizeof(sa));
	kread(sp->kvmd, (u_long)rt->rt_gateway, &sa, sizeof(sa), "gateway");
	a = &rtp->gw;
	sap = (struct sockaddr *)&sa;
	if (sap->sa_family == fam) {
		sa2addr((struct sockaddr *)&sa, NULL, a, &port);

		/* Is it a null zero route */
		if ((cf->c_nullzeroaddr.family == fam &&
		    cf->c_nullzeroaddr.addr4 == a->addr4) ||
		    (cf->c_nullzeroaddr6.family == fam &&
		    memcmp(cf->c_nullzeroaddr6.addr6, a->addr6,
		    sizeof(a->addr6)) == 0))
			rtp->type = ROUTE_NULLZERO;
	} else if (sap->sa_family == AF_LINK) {
		rtp->type = ROUTE_INTERFACE;
		if (nullzeroif_ind > 0) {
			/* Check for nullzero interface index or name */
			sdl = (struct sockaddr_dl *)&sa;
			if (sdl->sdl_nlen == 0 &&
			    sdl->sdl_alen == 0 &&
			    sdl->sdl_slen == 0 &&
			    sdl->sdl_index == nullzeroif_ind) {
				rtp->type = ROUTE_NULLZERO;
			} else {
				cp = link_ntoa(sdl);
				if (cp != NULL && *cp != '\0' &&
				    strcmp(cp, cf->c_nullzeroif) == 0)
					rtp->type = ROUTE_NULLZERO;
			}
		}
	}

	/* Provide some clues when we can't determine the route type */
	if (rtp->type == ROUTE_UNKNOWN) {
		cp = buf;
		size = sizeof(buf);
		(void)snprintf(cp, size, "%s", addr2str(&rtp->dst));
		if (rtp->gw.family != 0) {
			len = strlen(cp);
			size -= len;
			cp += len;
			(void)snprintf(cp, size, " %s", addr2str(&rtp->gw));
		}
		rtp->raw = strsave(buf);
	}

	if (rtp->type == ROUTE_NULLZERO)
		++sp->nullzerolen;

	/* Format if we're servicing a request */
	if (rp != NULL) {
		if (nullzeroonly) {
			if (rtp->type == ROUTE_NULLZERO)
				ioappendfmt(&rp->payload, "%s",
				    addr2str(&rtp->dst));
		} else
			ioappendfmt(&rp->payload, "%s", routeformat(rtp));
	}
}

static void
route_tree(struct req *rp, int fam, int nullzeroonly, struct radix_node *rn)
{
	struct radix_node rnode;
	struct rtentry rtentry;

	do {
		KREAD(rn, rnode);
		if ((rnode.rn_flags & RNF_ACTIVE) == 0)
			break;

		if (rnode.rn_bit < 0) {
			if ((rnode.rn_flags & RNF_ROOT) == 0) {
				KREAD(rn, rtentry);
				route_print(rp, fam, nullzeroonly,
				    &rtentry);
			}
			rn = rnode.rn_dupedkey;
		} else {
			rn = rnode.rn_right;
			route_tree(rp, fam, nullzeroonly, rnode.rn_left);
			route_tree(rp, fam, nullzeroonly, rn);
			rn = NULL;
		}
	} while (rn != NULL);
}
