/*
 * Copyright (c) 2018, 2019, 2020, 2022, 2024
 * The Regents of the University of California. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *     * Redistributions of source code must retain the above copyright
 *       notice, this list of conditions and the following disclaimer.
 *     * Redistributions in binary form must reproduce the above copyright
 *       notice, this list of conditions and the following disclaimer in the
 *       documentation and/or other materials provided with the distribution.
 *     * Neither the name of the University nor the names of its contributors
 *       may be used to endorse or promote products derived from this software
 *       without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#ifndef lint
static const char rcsid[] __attribute__ ((unused)) =
    "@(#) $Id$ (LBL)";
#endif

#include <sys/types.h>
#include <sys/param.h>
#include <sys/sysctl.h>

#include <errno.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sysexits.h>
#include <unistd.h>

#include "acld.h"
#include "freebsd.h"

/* Forwards */
static void route_print(struct req *, int, int,
    struct rt_msghdr *);

static int mib[7] = {
	CTL_NET,					/* first level */
	PF_ROUTE,					/* second level */
	0,						/* protocol number */
	0,						/* family (all) */
	NET_RT_DUMP,					/* fifth level */
	0,						/* six level (none) */
	0,						/* fibnum */
};
#define MIBLEN (sizeof(mib) / sizeof(mib[0]))

void
freebsdinit2(void)
{
	size_t intsize;
	int fibnum;

	intsize = sizeof(int);
	if (sysctlbyname("net.my_fibnum", &fibnum, &intsize, NULL, 0) < 0) {
		lg(LOG_ERR, "freebsdinit2: sysctlbyname: %s (1)",
		    strerror(errno));
		exit(EX_SOFTWARE);
	}
	mib[MIBLEN - 1] = fibnum;
}

void
report(struct req *rp, int nullzeroonly)
{
	size_t needed;
	char *buf, *next, *lim;
	struct rt_msghdr *rtm;
	struct sockaddr *sa;

	if (sysctl(mib, nitems(mib), NULL, &needed, NULL, 0) < 0) {
		lg(LOG_ERR, "sysctl CTL_NET PF_ROUTE 1: %s", strerror(errno));
		return;
	}
	buf = new(1, needed, "report buf");
	if (sysctl(mib, nitems(mib), buf, &needed, NULL, 0) < 0) {
		lg(LOG_ERR, "sysctl CTL_NET PF_ROUTE 2: %s", strerror(errno));
		free(buf);
		return;
	}
	lim = buf + needed;
	for (next = buf; next < lim; next += rtm->rtm_msglen) {
		rtm = (struct rt_msghdr *)next;
		if (rtm->rtm_version != RTM_VERSION)
			continue;
		sa = (struct sockaddr *)(rtm + 1);
		route_print(rp, sa->sa_family, nullzeroonly, rtm);
	}

	free(buf);
}

static void
route_print(struct req *rp, int fam, int nullzeroonly, struct rt_msghdr *rtm)
{
	u_int16_t port;
	int i;
	size_t size, len;
	char *cp;
	struct sockaddr *sap, *smp, *addr[RTAX_MAX];
	struct addr *a;
	struct sockaddr_dl *sdl;
	char buf[256];

	struct aroute *rtp;
	struct aroute route;

	rtp = &route;
	memset(rtp, 0, sizeof(*rtp));
	if ((rtm->rtm_flags & (RTF_DYNAMIC | RTF_PROTO1 | RTF_PROTO2)) != 0)
		rtp->type = ROUTE_DYNAMIC;
	else if ((rtm->rtm_flags & (RTF_STATIC | RTF_MULTICAST)) != 0)
		rtp->type = ROUTE_STATIC;

	/* Build array of sockaddrs pointers */
	sap = (struct sockaddr *)(rtm + 1);
	for (i = 0; i < RTAX_MAX; i++) {
		if (rtm->rtm_addrs & (1 << i))
			addr[i] = sap;
		sap = (struct sockaddr *)((char *)sap + SA_SIZE(sap));
	}

	/* Destination */
	a = &rtp->dst;
	sap = addr[RTAX_DST];

	/* Subnet mask */
	if ((rtm->rtm_flags & RTF_HOST) != 0)
		smp = NULL;
	else
		smp = addr[RTAX_NETMASK];
	sa2addr(sap, smp, a, &port);

	/* Gateway */
	a = &rtp->gw;
	sap = addr[RTAX_GATEWAY];
	if (sap->sa_family == AF_INET || sap->sa_family == AF_INET6) {
		sa2addr(sap, NULL, a, &port);

		/* Is it a null zero route */
		if ((cf->c_nullzeroaddr.family == sap->sa_family &&
		    cf->c_nullzeroaddr.addr4 == a->addr4) ||
		    (cf->c_nullzeroaddr6.family == sap->sa_family &&
		    memcmp(cf->c_nullzeroaddr6.addr6, a->addr6,
		    sizeof(a->addr6)) == 0))
			rtp->type = ROUTE_NULLZERO;
	} else if (sap->sa_family == AF_LINK) {
		rtp->type = ROUTE_INTERFACE;
		if (nullzeroif_ind > 0) {
			/* Check for nullzero interface index or name */
			// XXX probably wrong
			sdl = (struct sockaddr_dl *)sap;
			if (sdl->sdl_nlen == 0 &&
			    sdl->sdl_alen == 0 &&
			    sdl->sdl_slen == 0 &&
			    sdl->sdl_index == nullzeroif_ind) {
				rtp->type = ROUTE_NULLZERO;
			} else {
				cp = link_ntoa(sdl);
				if (cp != NULL && *cp != '\0' &&
				    strcmp(cp, cf->c_nullzeroif) == 0)
					rtp->type = ROUTE_NULLZERO;
			}
		}
	}

	/* Provide some clues when we can't determine the route type */
	if (rtp->type == ROUTE_UNKNOWN) {
		cp = buf;
		size = sizeof(buf);
		(void)snprintf(cp, size, "%s", addr2str(&rtp->dst));
		if (rtp->gw.family != 0) {
			len = strlen(cp);
			size -= len;
			cp += len;
			(void)snprintf(cp, size, " %s", addr2str(&rtp->gw));
		}
		rtp->raw = strsave(buf);
	}

	if (rtp->type == ROUTE_NULLZERO)
		++state.nullzerolen;

	/* Format if we're servicing a request */
	if (rp != NULL) {
		if (nullzeroonly) {
			if (rtp->type == ROUTE_NULLZERO)
				ioappendfmt(&rp->payload, "%s",
				    addr2str(&rtp->dst));
		} else {
			ioappendfmt(&rp->payload, "%s", routeformat(rtp));
		}
	}

}
