/*
 * Copyright (c) 2013, 2014, 2015, 2016, 2017, 2018, 2019, 2020, 2022, 2024
 * The Regents of the University of California. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *     * Redistributions of source code must retain the above copyright
 *       notice, this list of conditions and the following disclaimer.
 *     * Redistributions in binary form must reproduce the above copyright
 *       notice, this list of conditions and the following disclaimer in the
 *       documentation and/or other materials provided with the distribution.
 *     * Neither the name of the University nor the names of its contributors
 *       may be used to endorse or promote products derived from this software
 *       without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#ifndef lint
static const char rcsid[] __attribute__ ((unused)) =
    "@(#) $Id$ (LBL)";
#endif

#include <sys/types.h>

#include <errno.h>
#include <fcntl.h>
#include <ifaddrs.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sysexits.h>
#include <unistd.h>

#include "acld.h"
#include "freebsd.h"
#include "history.h"

/* Forwards */
static void freebsdchildlistroute(void);
static void freebsdroutes(struct client *, struct req *);
static void route_cmd(struct client *, struct req *);

/* Globals */
struct sockaddr_dl sanullzeroif;
struct sockaddr_in sanullzero4;
struct sockaddr_in6 sanullzero6;
u_int nullzeroif_ind;

/* Locals */
static pid_t pid;
static int seq;

static void
freebsdchildlistroute(void)
{
	/* Count the number of nullzero routes */
	freebsdlistroute(NULL, NULL, 1);
	++state.listedroutes;
}

void
freebsdinit(void)
{
	struct ifaddrs *ifap, *ifa;
	struct sockaddr_dl *sdl;
	struct state *sp;

	sp = &state;
	if (geteuid() != 0) {
		lg(LOG_ERR,
		    "freebsdinit: Must run as root to add/delete routes");
		exit(1);
	}

	/* Used by route_cmd() */
	pid = getpid();

	/* kvm/sysctl initialization */
	freebsdinit2();

	/* nullzero interface sockaddr */
	if (cf->c_nullzeroif != NULL) {
		if (getifaddrs(&ifap) < 0) {
			lg(LOG_ERR,
			    "freebsdinit: getifaddrs: %s", strerror(errno));
			exit(EX_OSERR);
		}
		sdl = NULL;
		for (ifa = ifap; ifa != NULL; ifa = ifa->ifa_next) {
			if (ifa->ifa_addr->sa_family != AF_LINK)
				continue;
			if (strcmp(ifa->ifa_name, cf->c_nullzeroif) != 0)
				continue;
			sdl = (struct sockaddr_dl *)ifa->ifa_addr;
			break;
		}
		if (sdl == NULL) {
			lg(LOG_ERR,
			    "freebsdinit: Can't find nullzero interface \"%s\"",
			    cf->c_nullzeroif);
			exit(EX_OSERR);
		}
		memmove(&sanullzeroif, sdl, sdl->sdl_len);
		freeifaddrs(ifap);

		/* Get the nullzero interface index */
		nullzeroif_ind = if_nametoindex(cf->c_nullzeroif);
	}

	/* nullzero IPv4 sockaddr */
	if (cf->c_nullzeroaddr.family != 0)
		(void)addr2sa(&cf->c_nullzeroaddr, 0,
		    (struct sockaddr *)&sanullzero4);

	/* nullzero IPv6 sockaddr */
	if (cf->c_nullzeroaddr6.family != 0)
		(void)addr2sa(&cf->c_nullzeroaddr6, 0,
		    (struct sockaddr *)&sanullzero6);

	/* Open the routing socket */
	sp->routefd = socket(PF_ROUTE, SOCK_RAW, 0);
	if (sp->routefd < 0) {
		lg(LOG_ERR, "freebsdinit: socket: %s", strerror(errno));
		exit(EX_OSERR);
	}

	/* Close on exec */
	if (fcntl(sp->routefd, F_SETFD, FD_CLOEXEC) < 0) {
		lg(LOG_ERR, "freebsdinit: fcntl FD_CLOEXEC: %s",
		    strerror(errno));
		exit(EX_OSERR);
	}

	/* Child hooks */
	sp->f_childlistroute = freebsdchildlistroute;

	/* Client hooks */
	sp->f_clientlistroute = freebsdlistroute;
	sp->f_clientnullzero = freebsdroutes;
	sp->f_clientquerynullzero = freebsdroutes;
}

void
freebsdlistroute(struct client *cl, struct req *rp,
    int nullzeroonly)
{
	struct state *sp;

	/* (re)count the number of nullzero routes as a side effect */
	sp = &state;
	sp->nullzerolen = 0;
	report(rp, nullzeroonly);
	stats_sethiwater(&sp->nullzerostats, sp->nullzerolen);
	if (cl != NULL) {
		rp->flags |= RFLAG_CONTINUE;
		clientsend(cl, rp);
	}
}

static void
freebsdroutes(struct client *cl, struct req *rp)
{
	struct addr *wp;
	struct state *sp;
	char buf[64];

	sp = &state;
	if (rp->nullzero.type != ROUTE_NULLZERO) {
		lg(LOG_ERR, "freebsdroutes: bad routetype %d",
		    rp->nullzero.type);
		exit(EX_SOFTWARE);
	}

	/*
	 * Everything is ok (subject to whitelist check) if no
	 * configured nullzero nets
	 */
	if (cf->c_nullzeronets != NULL &&
	    !goodnullzero(&rp->nullzero.dst)) {
		clientsenderr(cl, rp, "%s not part of a configured nullzeronet",
		    addr2str(&rp->nullzero.dst));
		return;
	}

	switch (rp->type) {

	case REQ_NULLZERO:
		wp = whitelist(&rp->nullzero.dst);
		if (wp != NULL) {
			snprintf(buf, sizeof(buf), "%s",
			    addr2str(&rp->nullzero.dst));
			clientsenderr(cl, rp, "%s is on the whitelist (%s)",
			    buf, addr2str(wp));

			/* Log for NETS */
			nets_log(cl, rp);
			return;
		}

		if (cf->c_nullzeromax != 0 &&
		    sp->nullzerolen >= cf->c_nullzeromax) {
			rp->flags |= RFLAG_IGNORE;
			clientsenderr(cl, rp,
			    "Too many nullzero routes (%d in use, %d allowed)",
			    sp->nullzerolen, cf->c_nullzeromax);

			/* Log for NETS */
			nets_log(cl, rp);
			return;
		}
		break;

	case REQ_NONULLZERO:
	case REQ_QUERYNULLZERO:
		break;

	default:
		lg(LOG_ERR, "freebsdroutes: bad type %d", rp->type);
		exit(EX_SOFTWARE);
	}

	route_cmd(cl, rp);

	/* Log for NETS */
	nets_log(cl, rp);

	/*
	 * Save successful nullzero completion for the replicant
	 * Wait until cts is populated
         */
	if (cf->c_replicate_nullzero && rp->type == REQ_NULLZERO &&
	    (rp->flags & (RFLAG_FAILED|RFLAG_IGNORE)) == 0)
		history_add(&rp->nullzero.dst, rp->type, &rp->cts);
}

static void
route_cmd(struct client *cl, struct req *rp)
{
	int n, nfds;
	size_t len, len2, cc, size2;
	char *cp, *errmsg, *msg;
	u_char *bp;
	fd_set readfds;
	struct timeval timeout;
	struct addr *a;
	struct sockaddr *sap;
	struct rt_msghdr *rh;
	struct timeval *tvp;
	struct state *sp;
	struct {
		struct rt_msghdr m_rtm;
		u_char m_space[512];
	} m_rtmsg;
	char buf[128];

	sp = &state;
	memset(&m_rtmsg, 0, sizeof(m_rtmsg));
	rh = &m_rtmsg.m_rtm;
	bp = (u_char *)&m_rtmsg.m_space;
	switch (rp->type) {

	case REQ_NONULLZERO:
		rh->rtm_type = RTM_DELETE;
		break;

	case REQ_NULLZERO:
		rh->rtm_type = RTM_ADD;
		break;

	case REQ_QUERYNULLZERO:
		rh->rtm_type = RTM_GET;
		break;

	default:
		lg(LOG_ERR, "route_cmd: unknown type %s",
		    val2str(cmd2req, rp->type));
		abort();
	}
	rh->rtm_flags = RTF_UP | RTF_STATIC;
	rh->rtm_version = RTM_VERSION;
	rh->rtm_seq = ++seq;
	rh->rtm_addrs = RTA_DST | RTA_GATEWAY;

	/* RTA_DST */
	a = &rp->nullzero.dst;
	(void)addr2sa(a, 0, (struct sockaddr *)bp);

	/* Round up the amount of data passed */
	bp += SA_SIZE(bp);

	/* RTA_GATEWAY */
	if (cf->c_nullzeroif != NULL)
		sap = (struct sockaddr *)&sanullzeroif;
	else if (a->family == sanullzero4.sin_family) {
		rh->rtm_flags |= RTF_GATEWAY;
		sap = (struct sockaddr *)&sanullzero4;
	} else if (a->family == sanullzero6.sin6_family) {
		rh->rtm_flags |= RTF_GATEWAY;
		sap = (struct sockaddr *)&sanullzero6;
	} else {
		lg(LOG_ERR,
		    "route_cmd: didn't find nullzero interface or addr");
		abort();
	}
	memmove(bp, sap, sap->sa_len);

	/* Round up the amount of data passed */
	bp += SA_SIZE(sap);

	if (ISHOST(a)) {
		rh->rtm_flags |= RTF_HOST;
	} else {
		/* RTA_NETMASK */
		rh->rtm_addrs |= RTA_NETMASK;
		(void)mask2sa(a, 0, (struct sockaddr *)bp);

		/* Round up the amount of data passed */
		bp += SA_SIZE(bp);
	}

	len = bp - (u_char *)&m_rtmsg;
	rh->rtm_msglen = len;
	errno = 0;

	msg = NULL;
	errmsg = NULL;
	cc = write(sp->routefd, (char *)rh, len);
	if ((int)cc < 0) {
		switch (errno) {

		case ESRCH:
			(void)snprintf(buf, sizeof(buf), "%s not found",
			    addr2str(a));
			errmsg = buf;
			break;

		case EEXIST:
			rp->flags |= RFLAG_IGNORE;
			(void)snprintf(buf, sizeof(buf),
			    "Note: %s is already nullzero routed", addr2str(a));

			/* Append timestamp if client is a replicant */
			if (cl->replicant &&
			    (tvp = history_find(a, REQ_NULLZERO)) != NULL) {
				len2 = strlen(buf);
				cp = buf + len2;
				size2 = sizeof(buf) - len2;
				(void)snprintf(cp, size2, " @ %ld.%06ld",
				    tvp->tv_sec, tvp->tv_usec);
			}

			msg = buf;
			break;

		default:
			(void)snprintf(buf, sizeof(buf), "%s failed: %s",
			    addr2str(a), strerror(errno));
			errmsg = buf;
			break;
		}
	} else if (cc != len) {
		lg(LOG_ERR, "route_cmd: bad write: %d != %d",
		    (int)cc, (int)len);
		exit(EX_OSERR);
	}

	/* Always try to read result */
	nfds = sp->routefd + 1;
	do {
		FD_ZERO(&readfds);
		FD_SET(sp->routefd, &readfds);
		timeout.tv_sec = 5;
		timeout.tv_usec = 0;
		n = select(nfds, &readfds, NULL, NULL, &timeout);
		if (n < 0) {
			/* Don't choke if we get ptraced */
			if (errno == EINTR)
				continue;
			cc = -1;
			break;
		} else if (n == 0) {
			/* Timeout */
			cc = -1;
			errno = ETIMEDOUT;
			break;
		}
		memset(&m_rtmsg, 0, sizeof(m_rtmsg));
		cc = read(sp->routefd, (char *)&m_rtmsg, sizeof(m_rtmsg));
	} while ((int)cc > 0 && (rh->rtm_seq != seq || rh->rtm_pid != pid));

	if ((int)cc < 0) {
		(void)snprintf(buf, sizeof(buf), "%s", strerror(errno));
		errmsg = buf;
	}

	switch (rp->type) {

	case REQ_NULLZERO:
		if (msg == NULL && errmsg == NULL) {
			++sp->nullzerolen;
			stats_sethiwater(&sp->nullzerostats, sp->nullzerolen);
			stats_setrate(&sp->nullzerostats);
		}
		break;

	case REQ_NONULLZERO:
		if (msg == NULL && errmsg == NULL) {
			--sp->nullzerolen;
			stats_setvalue(&sp->nullzerostats, sp->nullzerolen);
		}
		break;

	case REQ_QUERYNULLZERO:
		if (errmsg == NULL || rh->rtm_errno != ESRCH) {
			if (!route_sfmt(buf, sizeof(buf), a, rh, cc))
				errmsg = buf;
			else
				msg = buf;
		}
		break;

	default:
		lg(LOG_ERR, "route_cmd: unknown type %s",
		    val2str(cmd2req, rp->type));
		abort();
	}

	if (errmsg != NULL)
		clientsenderr(cl, rp, "%s", errmsg);
	else if (msg != NULL)
		clientsendfmt(cl, rp, "%s", msg);
	else
		clientsend(cl, rp);
}

int
route_sfmt(char *buf, size_t buflen, struct addr *a,
    struct rt_msghdr *rh, size_t len)
{
	size_t n;
	int rta;
	char *bp;
	struct addr *a2;
	const char *ifname;
	struct addr addr, addr2;
	struct sockaddr *sa;
	struct sockaddr *sa_dst;
	struct sockaddr *sa_gate;
	struct sockaddr *sa_mask;
	u_int16_t port;

	if (rh->rtm_version != RTM_VERSION) {
		strlcpy(buf, "bad RTM_VERSION", buflen);
		return (0);
	}
	if (rh->rtm_msglen != len) {
		snprintf(buf, buflen, "bad rtm_msglen (%d != %d)",
		    (int)rh->rtm_msglen, (int)len);
		return (0);
	}
	if (rh->rtm_errno != 0) {
		strlcpy(buf, strerror(rh->rtm_errno), buflen);
		return (0);
	}
	if (rh->rtm_addrs == 0) {
		strlcpy(buf, "missing addrs", buflen);
		return (0);
	}

	sa_dst = NULL;
	sa_gate = NULL;
	sa_mask = NULL;
	bp = (char *)(rh + 1);
	for (rta = 1; rta != 0; rta <<= 1) {
		if ((rta & rh->rtm_addrs) == 0)
			continue;
		sa = (struct sockaddr *)bp;
		switch (rta) {

		case RTA_DST:
			sa_dst = sa;
			break;

		case RTA_GATEWAY:
			sa_gate = sa;
			break;

		case RTA_NETMASK:
			sa_mask = sa;
			break;

		default:
			break;
		}
		bp += SA_SIZE(sa);
	}

	if (sa_dst == NULL) {
		strlcpy(buf, "missing destination", buflen);
		return (0);
	}

	if (sa_gate == NULL) {
		strlcpy(buf, "missing gateway", buflen);
		return (0);
	}

	/* Destination */
	sa2addr(sa_dst, sa_mask, &addr, &port);

	/*
	 * AF_LINK gateway is either an interface or a nullzero
	 * route (This only makes sense if we're using an interface
	 * for nullzero routes)
	 */
	if (cf->c_nullzeroif != NULL && sa_gate->sa_family == AF_LINK) {
		ifname = link_ntoa((struct sockaddr_dl *)sa_gate);
		if (strcmp(ifname, cf->c_nullzeroif) == 0) {
			/* Report actual nullzero route gateway */
			snprintf(buf, buflen, "N %s", addr2str(&addr));
			return (1);
		}
	}

	/* Check for nullzero IPv4 or IPv6 address */
	if (cf->c_nullzeroaddr.family == sa_gate->sa_family ||
	    cf->c_nullzeroaddr6.family == sa_gate->sa_family) {
		a2 = &addr2;
		sa2addr(sa_gate, NULL, a2, &port);
		if ((cf->c_nullzeroaddr.family == sa_gate->sa_family &&
		    cf->c_nullzeroaddr.addr4 == a2->addr4) ||
		    (cf->c_nullzeroaddr6.family == sa_gate->sa_family &&
		    memcmp(cf->c_nullzeroaddr6.addr6, a2->addr6,
		    sizeof(a2->addr6)) == 0)) {
			snprintf(buf, buflen, "N %s", addr2str(&addr));
			n = strlen(buf);
			buf += n;
			buflen -= n;
			snprintf(buf, buflen, " %s", addr2str(a2));
			return (1);
		}
	}

	/* Report query address failure */
	snprintf(buf, buflen, "%s not found", addr2str(a));
	return (0);
}
