/*
 * Copyright (c) 2004, 2006, 2008, 2009, 2010, 2013, 2014, 2015, 2016, 2018, 2019, 2020, 2021, 2022, 2024
 * The Regents of the University of California. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *     * Redistributions of source code must retain the above copyright
 *       notice, this list of conditions and the following disclaimer.
 *     * Redistributions in binary form must reproduce the above copyright
 *       notice, this list of conditions and the following disclaimer in the
 *       documentation and/or other materials provided with the distribution.
 *     * Neither the name of the University nor the names of its contributors
 *       may be used to endorse or promote products derived from this software
 *       without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#ifndef lint
static const char rcsid[] __attribute__ ((unused)) =
    "@(#) $Id$ (LBL)";
#endif

#include <sys/types.h>
#include <sys/time.h>

#include <ctype.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include "acld.h"

/* Locals */
static const char crnl[] = "\r\n";

/* Appends string to iobuf; assures EOS termination */
void
ioappendbuf(struct iobuf *ip, const char *buf)
{
	size_t len;
	struct array *dp;

	/* Make sure we have room for current, new, and EOS */
	len = strlen(buf);
	dp = &ip->io_array;
	DYNARRAY(dp, ip->buf, IOBUF_LEN(ip) + len + 1, "ioappendline");
	strcpy(ip->buf + IOBUF_LEN(ip), buf);
	IOBUF_LEN(ip) += len;
}

/* Appends formatted and cr/nl to iobuf; assures EOS termination */
void
ioappendfmt(struct iobuf *ip, const char *fmt, ...)
{
	char *cp;
	size_t cc, oldlen, newlen;
	struct array *dp;
	va_list ap;

	/* Initial estimate */
	oldlen = IOBUF_LEN(ip);
	newlen = oldlen + strlen(fmt) + sizeof(crnl);

	/* Loop until it fits */
	dp = &ip->io_array;
	for (;;) {
		DYNARRAY(dp, ip->buf, newlen, "ioappendfmt");
		va_start(ap, fmt);
		cp = ip->buf + oldlen;
		cc = vsnprintf(cp, IOBUF_SIZE(ip) - (oldlen + sizeof(crnl)),
		    fmt, ap);
		va_end(ap);
		if (cc == strlen(cp)) {
			IOBUF_LEN(ip) = oldlen + cc;
			break;
		}

		/* New length is old, new, cr/nl and EOS */
		newlen = oldlen + cc + sizeof(crnl);
		/* Always try and increase the size */
		if (newlen <= dp->size)
			newlen = dp->size + 1;
	}
	strcpy(ip->buf + IOBUF_LEN(ip), crnl);
	IOBUF_LEN(ip) += sizeof(crnl) - 1;
}

/* Appends string and cr/nl to iobuf; assures EOS termination */
void
ioappendline(struct iobuf *ip, const char *buf)
{
	ioappendbuf(ip, buf);
	ioappendbuf(ip, crnl);
}

/* Appends formatted and cr/nl to iobuf; assures EOS termination */
void
ioappendvfmt(struct iobuf *ip, const char *fmt, va_list ap)
{
	char *cp;
	size_t cc, oldlen, newlen;
	struct array *dp;
	va_list ap2;

	/* Initial estimate */
	oldlen = IOBUF_LEN(ip);
	newlen = oldlen + strlen(fmt) + sizeof(crnl);

	/* We must copy the passed ap to be able to use it more than once */
	va_copy(ap2, ap);

	/* Loop until it fits */
	dp = &ip->io_array;
	for (;;) {
		DYNARRAY(dp, ip->buf, newlen, "ioappendvfmt");
		cp = ip->buf + oldlen;
		cc = vsnprintf(cp, IOBUF_SIZE(ip) - (oldlen + sizeof(crnl)),
		    fmt, ap);
		IOBUF_LEN(ip) = oldlen + cc;
		if (cc == strlen(cp))
			break;

		/* New length is old, new, cr/nl, EOS, and some slop */
		newlen = oldlen + cc + sizeof(crnl) + 2;
		va_copy(ap, ap2);
	}
	strcpy(ip->buf + IOBUF_LEN(ip), crnl);
	IOBUF_LEN(ip) += sizeof(crnl) - 1;
}

/* Insert formatted and cr/nl to iobuf */
void
ioinsertline(struct iobuf *ip, const char *buf)
{
	size_t len;
	struct array *dp;

	/* Make sure we have room for current, new, cr/nl and EOS */
	len = strlen(buf);
	dp = &ip->io_array;
	DYNARRAY(dp, ip->buf, IOBUF_LEN(ip) + len + sizeof(crnl),
	    "ioappendline");

	/* Shift including EOS */
	memmove(ip->buf + len + sizeof(crnl) - 1, ip->buf, IOBUF_LEN(ip) + 1);

	/* Insert new string */
	memmove(ip->buf, buf, len);
	IOBUF_LEN(ip) += len;

	/* Insert crnl */
	memmove(ip->buf + len, crnl, sizeof(crnl) - 1);

	IOBUF_LEN(ip) += sizeof(crnl) - 1;
}

/* Returns 0 when there are no more comments to read */
int
iocomment(char **cpp, struct iobuf *ip)
{
	char *cp;

	/* Look for end of message (and strip leading dots) */
	while (iohaveline(ip) && (cp = iogetstr(ip)) != NULL) {
		if (*cp == '.') {
			++cp;
			if (strcmp("\r\n", cp) == 0 ||
			    strcmp("\n", cp) == 0 ||
			    *cp == '\0')
				return (0);
		}
		strappend(cpp, cp);
	}
	return (1);
}

/* Return 1 if able to consolidate payload messages to a single line */
int
ioconsolidate(struct iobuf *ip)
{
	int len1, len2;
	char *cp, *cp1, *cp2, *cp3;
	struct array *dp;

	cp1 = ip->buf;
	while (*cp1 != '\0') {
		/* First line */
		cp = strstr(cp1, crnl);
		if (cp == NULL)
			break;
		cp += 2;
		if (*cp == '\0')
			break;
		len1 = cp - cp1;
		cp2 = cp;
		cp3 = cp;

		while (*cp3 != '\0') {
			/* Following lines */
			cp = strstr(cp3, "\r\n");
			if (cp == NULL)
				break;
			cp += 2;
			len2 = cp - cp3;

			/* Remove second string of they are the same */
			if (len1 == len2 && strncmp(cp1, cp3, len1) == 0) {
				dp = &ip->io_array;
				memmove(cp3, cp, strlen(cp) + 1);
				dp->len -= len1;
				/* Don't udpate cp1 yet */
				cp2 = cp1;
			}
			cp3 = cp;
		}
		cp1 = cp2;
	}

	/* Return 0 if more than one line remains */
	cp = strstr(ip->buf, crnl);
	if (cp != NULL) {
		cp += sizeof(crnl) - 1;
		if (*cp != '\0')
			return (0);
	}
	return (1);
}

#ifdef HAVE_JUNOSCRIPT
/* Remove the first n characters from the iobuf */
void
ioeat(struct iobuf *ip, size_t size)
{
	if (IOBUF_LEN(ip) > 0) {
		if (IOBUF_LEN(ip) < size)
			IOBUF_LEN(ip) = 0;
		else {
			memmove(ip->buf, ip->buf + size,
			    IOBUF_LEN(ip) - size + 1);
			IOBUF_LEN(ip) -= size;
		}
		ip->buf[IOBUF_LEN(ip)] = '\0';
	}
}
#endif

void
iofree(struct iobuf *ip)
{
	struct array *dp;

	/* If size is zero, buf wasn't dynamically allocated */
	dp = &ip->io_array;
	FREEDYNARRAY(dp, ip->buf);
}

/* Return a zero terminated line (partial or complete including newline) */
char *
iogetstr(struct iobuf *ip)
{
	char *cp;
	size_t size;
	static char *buf = NULL;

	if (buf != NULL) {
		free(buf);
		buf = NULL;
	}
	if (IOBUF_LEN(ip) == 0)
		return (NULL);

	for (cp = ip->buf; *cp != '\n' && *cp != '\0'; ++cp)
		continue;

	size = cp - ip->buf + 1;
	buf = new(size + 1, sizeof(*buf), "iogetstr buf");
	memmove(buf, ip->buf, size);
	buf[size] = '\0';
	if (*cp == '\0')
		IOBUF_LEN(ip) = 0;
	else {
		memmove(ip->buf, ip->buf + size, IOBUF_LEN(ip) - size + 1);
		IOBUF_LEN(ip) -= size;
	}
	ip->buf[IOBUF_LEN(ip)] = '\0';
	return (buf);
}

/* Return true if there's a newline in the buffer */
int
iohaveline(struct iobuf *ip)
{
	int n;
	char *cp;

	for (n = IOBUF_LEN(ip), cp = ip->buf; n > 0; --n, ++cp)
		if (*cp == '\n')
			return (1);
	return (0);
}

#ifdef HAVE_JUNOSCRIPT
static char xml_rpc_reply[] = "<rpc-reply ";
static char xml_rpc_reply_end[] = "\n</rpc-reply>\n";

/* Return the length if there's a complete xml rpc-reply in the buffer else 0 */
int
iohaverpcreply(struct iobuf *ip)
{
	char *cp;

	if (IOBUF_LEN(ip) < sizeof(xml_rpc_reply) - 1)
		return (0);
	if (strncmp(ip->buf, xml_rpc_reply, sizeof(xml_rpc_reply) - 1) != 0)
		return (0);
	if ((cp = strstr(ip->buf, xml_rpc_reply_end)) != NULL)
		return ((cp - ip->buf) + (sizeof(xml_rpc_reply_end) - 1));
	return (0);
}
#endif

/*
 * Read from a socket and append to an iobuf
 *
 * Return -1 if there was a fatal error, 0 on EOF or else the iobuf len
 */
int
ioread(int fd, struct iobuf *ip)
{
	ssize_t cc;
	struct array *dp;

	dp = &ip->io_array;
	DYNARRAY(dp, ip->buf, dp->len + 1, "ioread");
	cc = read(fd, ip->buf + IOBUF_LEN(ip),
	    IOBUF_SIZE(ip) - IOBUF_LEN(ip) - 1);
	if (cc < 0)
		return (cc);

	IOBUF_LEN(ip) += cc;
	ip->buf[IOBUF_LEN(ip)] = '\0';
	return (IOBUF_LEN(ip));
}

/* Write from an iobuf to a socket */
int
iowrite(int fd, struct iobuf *ip)
{
	ssize_t cc;

	cc = write(fd, ip->buf, IOBUF_LEN(ip));
	if (cc < 0) {
		/* Bummer */
		IOBUF_LEN(ip) = 0;
		return (cc);
	}
	if (cc > IOBUF_LEN(ip)) {
		lg(LOG_ERR, "iowrite: cc > ip-len, %d > %d (Can't happen?)",
		    (int)cc, (int)IOBUF_LEN(ip));
		cc = IOBUF_LEN(ip);
	}
	if (cc == IOBUF_LEN(ip)) {
		IOBUF_LEN(ip) = 0;
		return (cc);
	}

	/* Move leftover chars on over */
	memmove(ip->buf, ip->buf + cc, IOBUF_LEN(ip) - cc);
	IOBUF_LEN(ip) -= cc;
	return (cc);
}
