/*
 * Copyright (c) 2016, 2017, 2018, 2020, 2021, 2022, 2024
 * The Regents of the University of California. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *     * Redistributions of source code must retain the above copyright
 *       notice, this list of conditions and the following disclaimer.
 *     * Redistributions in binary form must reproduce the above copyright
 *       notice, this list of conditions and the following disclaimer in the
 *       documentation and/or other materials provided with the distribution.
 *     * Neither the name of the University nor the names of its contributors
 *       may be used to endorse or promote products derived from this software
 *       without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#ifndef lint
static const char rcsid[] __attribute__ ((unused)) =
    "@(#) $Id$ (LBL)";
#endif

/* Generate Junoscript configuration RPCs */

#include <sys/types.h>

#include <libxml/encoding.h>
#include <libxml/xmlwriter.h>

#include <stdio.h>
#include <string.h>

#include "acld.h"
#include "jreq.h"
#include "junoscript.h"

#define ADDELEMENTWITHATTR(tag, key, val) do { \
    const char *_errmsg = addelementwithattr(tag, key, val); \
    if (_errmsg != NULL) \
	    return (_errmsg); \
    } while (0)

#define ADDNAMEDELEMENT(tag, val) do { \
    const char *_errmsg = addnamedelement(tag, val); \
    if (_errmsg != NULL) \
	    return (_errmsg); \
    } while (0)

#define ADDNAMEDELEMENTWITHATTR(tag, name, key, val) do { \
    const char *_errmsg = addnamedelementwithattr(tag, name, key, val); \
    if (_errmsg != NULL) \
	    return (_errmsg); \
    } while (0)

#define CLIMBUP(depth) do { \
    const char *_errmsg = climbup(depth); \
    if (_errmsg != NULL) \
	    return (_errmsg); \
    } while (0)

#define JREQ_ADDDEL_BLOCKMAC(jp) do { \
    const char *_errmsg = jreq_adddel_blockmac(jp); \
    if (_errmsg != NULL) \
	    return (_errmsg); \
    } while (0)

#define JREQ_ADD_FIREWALL(jp) do { \
    const char *_errmsg = jreq_add_firewall(jp); \
    if (_errmsg != NULL) \
	    return (_errmsg); \
    } while (0)

#define JREQ_ADD_PREFIX(jp) do { \
    const char *_errmsg = jreq_add_prefix(jp); \
    if (_errmsg != NULL) \
	    return (_errmsg); \
    } while (0)

#define JREQ_ADD_PREFIX_ADDR(jp) do { \
    const char *_errmsg = jreq_add_prefix_addr(jp); \
    if (_errmsg != NULL) \
	    return (_errmsg); \
    } while (0)

#define JREQ_START(depth) do { \
    const char *_errmsg = jreq_start(); \
    if (_errmsg != NULL) \
	    return (_errmsg); \
    } while (0)

#define STARTCHILD(tag) do { \
    const char *_errmsg = startchild(tag); \
    if (_errmsg != NULL) \
	    return (_errmsg); \
    } while (0)

#define STARTCHILDWITHATTR(tag, key, val) do { \
    const char *_errmsg = startchildwithattr(tag, key, val); \
    if (_errmsg != NULL) \
	    return (_errmsg); \
    } while (0)

/* Locals */
static int depth;				/* current XML depth */
static int cdepth;				/* "configuration" XML depth */
static xmlTextWriterPtr xp;
static xmlBufferPtr jreq_buf;
static char currentprefix[64];			/* currently configuring this */
static const char acld_mac_filter_dst[] = "ACLD-MAC-FILTER-DST";
static const char acld_mac_filter_src[] = "ACLD-MAC-FILTER-SRC";
static const char filter_dst[] = "filter-dst";
static const char filter_src[] = "filter-src";
static const char filter_default[] = "default";

/* Forwards */
static const char *addelementwithattr(const char *, const char *, const char *);
static const char *addnamedelement(const char *, const char *);
static const char *addnamedelementwithattr(const char *, const char *,
    const char *, const char *);
static const char *climbup(int);
static const char *jreq_adddel_blockmac(struct jrequest *);
static const char *jreq_add_firewall(struct jrequest *);
static const char *jreq_add_prefix(struct jrequest *);
static const char *jreq_add_prefix_addr(struct jrequest *);
static const char *jreq_start(void);
static const char *startchild(const char *);
static const char *startchildwithattr(const char *, const char *, const char *);

static const char *
addelementwithattr(const char *tag, const char *key, const char *val)
{
	int rc;
	const char *errmsg;

	rc = xmlTextWriterStartElement(xp, (xmlChar *)tag);
	if (rc < 0) {
		errmsg =
		    "addnamedelementwithattr: xmlTextWriterStartElement failed";
		return (errmsg);
	}
	++depth;
	rc = xmlTextWriterWriteAttribute(xp, (xmlChar *)key, (xmlChar *)val);
	if (rc < 0) {
		errmsg = "addnamedelementwithattr:"
		    " xmlTextWriterWriteAttribute failed";
		return (errmsg);
	}

	return (NULL);
}

static const char *
addnamedelement(const char *tag, const char *name)
{
	int rc;

	rc = xmlTextWriterWriteElement(xp, (xmlChar *)tag, (xmlChar *)name);
	if (rc < 0)
		return ("addnamedelement: xmlTextWriterWriteElement failed");
	return (NULL);
}

static const char *
addnamedelementwithattr(const char *tag, const char *name,
    const char *key, const char *val)
{
	int rc;
	const char *errmsg;

	rc = xmlTextWriterStartElement(xp, (xmlChar *)tag);
	if (rc < 0) {
		errmsg =
		    "addnamedelementwithattr: xmlTextWriterStartElement failed";
		return (errmsg);
	}
	++depth;
	rc = xmlTextWriterWriteAttribute(xp, (xmlChar *)key, (xmlChar *)val);
	if (rc < 0) {
		errmsg = "addnamedelementwithattr:"
		    " xmlTextWriterWriteAttribute failed";
		return (errmsg);
	}
	rc = xmlTextWriterWriteString(xp, (xmlChar *)name);
	if (rc < 0) {
		errmsg =
		    "addnamedelementwithattr: xmlTextWriterWriteString failed";
		return (errmsg);
	}

	return (NULL);
}

/* Climb back to depth2 */
static const char *
climbup(int depth2)
{
	int rc;

	if (depth2 < 0)
		depth2 = 0;
	while (depth > depth2) {
		rc = xmlTextWriterEndElement(xp);
		if (rc < 0)
			return ("endchild: xmlTextWriterEndElement failed");
		--depth;
	}
	return (NULL);
}

static const char *
jreq_adddel_blockmac(struct jrequest *jp)
{
	int doadd, len, depth2, depth3;
	const char *macstr;
	struct aclgroup *gp;
	struct acllist *ap;
	struct array *dp;

	doadd = (jp->type == REQ_BLOCKMAC);

	gp = aclgetmacgroup();
	if (gp == NULL)
		return ("jreq_adddel_blockmac: missing mac ACL group");
	ap = gp->acllist;
	if (ap == NULL)
		return ("jreq_adddel_blockmac: missing mac ACL");
	dp = &ap->acl_array;

	/* Length with the current request *already* appiled */
	len = dp->len;

	/* We need to start at configuration depth */
	if (depth != cdepth)
		CLIMBUP(cdepth);

	/* Prefix list change */
	STARTCHILD("firewall");
	STARTCHILD("family");
	STARTCHILD(jp->family);
	STARTCHILD("filter");
	ADDNAMEDELEMENT("name", acld_filter_name);

	depth2 = depth;
	/* Adding or deleting when any will be left */
	if (doadd || len > 0) {
		/*
		 * Remove default term (add then add back later)
		 * if we're adding dst/src terms
		 */
		if (have_acld_mac_filter && doadd && len == 1) {
			ADDELEMENTWITHATTR("term", "delete", "delete");
			ADDNAMEDELEMENT("name", filter_default);
			CLIMBUP(depth2);
		}

		STARTCHILD("term");
		ADDNAMEDELEMENT("name", filter_dst);
		depth3 = depth;
		STARTCHILD("from");
		macstr = mac2str(&jp->mac);
		if (doadd)
			STARTCHILD("destination-mac-address");
		else
			ADDELEMENTWITHATTR("destination-mac-address",
			    "delete", "delete");
		ADDNAMEDELEMENT("name", macstr);
		if (doadd && len == 1) {
			CLIMBUP(depth3);
			STARTCHILD("then");
			ADDNAMEDELEMENT("count", acld_mac_filter_dst);
			STARTCHILD("discard");
		}

		CLIMBUP(depth2);
		STARTCHILD("term");
		ADDNAMEDELEMENT("name", filter_src);
		depth3 = depth;
		STARTCHILD("from");
		if (doadd)
			STARTCHILD("source-mac-address");
		else
			ADDELEMENTWITHATTR("source-mac-address",
			    "delete", "delete");
		ADDNAMEDELEMENT("name", macstr);
		if (doadd && len == 1) {
			CLIMBUP(depth3);
			STARTCHILD("then");
			ADDNAMEDELEMENT("count", acld_mac_filter_src);
			STARTCHILD("discard");
		}

		if (doadd && len == 1) {
			CLIMBUP(depth2);
			STARTCHILD("term");
			ADDNAMEDELEMENT("name", filter_default);
			STARTCHILD("then");
			STARTCHILD("accept");
		}
	} else {
		/* Delete the src/dst terms when none will be left */
		depth3 = depth;
		ADDELEMENTWITHATTR("term", "delete", "delete");
		ADDNAMEDELEMENT("name", filter_dst);
		CLIMBUP(depth3);
		ADDELEMENTWITHATTR("term", "delete", "delete");
		ADDNAMEDELEMENT("name", filter_src);
	}

	// XXX we need to undo this if we fail
	if (doadd)
		have_acld_mac_filter = 1;

	return (NULL);
}

/* Add a firewall change */
static const char *
jreq_add_firewall(struct jrequest *jp)
{
	int dodelete, depth3;
	const char *thenaction, *prefixdirection;

	/* XXX need to delete port firewall config if removing last port */

	thenaction = "discard";
	prefixdirection = "prefix-list";
	dodelete = 0;
	switch (jp->type) {

	case REQ_RESTORETCPPORT:
	case REQ_RESTOREUDPPORT:
	case REQ_RESTORETCPSYNPORT:
		++dodelete;
		break;

	case REQ_PERMITUDPDSTHOSTPORT:
	case REQ_PERMITTCPDSTHOSTPORT:
		thenaction = "accept";
		prefixdirection = "destination-prefix-list";
		break;

	default:
		break;
	}

	switch (jp->type) {

	case REQ_DROP:
	case REQ_DROPUDPPORT:
	case REQ_DROPTCPPORT:
	case REQ_RESTORETCPPORT:
	case REQ_RESTOREUDPPORT:
	case REQ_DROPTCPSYNPORT:
	case REQ_RESTORETCPSYNPORT:
	case REQ_PERMITUDPDSTHOSTPORT:
	case REQ_PERMITTCPDSTHOSTPORT:
		/* We need to start at configuration depth */
		if (depth != cdepth)
			CLIMBUP(cdepth);
		currentprefix[0] = '\0';

		STARTCHILD("firewall");
		STARTCHILD("family");
		STARTCHILD(jp->family);
		STARTCHILD("filter");
		ADDNAMEDELEMENT("name", jp->filter);

		if (!dodelete) {
			STARTCHILD("interface-specific");
			CLIMBUP(depth - 1);
		}

		if (!jp->doremoveterm)
			STARTCHILD("term");
		else
			STARTCHILDWITHATTR("term", "delete", "delete");
		ADDNAMEDELEMENT("name", jp->seq);

		/* This is all we need to do when deleting the term */
		if (jp->doremoveterm)
			break;

		depth3 = depth;
		STARTCHILD("from");

		/* Add address if present */
		if (jp->addr[0] != '\0') {
			STARTCHILD(prefixdirection);
			ADDNAMEDELEMENT("name", jp->prefix);
			CLIMBUP(depth - 1);
		}

		/* Add protocol if present (and not deleting a port) */
		if (!dodelete && jp->proto[0] != '\0') {
			ADDNAMEDELEMENT(jp->protokey, jp->proto);
		}

		/* Add port if present */
		if (jp->port[0] != '\0') {
			if (!dodelete)
				ADDNAMEDELEMENT("destination-port", jp->port);
			else
				ADDNAMEDELEMENTWITHATTR("destination-port",
				    jp->port, "delete", "delete");
		}

		/* Add optional syn requirement */
		if (jp->type == REQ_DROPTCPSYNPORT) {
			STARTCHILD("tcp-initial");
			CLIMBUP(depth - 1);
		}

		CLIMBUP(depth3);
		if (!dodelete) {
			STARTCHILD("then");

			/* Want "<discard></discard>" instead of "<discard/>" */
			ADDNAMEDELEMENT(thenaction, "");
		}
		break;

	case REQ_RESTORE:
	case REQ_UNPERMITTCPDSTHOSTPORT:
	case REQ_UNPERMITUDPDSTHOSTPORT:
		/* We don't currently try and clean up unused filter terms */
		break;

	default:
		abort();
	}
	return (NULL);
}

static const char *
jreq_add_prefix(struct jrequest *jp)
{
	/* Optimize for updating prefix lists */
	if (jp->jtype == JREQ_PREFIX &&
	    jp->addr[0] != '\0' &&
	    jp->prefix[0] != '\0' &&
	    strcmp(currentprefix, jp->prefix) == 0) {
		JREQ_ADD_PREFIX_ADDR(jp);
		return (NULL);
	}

	switch (jp->type) {

	case REQ_DROP:
	case REQ_RESTORE:
	case REQ_PERMITUDPDSTHOSTPORT:
	case REQ_PERMITTCPDSTHOSTPORT:
	case REQ_UNPERMITTCPDSTHOSTPORT:
	case REQ_UNPERMITUDPDSTHOSTPORT:
	case REQ_PREFIX:
	case REQ_NOPREFIX:
		/* We need to start at configuration depth */
		if (depth != cdepth)
			CLIMBUP(cdepth);
		currentprefix[0] = '\0';

		/* Prefix list change */
		STARTCHILD("policy-options");
		STARTCHILD("prefix-list");
		ADDNAMEDELEMENT("name", jp->prefix);
		strlcpy(currentprefix, jp->prefix, sizeof(currentprefix));

		JREQ_ADD_PREFIX_ADDR(jp);
		break;

	case REQ_BLOCKMAC:
	case REQ_RESTOREMAC:
	case REQ_DROPUDPPORT:
	case REQ_DROPTCPPORT:
	case REQ_RESTORETCPPORT:
	case REQ_RESTOREUDPPORT:
	case REQ_DROPTCPSYNPORT:
	case REQ_RESTORETCPSYNPORT:
		/* Ports and macs don't use prefix-lists */
		break;

	default:
		abort();
	}
	return (NULL);
}

static const char *
jreq_add_prefix_addr(struct jrequest *jp)
{
	switch (jp->type) {

	case REQ_RESTORE:
	case REQ_UNPERMITTCPDSTHOSTPORT:
	case REQ_UNPERMITUDPDSTHOSTPORT:
	case REQ_NOPREFIX:
		STARTCHILDWITHATTR("prefix-list-item", "delete", "delete");
		break;
	default:
		STARTCHILD("prefix-list-item");
		break;
	}
	ADDNAMEDELEMENT("name", jp->addr);
	CLIMBUP(depth - 1);
	return (NULL);
}

const char *
jreq_add(struct jrequest *jp)
{
	/* Start the configuration if we haven't already */
	if (xp == NULL)
		JREQ_START();

	switch (jp->jtype) {

	case JREQ_PREFIX:
		/* Only add firewall config if necessary */
		if (jp->dofirewall)
			JREQ_ADD_FIREWALL(jp);
		JREQ_ADD_PREFIX(jp);
		break;

	case JREQ_MACFILTER:
		JREQ_ADDDEL_BLOCKMAC(jp);
		break;

	default:
		lg(LOG_ERR, "jreq_add: unexpected jtype value %d", jp->jtype);
		abort();
	}

	return (NULL);
}

/* Returns true if there is XML that hasn't been rendered */
int
jreq_dirty(void)
{
	return (xp != NULL);
}

void
jreq_cleanup(void)
{
	if (xp != NULL) {
		xmlFreeTextWriter(xp);
		xp = NULL;
	}
	if (jreq_buf != NULL) {
		xmlBufferFree(jreq_buf);
		jreq_buf = NULL;
	}
	depth = 0;
	cdepth = 0;
	currentprefix[0] = '\0';
}

void
jreq_init(void)
{
	depth = 0;
	if (xp != NULL)
		jreq_cleanup();
}

const char *
jreq_render(const char **pp)
{
	int rc;

	*pp = NULL;
	rc = xmlTextWriterEndDocument(xp);
	if (rc < 0)
		return ("Error at xmlTextWriterEndDocument");

	*pp = (const char *)jreq_buf->content;
	return (NULL);
}

static const char *
jreq_start(void)
{
	int rc;

	/* Create the XML buffer */
	jreq_buf = xmlBufferCreate();
	if (jreq_buf == NULL)
		return ("Error creating the xml buffer");

	/* Create the XML writer */
	xp = xmlNewTextWriterMemory(jreq_buf, 0);
	if (xp == NULL)
		return ("Error creating the xml writer");

	/* Indent output */
	rc = xmlTextWriterSetIndent(xp, 1);
	if (rc < 0)
		return ("Error at xmlTextWriterSetIndent");

	/* Indent using 4 blanks */
	rc = xmlTextWriterSetIndentString(xp, (const xmlChar *)"    ");
	if (rc < 0)
		return ("Error at xmlTextWriterSetIndentString");

	depth = 0;

	STARTCHILD("rpc");
	STARTCHILDWITHATTR("load-configuration", "action", "merge");
	STARTCHILD("configuration");

	/* Save "configuration" depth */
	cdepth = depth;
	return (NULL);
}

static const char *
startchild(const char *tag)
{
	int rc;

	rc = xmlTextWriterStartElement(xp, (xmlChar *)tag);
	if (rc < 0)
		return ("startchild: xmlTextWriterStartElement failed");
	++depth;
	return (NULL);
}

static const char *
startchildwithattr(const char *tag, const char *key, const char *val)
{
	int rc;
	const char *errmsg;

	rc = xmlTextWriterStartElement(xp, (xmlChar *)tag);
	if (rc < 0) {
		errmsg = "startchildwithattr: xmlTextWriterStartElement failed";
		return (errmsg);
	}
	rc = xmlTextWriterWriteAttribute(xp, (xmlChar *)key, (xmlChar *)val);
	if (rc < 0) {
		errmsg =
		    "startchildwithattr: xmlTextWriterWriteAttribute failed";
		return (errmsg);
	}
	++depth;
	return (NULL);
}
