/*
 * Copyright (c) 2012, 2013, 2014, 2015, 2016, 2017, 2018, 2019, 2020, 2021, 2022, 2024
 * The Regents of the University of California. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *     * Redistributions of source code must retain the above copyright
 *       notice, this list of conditions and the following disclaimer.
 *     * Redistributions in binary form must reproduce the above copyright
 *       notice, this list of conditions and the following disclaimer in the
 *       documentation and/or other materials provided with the distribution.
 *     * Neither the name of the University nor the names of its contributors
 *       may be used to endorse or promote products derived from this software
 *       without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#ifndef lint
static const char rcsid[] __attribute__ ((unused)) =
    "@(#) $Id$ (LBL)";
#endif

#include <sys/types.h>
#include <sys/mman.h>
#include <sys/time.h>

#include <netinet/tcp.h>

#include <arpa/inet.h>

#include <ctype.h>
#include <errno.h>
#include <stdio.h>
#include <string.h>
#include <sysexits.h>
#include <unistd.h>

#include <libssh2.h>

#include "acld.h"
#include "child.h"
#include "jreq.h"
#include "junoscript.h"
#include "prefix.h"
#include "rpc.h"
#include "timer.h"

/* Globals */
int havempc;				/* True if we have a MPC else DPC */
int sentconfig;				/* Sent the configuration change */
int sentcommit;				/* Sent the commit */
int have_acld_mac_filter;

const char acld_filter_name[] = "ACLD-MAC-FILTER";

struct ifacemac *ifacemacs;
struct array ifacemacs_array;

/* Locals */
static const char suffix_host[] = "HOST";
static const char suffix_hostport[] = "HOSTPORT";
static const char suffix_port[] = "PORT";
static struct log log_js;
static char remote[64];			/* name of router we're connecting to */
static int sentprivate;			/* sent the private configuration rpc */
static struct iobuf wbuf;		/* child write buffer */

/* Used to fetch the relevant ACL rules */
static const char *aclsuffixes[] = {
	suffix_host,
	suffix_hostport,
	suffix_port,
	NULL,
};
#define NUMACLSUFFIXES ((sizeof(aclsuffixes) / sizeof(*aclsuffixes)) - 1)

/* Points to next port block acllist and acl */
static struct acl *nextacl;
static struct acllist *nextacllist;

/* Points to next ACL or prefix list suffix to list */
static const char **nextsuffix;

static struct s2v acl2aclsuffix[] = {
	{ suffix_host,		ATYPE_BLOCKHOST },
	{ suffix_host,		ATYPE_BLOCKNET },
	{ suffix_hostport,	ATYPE_PERMITUDPDSTHOSTPORT },
	{ suffix_hostport,	ATYPE_PERMITTCPDSTHOSTPORT },
	{ suffix_port,		ATYPE_BLOCKUDPPORT },
	{ suffix_port,		ATYPE_BLOCKTCPPORT },
	{ suffix_port,		ATYPE_BLOCKTCPSYNPORT },
	{ "",			ATYPE_BLOCKMAC },
	{ NULL,			-1, }
};

static LIBSSH2_SESSION *session;
static LIBSSH2_CHANNEL *channel;

static const char js_vers[] = "1.0";
static const char js_rel[] = "13.3R8.7";

static const char js_pi_fmt[64] =
    "<?xml version=\"%s\" encoding=\"us-ascii\"?>";
static const char js_js_fmt[64] =
    "<junoscript version=\"%s\" hostname=\"%s\" release=\"%s\">";

// static const char js_abort[] = "<abort/>";
static char js_abort_acknowledgement[] = "<abort-acknowledgement/>\n";

static const char xml_comment[] = "<!--";
static const char xml_junoscript[] = "<junoscript ";
static const char xml_xml[] = "<?xml ";

/*
 * Given:
 *
 *     acl ACLD 0.0.0.0/0
 *
 * Here are firewall filters names we'll have:
 *
 *     ACLD-HOST
 *     ACLD-HOSTPORT
 *     ACLD-PORT
 *
 * Here are prefix list names we'll have (port 53 example):
 *
 *     ACLD-HOST
 *     ACLD-TCP-HOSTPORT53
 *     ACLD-UDP-HOSTPORT53
 *     ACLD-PORT
 */

/*
 * Hint: To discover the XML tag names use something like:
 *
 *    show configuration ... | display xml
 *
 */

/* Request chassis modules, prefix lists and firewall filters */
static const char js_getconfiguration[] =
    "<rpc>"
	"<get-chassis-inventory></get-chassis-inventory>"
    "</rpc>"
    "<rpc>"
	"<get-configuration>"
	    "<configuration>"
		"<policy-options>"
		    "<prefix-list></prefix-list>"
		"</policy-options>"
		"<firewall>"
		    "<family>"
			"<inet>"
			    "<filter></filter>"
			"</inet>"
			"<inet6>"
			    "<filter></filter>"
			"</inet6>"
			"<bridge>"
			    "<filter></filter>"
			"</bridge>"
		    "</family>"
		"</firewall>"
		"<bridge-domains>"
		    "<domain>"
			"<name></name>"
			"<vlan-id></vlan-id>"
			"<forwarding-options>"
			    "<filter></filter>"
			"</forwarding-options>"
		    "</domain>"
		"</bridge-domains>"
	    "</configuration>"
	"</get-configuration>"
    "</rpc>";

/* Arguments: comment */
static const char js_commit_fmt[] =
    "<rpc>"
	 "<commit-configuration>"
	     "<synchronize/>"
	     "<log>%s@%s</log>"
	 "</commit-configuration>"
    "</rpc>";

static const char js_open_private_configuration[] =
    "<rpc>"
	"<open-configuration>"
	    "<private/>"
	"</open-configuration>"
    "</rpc>";

static const char js_discard_private_configuration[] =
    "<rpc>"
	"<close-configuration/>"
    "</rpc>";

static const char js_uptime_cmd[] =
    "<rpc>"
	"<get-system-uptime-information/>"
    "</rpc>";

static const char js_request_end_session[] =
    "<rpc>"
	"<request-end-session/>"
    "</rpc>";

/* Forwards */
static const char *junoscriptcheckreq(struct req *);
static void junoscriptchildlistroute(void);
static void junoscriptclientcompact(struct client *,
    struct req *);
static void junoscriptcommit(void);
static void junoscriptconnect(void);
static int junoscriptinput(void);
static void junoscriptkill(void);
static void junoscriptlistacl(void);
static void junoscriptlistroute(struct client *,
    struct req *, int);
static void junoscriptlog(const char *, const char *);
static void junoscriptlogin(void);
static const char *junoscriptmapaclname(const char *, struct acl *);
static void junoscriptnullzero(struct client *, struct req *);
static void junoscriptprefix(struct client *, struct req *);
static void __attribute__ ((format (printf, 1, 2)))
    junoscriptqueue(const char *, ...);
static void junoscriptquerynullzero(struct client *,
    struct req *);
static int junoscriptread(int, struct iobuf *);
static int junoscriptready(void);
static void junoscriptrequest(struct aclgroup *, struct req *);
static void __attribute__ ((format (printf, 1, 2)))
    junoscriptsend(const char *, ...);
static void junoscriptsendattr(void);
static void junoscriptserverayt(void);
static void junoscriptservercompact(void);
static void junoscriptserversync(void);
static void junoscriptwrite(struct iobuf *);

static const char *
junoscriptcheckreq(struct req *rp)
{
	const char *stype, *sreq;
	static char buf[132];

	switch (rp->type) {

	case REQ_BLOCKMAC:
	case REQ_RESTOREMAC:
		/* No other checks are necessary */
		return (NULL);

	case REQ_DROP:
	case REQ_DROPTCPPORT:
	case REQ_DROPTCPSYNPORT:
	case REQ_DROPUDPPORT:
	case REQ_PERMITTCPDSTHOSTPORT:
	case REQ_PERMITUDPDSTHOSTPORT:
	case REQ_RESTORE:
	case REQ_RESTORETCPPORT:
	case REQ_RESTORETCPSYNPORT:
	case REQ_RESTOREUDPPORT:
	case REQ_UNPERMITTCPDSTHOSTPORT:
	case REQ_UNPERMITUDPDSTHOSTPORT:
		sreq = NULL;
		break;

	default:
		sreq = val2str(cmd2req, rp->type);
		break;
	}

	switch (rp->acl.type) {

	case ATYPE_BLOCKHOST:
	case ATYPE_BLOCKNET:
	case ATYPE_BLOCKTCPPORT:
	case ATYPE_BLOCKTCPSYNPORT:
	case ATYPE_BLOCKUDPPORT:
	case ATYPE_PERMITTCPDSTHOSTPORT:
	case ATYPE_PERMITUDPDSTHOSTPORT:
		stype = NULL;
		break;

	default:
		stype = val2str(str2acl, rp->acl.type);
		break;
	}
	if (stype == NULL && sreq == NULL)
		return (NULL);

	if (sreq == NULL)
		sreq = val2str(cmd2req, rp->type);
	if (stype == NULL)
		stype = val2str(str2acl, rp->acl.type);

	if (strcmp(sreq, stype) == 0)
		(void)snprintf(buf, sizeof(buf),
		    "junoscript request %s unsupported", sreq);
	else
		(void)snprintf(buf, sizeof(buf),
		    "junoscript request %s unsupported (type %s)", sreq, stype);
	return (buf);
}

static void
junoscriptchildlistroute(void)
{
	lg(LOG_DEBUG, "junoscriptchildlistroute: not implemented");
	++state.listedroutes;
}

static void
junoscriptclientcompact(struct client *cl, struct req *rp)
{
	lg(LOG_DEBUG, "junoscriptclientcompact: noop");
}

/* Child hook to start the commit process */
static void
junoscriptcommit(void)
{
	const char *errmsg, *p;
	struct client *cl;
	struct req *rp;
	struct timer *tp;
	struct state *sp;

	/* Bail if no pending configuration to load */
	sp = &state;
	tp = &sp->t_child;
	if (!jreq_dirty()) {
		timerreset(tp);
		return;
	}

	/* XXX should we check sentconfig and sentcommit? */

	/* Bail if there are any pending requests that might come our way */
	for (cl = sp->clients; cl != NULL; cl = cl->next) {
		for (rp = cl->req; rp != NULL; rp = rp->next) {
			if (rp->state != RSTATE_PENDING)
				continue;
			switch (rp->type) {

			case REQ_BLOCKHOSTHOST:
			case REQ_DROP:
			case REQ_DROPTCPDSTHOSTPORT:
			case REQ_DROPTCPPORT:
			case REQ_DROPTCPSYNPORT:
			case REQ_DROPUDPPORT:
			case REQ_NOPREFIX:
			case REQ_PREFIX:
			case REQ_PERMITTCPDSTHOSTPORT:
			case REQ_PERMITUDPDSTHOSTPORT:
			case REQ_RESTORE:
			case REQ_RESTOREHOSTHOST:
			case REQ_RESTORETCPPORT:
			case REQ_RESTORETCPSYNPORT:
			case REQ_RESTOREUDPPORT:
			case REQ_UNPERMITTCPDSTHOSTPORT:
			case REQ_UNPERMITUDPDSTHOSTPORT:
				timerreset(tp);
				return;

			default:
				break;
			}
		}
	}

	/* Wait a short time to allow client I/O to catch up */
	switch (timercheck(tp)) {

	case -1:
		/* Start the timer */
		timerset(tp, 1);
		return;

	case 0:
		/* It's time! */
		timerreset(tp);
		break;

	default:
		/* Not yet */
		return;
	}

	errmsg = jreq_render(&p);
	if (errmsg) {
		// XXX XXX XXX
		/* XXX how to handle error? */
		lg(LOG_ERR, "jreq_render: %s", errmsg);
	} else {
		if (debug == 3)
			printf("%s\n", p);
		else
			junoscriptqueue("%s", p);
	}
	++sentconfig;

	jreq_cleanup();

	/* Wait for <load-configuration> before doing actual commit */
}

static void
junoscriptconnect(void)
{
	int s, rc, opterrno;
	socklen_t len;
	const char *p;
	char *errmsg;
	struct state *sp;

	/* Always disable the child connect hook */
	sp = &state;
	sp->f_childconnect = NULL;

	/* Make sure we still have a socket */
	s = sp->wfd;
	if (s < 0)
		return;

	/* We're committed now */
	sp->wfd = -1;

	/* Status check on the socket */
	len = sizeof(opterrno);
	if (getsockopt(s, SOL_SOCKET, SO_ERROR, &opterrno, &len) < 0) {
		lg(LOG_ERR, "junoscriptconnect: %s: getsockopt: %s",
		    remote, strerror(errno));

		/* Clean up */
		(void)close(s);
		return;
	}
	if (opterrno != 0) {
		lg(LOG_ERR, "connect error to %s: %s",
		    remote, strerror(opterrno));

		/* Clean up */
		(void)close(s);
		return;
	}

	/* Turn blocking I/O back on */
	if (blocking(s, 1) < 0)
		exit(EX_OSERR);

	/* Open a channel */
	session = libssh2_session_init();
	if (libssh2_session_handshake(session, s) < 0) {
		lg(LOG_ERR,
		    "junoscriptconnect: %s: libssh2_session_handshake failed",
		    remote);

		/* Clean up */
		(void)close(s);
		return;
	}

#ifdef notdef
	/* XXX todo: check the router's ssh fingerprint (SSHFP?) */
	p = libssh2_hostkey_hash(session, LIBSSH2_HOSTKEY_HASH_SHA1);
#endif
	errno = 0;
	rc = libssh2_userauth_publickey_fromfile(session, cf->c_euser,
	    cf->c_sshpubfn, cf->c_sshkeyfn, NULL);
	if (rc < 0) {
		errmsg = NULL;
		(void)libssh2_session_last_error(session, &errmsg, NULL, 0);
		lg(LOG_ERR, "libssh2_userauth_publickey_fromfile: %s", errmsg);

		/* Clean up */
		libssh2_session_free(session);
		session = NULL;
		(void)close(s);
		return;
	}

	/* Open a channel */
	channel = libssh2_channel_open_session(session);
	if (channel == NULL) {
		lg(LOG_ERR,
		    "junoscriptconnect: %s: failed to open a channel", remote);

		/* Clean up */
		libssh2_session_free(session);
		session = NULL;
		(void)close(s);
		return;
	}

	p = libssh2_session_banner_get(session);
	if (p == NULL) {
		lg(LOG_ERR, "junoscriptconnect:"
		    " %s: libssh2_session_banner_get failed", remote);

		/* Clean up */
		(void)close(s);
		libssh2_channel_free(channel);
		channel = NULL;
		libssh2_session_free(session);
		session = NULL;
		return;
	}

	/* We're connected now */
	lg(LOG_DEBUG, "connected to %s: %s", remote, p);
	setastate(ASTATE_CONNECTED);

	/* Get a channel */
	for (;;) {
		rc = libssh2_channel_exec(channel, "junoscript");
		if (rc == LIBSSH2_ERROR_EAGAIN) {
			lg(LOG_ERR, "junoscriptconnect: Would block; sleeping");
			sleep(1);
			continue;
		}
		break;
	}
	if (rc < 0) {
		lg(LOG_ERR, "junoscriptconnect:"
		    " %s: libssh2_channel_exec failed: %d", remote, rc);

		/* Clean up */
		(void)close(s);
		libssh2_channel_free(channel);
		channel = NULL;
		libssh2_session_free(session);
		session = NULL;
		setastate(ASTATE_NOTCONNECTED);
		return;
	}

	/* Configure non-blocking */
	libssh2_session_set_blocking(session, 0);

	sp->rfd = s;
	sp->wfd = s;

	/* Emit PI */
	junoscriptsend(js_pi_fmt, js_vers);

	/* Emit tag */
	junoscriptsend(js_js_fmt, js_vers, hostname, js_rel);

	/* Reset login timer */
	timerreset(&sp->t_login);
}

/* Actually do the commit */
void
junoscriptdocommit(void)
{
	junoscriptsend(js_commit_fmt, cf->c_euser, hostname);
	++sentcommit;
}

void
junoscriptexitprivate(void)
{
	if (sentprivate) {
		junoscriptsend(js_discard_private_configuration);
		sentprivate = 0;
	}
}

void
junoscriptinit(void)
{
	int i, j;
	const char **pp;
	struct aclgroup *gp;
	struct acllist *ap;
	struct array *dp, *dp2, *dp3;
	char aclname[64];
	struct state *sp;

	/* Static initialization */
	sp = &state;
	nextsuffix = NULL;
	nextacllist = NULL;
	nextacl = NULL;
	sentprivate = 0;
	sentconfig = 0;
	sentcommit = 0;
	havempc = 0;
	have_acld_mac_filter = 0;
	sp->multiball = 1;
	IOBUF_INIT(&wbuf);

	/* Child hooks */
	sp->f_childcheckreq = junoscriptcheckreq;
	sp->f_childconnect = NULL;
	sp->f_childinput = junoscriptinput;
	sp->f_childkill = junoscriptkill;
	sp->f_childlistacl = junoscriptlistacl;
	sp->f_childlistroute = junoscriptchildlistroute;
	sp->f_childlogin = junoscriptlogin;
	sp->f_childread = junoscriptread;
	sp->f_childrequest = junoscriptrequest;
	sp->f_childsend = junoscriptsend;
	sp->f_childsendattr = junoscriptsendattr;
	sp->f_childcommit = junoscriptcommit;
	sp->f_childready = junoscriptready;

	/* Client hooks */
	sp->f_clientcompact = junoscriptclientcompact;
	sp->f_clientlistroute = junoscriptlistroute;
	sp->f_clientmapaclname = junoscriptmapaclname;
	sp->f_clientnullzero = junoscriptnullzero;
	sp->f_clientquerynullzero = junoscriptquerynullzero;
	sp->f_clientprefix = junoscriptprefix;

	/* Server hooks */
	sp->f_serverayt = junoscriptserverayt;
	sp->f_servercompact = junoscriptservercompact;
	sp->f_serversync = junoscriptserversync;

	/* Junoscript log */
	if (cf->c_junoscriptlog != NULL) {
		log_js.fn = cf->c_junoscriptlog;
		checklog(&log_js);
	}

	/* Clobber remote server */
	strlcpy(remote, "?", sizeof(remote));

	/* Check API for match with the DLL we are using */
	LIBXML_TEST_VERSION;

	/* Initialize Junoscript configuration generation */
	jreq_init();

	/* Expand each ACL group to have the necessary number of ACL lists */
	dp = &cf->c_aclgroup_array;
	for (i = 0, gp = cf->c_aclgroup; i < dp->len; ++i, ++gp) {
		if (gp->ismac)
			continue;
		dp2 = &gp->acllist_array;
		DYNARRAY(dp2, gp->acllist, NUMACLSUFFIXES, "junoscriptinit");
		dp2->len = NUMACLSUFFIXES;
		free(gp->acllist->name);
		gp->acllist->name = NULL;
		pp = aclsuffixes;
		for (j = 0, ap = gp->acllist; j < dp2->len; ++j, ++ap) {
			(void)snprintf(aclname, sizeof(aclname), "%s-%s",
			    gp->name, *pp);
			++pp;
			ap->name = strsave(aclname);

			dp3 = &ap->acl_array;
			dp3->osize = sizeof(*ap->acl);
			dp3->inclen = gp->acllist->acl_array.inclen;
		}
	}
}

/* Process input from the rotuer, return 1 if done */
static int
junoscriptinput(void)
{
	int i;
	size_t cc;
	char *cp;
	struct iobuf *ip;
	struct req *rp;
	struct rpcresult *rrp;
	xmlDocPtr doc;
	xmlNode *root;
	char buf[128];
	struct state *sp;

	sp = &state;
	ip = &sp->rbuf;
	if (IOBUF_LEN(ip) == 0)
		return (1);

	switch (sp->state) {

	case ASTATE_CONNECTED:
		/*
		 * Look for the comment that says we're logged in:
		 *
		 *    <!-- user zeek, class j-superuser-enforce-private -->\n
		 */
		while (iohaveline(ip) && (cp = iogetstr(ip)) != NULL) {
			if (ISXMLCOMMENT(cp)) {
				if (strstr(cp, "session start") != NULL ||
				    strstr(cp, "No zombies") != NULL)
					continue;
				(void)snprintf(buf, sizeof(buf),
				    "<!-- user %s,", cf->c_euser);
				if (strncmp(cp, buf, strlen(buf)) == 0) {
					setastate(ASTATE_LOGGEDIN);
					timerreset(&sp->t_login);
					break;
				}
				if (debug)
					lg(LOG_DEBUG, "junoscriptinput: \"%s\"",
					    pretty(cp));
				continue;
			}
			if (ISXMLTAG(cp, xml_xml) ||
			    ISXMLTAG(cp, xml_junoscript)) {
				if (debug > 1)
					lg(LOG_DEBUG, "junoscriptinput: \"%s\"",
					    pretty(cp));
				continue;
			}
			lg(LOG_DEBUG, "junoscriptinput: unexpected \"%s\"",
				pretty(cp));
		}
		break;

	case ASTATE_LOGGEDIN:
	case ASTATE_SENTLISTACL:
		break;

	default:
		lg(LOG_DEBUG, "junoscriptinput: %s not implemented",
		    val2str(astate2str, sp->state));
		abort();
	}

	while (iohaveline(ip) && ISXMLCOMMENT(ip->buf) &&
	    (cp = iogetstr(ip)) != NULL) {
		/* Did the router time our session out? */
		if (strstr(cp, "session end") != NULL) {
			lg(LOG_ERR, "junoscriptinput: router ended session");
			setastate(ASTATE_NOTCONNECTED);
			junoscriptkill();
			return (1);
		}

		if (debug)
			lg(LOG_DEBUG, "junoscriptinput: # %s", pretty(cp));
	}

	/* Look for an abort acknowledgement */
	cc = sizeof(js_abort_acknowledgement) - 1;
	if (IOBUF_LEN(ip) >= cc &&
	    strncmp(ip->buf, js_abort_acknowledgement, cc) == 0) {
		lg(LOG_INFO, "abort acknowled");
		ioeat(ip, cc);
	}

	cc = iohaverpcreply(ip);
	if (cc == 0)
		return (1);
	timerset(&sp->t_ayt, cf->c_ayt_secs);
	// XXX could make j.xml something more descriptive
	doc = xmlReadMemory(ip->buf, cc, "j.xml", NULL, XML_PARSE_NOERROR);
	ioeat(ip, cc);
	if (doc == NULL) {
		i = cc;
		if (i > 16)
			i = 16;
		lg(LOG_DEBUG,
		    "junoscriptinput: xmlReadMemory failed: \"%.*s ...\"",
		    i, ip->buf);
		return (0);
	}

	/* Parse the XML */
	root = xmlDocGetRootElement(doc);
	rrp = NULL;
	switch (sp->state) {

	case ASTATE_LOGGEDIN:
	case ASTATE_SENTLISTACL:
		rrp = rpc_process(root);
		break;

	default:
		lg(LOG_ERR, "junoscriptinput: Can't handle state %s",
		    val2str(astate2str, sp->state));
		break;
	}

	if (rrp != NULL && rrp->errmsg[0] != '\0')
		lg(LOG_DEBUG, "junoscriptinput: %s", pretty(rrp->errmsg));

	rpc_cleanup(doc);
	doc = NULL;

	/* Fix up things if the router command failed */
	/* XXX can we test more things here? */
	ip = &wbuf;
	if (rrp->loaderror) {
		/* Server request: revert changes */
		if ((rp = sp->req) != NULL) {
			childaclcleanup(rp);
			return (0);
		}

		/* Kill queued output? will there be any? */
		// IOBUF_LEN(ip) = 0;

		return (0);
	}

	/*
	 * If we requested a private copy of the configuration, have
	 * pending configuration changes to send and got a reply without
	 * an error, send the changes.
	 */
	if (sentprivate && IOBUF_LEN(ip) > 0)
		junoscriptwrite(ip);
	return (0);
}

static void
junoscriptkill(void)
{
	struct state *sp;

	sp = &state;
	if (sp->wfd >= 0 && sp->state == ASTATE_LOGGEDIN)
		junoscriptsend(js_request_end_session);
	setastate(ASTATE_NOTCONNECTED);
	sentprivate = 0;
	havempc = 0;
	sp->cmd = NULL;
	IOBUF_INIT(&wbuf);

	(void)libssh2_session_disconnect(session, "goodbye");

	if (channel != NULL) {
		libssh2_channel_free(channel);
		channel = NULL;
	}
	if (session != NULL) {
		libssh2_session_free(session);
		session = NULL;
	}

	if (sp->rfd >= 0) {
		(void)close(sp->rfd);
		/* In case they are the same fd (e.g. libssh2) */
		if (sp->rfd == sp->wfd)
			sp->wfd = -1;
		sp->rfd = -1;
	}

	if (sp->wfd >= 0) {
		(void)close(sp->wfd);
		sp->wfd = -1;
	}

	iofree(&sp->rbuf);

	timerreset(&sp->t_ayt);
	timerreset(&sp->t_login);
	timerreset(&sp->t_sync);
}

/* Loop over the junos ACL suffix types */
static void
junoscriptlistacl(void)
{
	setastate(ASTATE_SENTLISTACL);
	junoscriptsend(js_getconfiguration);
}

static void
junoscriptlistroute(struct client *cl, struct req *rp,
    int nullzeroonly)
{
	clientsenderr(cl, rp, "junoscript doesn't support listroute");
}

static void
junoscriptlog(const char *what, const char *p)
{
	int len;
	const char *p2, *tssp;

	if (log_js.fn != NULL) {
		checklog(&log_js);
		tssp = tsstr();
		p2 = p;
		while (*p2 != '\0') {
			if (*p2 != '\r' && *p2 != '\n' && *p2 != '\0') {
				++p2;
				continue;
			}
			len = p2 - p;
			fprintf(log_js.f, "%s %s: %.*s", tssp, what, len, p);
			while (*p2 == '\r' || *p2 == '\n') {
#ifdef notdef
				fprintf(log_js.f, "\\%c",
				    (*p2 == '\r') ?  'r' : 'n');
#endif
				++p2;
			}
			fprintf(log_js.f, "\n");
			p = p2;
		}
		if (*p != '\0')
			fprintf(log_js.f, "%s %s: %s\n", tssp, what, p);
		fflush(log_js.f);
	}
}

static void
junoscriptlogin(void)
{
	int s;
	socklen_t salen;
	const char *errmsg;
	struct addr *a;
	struct addr addr;
	struct sockaddr_storage sa;
	struct sockaddr *sap;
	struct state *sp;

	sp = &state;
	a = &addr;
	memset(a, 0, sizeof(*a));
	errmsg = extractaddr(cf->c_router, NULL, a);
	if (errmsg != NULL) {
		lg(LOG_ERR, "junoscriptlogin: bad router address: %s: %s",
		    cf->c_router, errmsg);
		return;
	}

	/* Free ACL lists so we'll re-aquire them from the router */
	aclgroupsfree();

	sap = (struct sockaddr *)&sa;
	salen = addr2sa(a, cf->c_portssh, sap);
	(void)snprintf(remote, sizeof(remote), "%s.%d", addr2str(a),
	    (int)cf->c_portssh);

	if (libssh2_init(0) < 0) {
		lg(LOG_ERR, "junoscriptlogin: libssh2_init failed");
		return;
	}

	/* Clean up old sockets in case the last attempt failed */
	if (sp->rfd >= 0) {
		(void)close(sp->rfd);
		/* In case they are the same fd (e.g. libssh2) */
		if (sp->rfd == sp->wfd)
			sp->wfd = -1;
		sp->rfd = -1;
	}

	if (sp->wfd >= 0) {
		(void)close(sp->wfd);
		sp->wfd = -1;
	}

	s = socket(a->family, SOCK_STREAM, 0);
	if (s < 0) {
		lg(LOG_ERR, "junoscriptlogin: socket %s: %s",
		    remote, strerror(errno));
		return;
	}

	if (blocking(s, 0) < 0)
		exit(EX_OSERR);
	errno = 0;
	if (connect(s, sap, salen) < 0) {
		if (errno != EINPROGRESS) {
			lg(LOG_ERR, "connect to %s: %s",
			    remote, strerror(errno));

			/* Clean up */
			(void)close(s);
			return;
		}
		lg(LOG_DEBUG, "connecting to %s", remote);
	} else {
		lg(LOG_ERR, "junoscriptlogin: Connect can't happen?: %s",
		    remote);
	}

	/* Enable child hook */
	sp->wfd = s;
	sp->f_childconnect = junoscriptconnect;
}

static const char *
junoscriptmapaclname(const char *groupname, struct acl *al)
{
	const char *p;
	static char aclname[64];

	p = val2str(acl2aclsuffix, al->type);
	if (p != NULL && *p != '\0') {
		(void)snprintf(aclname, sizeof(aclname), "%s-%s", groupname, p);
		return (aclname);
	}
	return (groupname);
}

static void
junoscriptnullzero(struct client *cl, struct req *rp)
{
	clientsenderr(cl, rp, "junoscript doesn't support null zero routes");
}

/* Queue config output to be sent after we have a private config */
static void
junoscriptqueue(const char *fmt, ...)
{
	struct iobuf *ip;
	va_list ap;

	if (!sentprivate) {
		junoscriptsend(js_open_private_configuration);
		++sentprivate;
	}

	ip = &wbuf;
	va_start(ap, fmt);
	ioappendvfmt(ip, fmt, ap);
	va_end(ap);
}

static void
junoscriptprefix(struct client *cl, struct req *rp)
{
	const char *errmsg, *fmt;
	struct jrequest jrequest;
	struct jrequest *jp;

	switch (rp->type) {

	case REQ_PREFIX:
		errmsg = prefixaddaddr(rp->prefixname, &rp->prefix);
		if (errmsg != NULL) {
			lg(LOG_ERR, "junoscriptprefix: prefix %s", errmsg);
			clientsenderr(cl, rp, "%s", errmsg);
			return;
		}
		break;

	case REQ_NOPREFIX:
		errmsg = prefixdeleteaddr(rp->prefixname, &rp->prefix);
		if (errmsg != NULL) {
			lg(LOG_ERR, "junoscriptprefix: noprefix %s", errmsg);
			clientsenderr(cl, rp, "%s", errmsg);
			return;
		}
		break;

	case REQ_LISTPREFIX:
		prefixlist(cl, rp);
		return;

	default:
		abort();
	}

	/* Start building our junos request */
	jp = &jrequest;
	memset(jp, 0, sizeof(*jp));
	jp->type = rp->type;
	jp->jtype = JREQ_PREFIX;
	strlcpy(jp->addr, addr2str(&rp->prefix), sizeof(jp->addr));
	snprintf(jp->prefix, sizeof(jp->prefix), "%s%s",
	    acldprefix, rp->prefixname);
	errmsg = jreq_add(jp);
	if (errmsg != NULL) {
		fmt = "junoscriptprefix: internal error: %s";
		lg(LOG_ERR, fmt, errmsg);
		clientsenderr(cl, rp, fmt, errmsg);
		return;
	}

	/* Update request state */
	rp->state = RSTATE_CHILD;

	/* Brand request as a multiball */
	++rp->multiball;
}

static void
junoscriptquerynullzero(struct client *cl, struct req *rp)
{
	clientsenderr(cl, rp, "junoscript doesn't support querynullzero");
}

/*
 * Read from a socket and append to an iobuf
 *
 * Return 1 on success, -1 when the caller should report the errno.
 * Return 0 on EOF and any other other error fatal to the child.
 */
int
junoscriptread(int fd, struct iobuf *ip)
{
	int rc, cc;
	char *cp, *errmsg;
	struct array *dp;

	dp = &ip->io_array;
	/* Loop until there's room for the EOS plus at least one new char */
	do {
		/* Call libssh2_channel_read() with at least inclen bytes */
		DYNARRAY(dp, ip->buf, dp->len + dp->inclen + 1,
		    "junoscriptread");
		cp = ip->buf + IOBUF_LEN(ip);
		cc = IOBUF_SIZE(ip) - IOBUF_LEN(ip) - 1;
		rc = libssh2_channel_read(channel, cp, cc);
		if (rc < 0) {
			if (rc == LIBSSH2_ERROR_EAGAIN)
				return (1);
			if (rc == LIBSSH2_ERROR_CHANNEL_CLOSED)
				return (0);
			errmsg = NULL;
			(void)libssh2_session_last_error(session, &errmsg,
			    NULL, 0);
			lg(LOG_ERR, "libssh2_channel_read: %s", errmsg);
			errno = EIO;
			return (0);
		}
		if (rc > 0) {
			IOBUF_LEN(ip) += rc;
			ip->buf[IOBUF_LEN(ip)] = '\0';
			junoscriptlog("recv", cp);
		}
	} while (rc == cc);
	return (1);
}

static int
junoscriptready(void)
{
	return (!sentconfig && !sentcommit);
}

static int filterexists(struct aclgroup *, struct acl *);

/* Returns true if a matching firewall filter and term exists on the router */
static int
filterexists(struct aclgroup *gp, struct acl *al)
{
	int i;
	struct acl *al2;
	struct acllist *ap;
	struct array *dp;

	ap = aclfindlist(gp, al);
	if (ap == NULL)
		return (0);
	/*
	 * Look a term in the acllist with the same sequence number.
	 * If we find it, we know the rotuer already has the necessary
	 * firewall filter term.
	 * Finding a term with the same sequence number means the router
	 */
	dp = &ap->acl_array;
	for (i = 0, al2 = ap->acl; i < dp->len; ++i, ++al2)
		if (al2->seq == al->seq)
			return (1);
	return (0);
}

static void
junoscriptrequest(struct aclgroup *gp, struct req *rp)
{
	int n, i, doadd, iproto;
	const char *p, *p2, *errmsg;
	struct acl *al, *al2;
	struct array *dp;
	struct acllist *ap;
	struct jrequest jrequest;
	struct jrequest *jp;

	/* Start building our junos request */
	jp = &jrequest;
	memset(jp, 0, sizeof(*jp));

	jp->type = rp->type;
	jp->jtype = JREQ_PREFIX;
	al = &rp->acl;
	if (al->addr1.family == AF_INET || al->addr1.family == AF_INET6)
		strlcpy(jp->addr, addr2str(&al->addr1), sizeof(jp->addr));
	if (al->port1 != NULL)
		strlcpy(jp->port, al->port1, sizeof(jp->port));

	/* Map the ACL group name to the filter name () */
	if (gp != NULL) {
		if (gp->ismac) {
			strlcpy(jp->family, "bridge", sizeof(jp->family));
		} else {
			p = junoscriptmapaclname(gp->name, al);
			strlcpy(jp->filter, p, sizeof(jp->filter));

			/* The prefix name defaults to the filter name */
			strlcpy(jp->prefix, jp->filter, sizeof(jp->prefix));

			/* Family string for templates */
			switch (ACLGROUP_FAMILY(gp)) {

			case AF_INET:
				p = "inet";
				p2 = "protocol";
				break;

			case AF_INET6:
				p = "inet6";
				p2 = havempc ?
				    "payload-protocol" : "next-header";
				break;

			default:
				abort();
			}
			strlcpy(jp->family, p, sizeof(jp->family));
			strlcpy(jp->protokey, p2, sizeof(jp->protokey));
		}
	}

// lg(LOG_DEBUG, "junoscriptrequest: \"%s\"", jp->addr);

	/* Handle per-request type setup */
	doadd = 0;
	switch (rp->type) {

	case REQ_DROP:
		++doadd;
		/* fall through */

	case REQ_RESTORE:
		/* Use the address family as the ACL seq number */
		al->seq = al->addr1.family;
		(void)snprintf(jp->seq, sizeof(jp->seq), "%d", al->seq);
		break;

	case REQ_BLOCKMAC:
		++doadd;
		/* fall through */

	case REQ_RESTOREMAC:
		jp->mac = al->mac;
		jp->jtype = JREQ_MACFILTER;
		break;

	case REQ_DROPUDPPORT:
	case REQ_DROPTCPPORT:
	case REQ_DROPTCPSYNPORT:
		++doadd;
		/* fall through */

	case REQ_RESTORETCPPORT:
	case REQ_RESTOREUDPPORT:
	case REQ_RESTORETCPSYNPORT:
		switch (rp->type) {

		case REQ_DROPTCPPORT:
		case REQ_RESTORETCPPORT:
			p = "tcp";
			iproto = IPPROTO_TCP;
			break;

		case REQ_DROPUDPPORT:
		case REQ_RESTOREUDPPORT:
			p = "udp";
			iproto = IPPROTO_UDP;
			break;

		case REQ_DROPTCPSYNPORT:
		case REQ_RESTORETCPSYNPORT:
			p = "tcp";
			iproto = (TH_SYN * 100000) + IPPROTO_TCP;
			break;

		default:
			abort();
		}
		strlcpy(jp->proto, p, sizeof(jp->proto));

		/* Base the ACL seq on the protocol number */
		al->seq = iproto;
		(void)snprintf(jp->seq, sizeof(jp->seq), "%d", al->seq);

		/* Delete the filter term when removing the last port */
		ap = aclfindlist(gp, al);
		n = 0;
		dp = &ap->acl_array;
		for (i = 0, al2 = ap->acl; i < dp->len; ++i, ++al2)
			if (al->type == al2->type)
				++n;
		if (n == 0)
			++jp->doremoveterm;
		break;

	case REQ_PERMITUDPDSTHOSTPORT:
	case REQ_PERMITTCPDSTHOSTPORT:
		++doadd;
		/* fall through */

	case REQ_UNPERMITTCPDSTHOSTPORT:
	case REQ_UNPERMITUDPDSTHOSTPORT:
		switch (rp->type) {

		case REQ_PERMITTCPDSTHOSTPORT:
		case REQ_UNPERMITTCPDSTHOSTPORT:
			p = "tcp";
			p2 = "TCP";
			iproto = IPPROTO_TCP;
			break;

		case REQ_PERMITUDPDSTHOSTPORT:
		case REQ_UNPERMITUDPDSTHOSTPORT:
			p = "udp";
			p2 = "UDP";
			iproto = IPPROTO_UDP;
			break;

		default:
			abort();
		}
		strlcpy(jp->proto, p, sizeof(jp->proto));

		/* Use the protocol+port as the ACL seq number */
		al->seq = (iproto * 100000) + atoi(jp->port);
		(void)snprintf(jp->seq, sizeof(jp->seq), "%d", al->seq);

		/* Build the prefix list name */
		(void)snprintf(jp->prefix, sizeof(jp->prefix), "%s-%s-%s%s",
		    gp->name, p2, suffix_hostport, jp->port);

		break;

	default:
		abort();
	}

	switch (al->type) {

	case ATYPE_PERMITUDPPORT:
	case ATYPE_BLOCKUDPPORT:
	case ATYPE_PERMITTCPPORT:
	case ATYPE_BLOCKTCPPORT:
	case ATYPE_PERMITTCPSYNPORT:
	case ATYPE_BLOCKTCPSYNPORT:
		/* We always update the filter when making group changes */
		++jp->dofirewall;
		break;

	case ATYPE_BLOCKMAC:
		break;

	default:
		/*
		 * Add the firewall filter config?
		 * Do this before updating the sequence number
		 * (otherwise the test is always true).
		 */
		if (doadd && !filterexists(gp, al))
			++jp->dofirewall;
		break;
	}

	/* When adding, fix the bogus seq number clientdroprestore() picked */
	if (doadd && gp != NULL) {
		al2 = aclfindacl(gp, al, 0);
		if (al2 != NULL)
			al2->seq = al->seq;
		else
			lg(LOG_ERR,
			    "junoscriptrequest: Can't find ACL to update seq");
	}

	errmsg = jreq_add(jp);
	if (errmsg != NULL) {
		lg(LOG_ERR, "junoscriptrequest: internal error: %s", errmsg);
		abort();
	}

	/* Update request state */
	rp->state = RSTATE_CHILD;

	/* Brand request as a multiball */
	++rp->multiball;
}

static void
junoscriptsend(const char *fmt, ...)
{
	char *cp;
	size_t n;
	va_list ap;
	struct iobuf *ip;
	struct iobuf iobuf;
	char buf[1024];

	va_start(ap, fmt);
	(void)vsnprintf(buf, sizeof(buf) - 1, fmt, ap);
	va_end(ap);

	/* Always append a newline */
	n = strlen(buf);
	cp = buf + n - 1;
	if (cp >= buf) {
		*++cp = '\n';
		*++cp = '\0';
		++n;
	}

	/* Send the line */
	ip = &iobuf;
	memset(ip, 0, sizeof(*ip));
	ip->buf = buf;
	IOBUF_LEN(ip) = n;
	junoscriptwrite(ip);
}

static void
junoscriptsendattr(void)
{
	++state.sentattrs;
}

static void
junoscriptserverayt(void)
{
	junoscriptsend(js_uptime_cmd);
}

static void
junoscriptservercompact(void)
{
	lg(LOG_DEBUG, "junoscriptservercompact: not implemented");
}

static void
junoscriptserversync(void)
{
	lg(LOG_DEBUG, "junoscriptserversync: noop");
}

static void
junoscriptwrite(struct iobuf *ip)
{
	int rc, sentlen;
	char *cp, *errmsg;
	size_t n;

	n = IOBUF_LEN(ip);
	cp = ip->buf;
	sentlen = 0;
	while (n > 0) {
		rc = libssh2_channel_write(channel, cp, n);
		if (rc < 0) {
			errmsg = NULL;
			libssh2_session_last_error(session, &errmsg, NULL, 0);
			lg(LOG_ERR, "libssh2_channel_write: %s", errmsg);
			if (rc == LIBSSH2_ERROR_EAGAIN)
				continue;
			if (rc == LIBSSH2_ERROR_SOCKET_SEND)
				continue;
			/* XXX kill off child on fatal errors? */
			break;
		}

		if (libssh2_channel_eof(channel)) {
			/* XXX kill off child on fatal errors? */
			lg(LOG_ERR, "junoscriptwrite: EOF");
			break;
		}

		sentlen += rc;
		n -= rc;
		cp += rc;
	}

	/* Log the line(s) (but only what we successfully sent) */
	ip->buf[sentlen] = '\0';
	junoscriptlog("send", ip->buf);
	IOBUF_LEN(ip) = 0;
}
