/*
 * Copyright (c) 2021, 2023
 * The Regents of the University of California. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *     * Redistributions of source code must retain the above copyright
 *       notice, this list of conditions and the following disclaimer.
 *     * Redistributions in binary form must reproduce the above copyright
 *       notice, this list of conditions and the following disclaimer in the
 *       documentation and/or other materials provided with the distribution.
 *     * Neither the name of the University nor the names of its contributors
 *       may be used to endorse or promote products derived from this software
 *       without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#ifndef lint
static const char rcsid[] __attribute__ ((unused)) =
    "@(#) $Id$ (LBL)";
#endif

#include <sys/types.h>

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "acld.h"

/* Locals */
static const char invalid[] = "invalid mac address";

/* Extract an ethernet mac address */
const char *
extractmac(const char *s, struct mac *ma)
{
	int i;
	u_long uv;
	char *ep;
	u_char *bp;

	bp = ma->addr;

	/* Colon deliminated */
	if (strchr(s, ':') != NULL) {
		for (i = 6; i > 0 && *s != '\0'; --i) {
			uv = strtoul(s, &ep, 16);
			if  (uv > 0xff || (*ep != ':' && *ep != '\0'))
				return (invalid);
			*bp++ = (u_char)uv;
			s = ep;
			if (*s == ':')
				++s;
		}
		if (*s != '\0')
			return (invalid);
		return (NULL);
	}

	/* Dot deliminated (Cisco style) */
	if (strchr(s, '.') != NULL) {
		for (i = 3; i > 0 && *s != '\0'; --i) {
			uv = strtoul(s, &ep, 16);
			if  (uv > 0xffff || (*ep != '.' && *ep != '\0'))
				return (invalid);
			*bp++ = (u_char)((uv >> 8) & 0xff);
			*bp++ = (u_char)(uv & 0xff);
			s = ep;
			if (*s == ':')
				++s;
		}
		if (*s != '\0')
			return (invalid);
		return (NULL);
	}

	return ("invalid mac address");
}

const char *
mac2str(struct mac *ma)
{
	u_char *bp;
	static char buf[sizeof("00:00:00:00:00:00")];

	bp = ma->addr;
	(void)snprintf(buf, sizeof(buf), "%02x:%02x:%02x:%02x:%02x:%02x",
	    bp[0], bp[1], bp[2], bp[3], bp[4], bp[5]);
	return (buf);
}

int
maccmp(const void *arg1, const void *arg2)
{
	const struct mac *ma1, *ma2;

	ma1 = (const struct mac *)arg1;
	ma2 = (const struct mac *)arg2;

	return (memcmp(ma1, ma2, sizeof(*ma1)));
}
