/*
 * Copyright (c) 2018, 2019, 2020
 * The Regents of the University of California. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *     * Redistributions of source code must retain the above copyright
 *       notice, this list of conditions and the following disclaimer.
 *     * Redistributions in binary form must reproduce the above copyright
 *       notice, this list of conditions and the following disclaimer in the
 *       documentation and/or other materials provided with the distribution.
 *     * Neither the name of the University nor the names of its contributors
 *       may be used to endorse or promote products derived from this software
 *       without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#ifndef lint
static const char rcsid[] __attribute__ ((unused)) =
    "@(#) $Id$ (LBL)";
#endif

#include <sys/types.h>
#include <sys/time.h>

#include <arpa/inet.h>

#include <ctype.h>
#include <errno.h>
#ifdef HAVE_JUNOSCRIPT
#include <netdb.h>
#endif
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sysexits.h>

#include "acld.h"
#include "prefix.h"

struct prefixlist {
	const char *name;
	struct addr *addr;
	struct array addr_array;
};

/* Globals */

/* Locals */
static struct array prefixlists_array = {
	sizeof(struct prefixlist),			/* osize */
	1,						/* inclen */
	0,						/* len */
	0,						/* size */
};
static struct prefixlist *prefixlists;

/* Forwards */
static struct prefixlist *prefixfind(const char *);
static int prefixlistcmp(const void *, const void *);

/* XXX should just create the preflixlist at config reading time */
int
prefixadd(const char *str)
{
	int i, an, ret;
	char **prefixes, **av;
	struct array *dp;

	an = makeargv(str, &av);
	if (an < 1) {
		lg(LOG_ERR, "prefix missing args");
		freeargv(av);
		return (0);
	}

	dp = &cf->c_prefixes_array;
	DYNARRAY(dp, cf->c_prefixes, dp->len + an, "cf prefixes");

	ret = 1;
	prefixes = cf->c_prefixes + dp->len;
	for (i = 0; i < an; ++i, ++prefixes) {
		if (prefixexists(av[i])) {
			lg(LOG_ERR, "duplicate prefix: %s", av[i]);
			ret = 0;
			continue;
		}
		*prefixes = strsave(av[i]);
	}

	dp->len += an;
	freeargv(av);
	return (ret);
}

const char *
prefixaddaddr(const char *name, struct addr *a)
{
	int i;
	struct addr *a2;
	struct array *dp;
	struct prefixlist *plp;
	static char buf[128];

	plp = prefixfind(name);
	if (plp == NULL) {
		if (!prefixexists(name)) {
			snprintf(buf, sizeof(buf),
			    "unconfigured prefix list \"%s\"", name);
			return (buf);
		}
		dp = &prefixlists_array;
		DYNARRAY(dp, prefixlists, dp->len + 1, "prefixaddaddr");
		plp = prefixlists + dp->len;
		++dp->len;
		plp->name = strsave(name);

		dp = &plp->addr_array;
		dp->osize = sizeof(*plp->addr);
		dp->inclen = 32;
	}
	dp = &plp->addr_array;
	DYNARRAY(dp, plp->addr, dp->len + 1, "prefixaddaddr (addr)");

	for (i = 0, a2 = plp->addr; i < dp->len; ++i, ++a2) {
		if (addrcmp(a2, a) == 0) {
			snprintf(buf, sizeof(buf),
			    "%s is already a member of %s",
			    addr2str(a), plp->name);
			return (buf);
		}
	}

	a2 = plp->addr + dp->len;
	*a2 = *a;
	++dp->len;

	/* Keep everything sorted */
	qsort(plp->addr, dp->len, sizeof(*plp->addr), addrcmp);
	dp = &prefixlists_array;
	qsort(prefixlists, dp->len, sizeof(*plp), prefixlistcmp);
	return (NULL);
}

const char *
prefixdeleteaddr(const char *name, struct addr *a)
{
	int i, n;
	struct addr *a2;
	struct array *dp;
	struct prefixlist *plp;
	static char buf[132];

	plp = prefixfind(name);
	if (plp == NULL)
		return (0);

	dp = &plp->addr_array;
	for (i = 0, a2 = plp->addr; i < dp->len; ++i, ++a2) {
		if (addrcmp(a2, a) == 0) {
			n = dp->len - i - 1;
			if (n > 0)
				memmove(a2, a2 + 1, n * sizeof(*a2));
			--dp->len;
			return (NULL);
		}
	}

	snprintf(buf, sizeof(buf), "%s is not a member of %s",
	    addr2str(a), plp->name);
	return (buf);
}

/* Return 1 if the prefix list is valid */
int
prefixexists(const char *name)
{
	int i;
	struct array *dp;
	char **prefixes;

	dp = &cf->c_prefixes_array;
	prefixes = cf->c_prefixes;
	for (i = 0; i < dp->len; ++i, ++prefixes)
		if (strcmp(name, *prefixes) == 0)
			return (1);
	return (0);
}

static struct prefixlist *
prefixfind(const char *name)
{
	int i;
	struct array *dp;
	struct prefixlist *plp;

	dp = &prefixlists_array;
	for (i = 0, plp = prefixlists; i < dp->len; ++i, ++plp)
		if (strcmp(plp->name, name) == 0)
			return (plp);
	return (NULL);
}

void
prefixlist(struct client *cl, struct req *rp)
{
	int i;
	struct addr *a;
	struct prefixlist *plp;
	struct array *dp;

	if (cf->c_prefixes_array.len == 0) {
		clientsenderr(cl, rp,
		    "prefix not supported by this configuration");
		return;
	}

	plp = prefixfind(rp->prefixname);
	if (plp == NULL) {
		/* Send empty response if prefix is configured but empty */
		if (prefixexists(rp->prefixname))
			clientsend(cl, rp);
		else
			clientsenderr(cl, rp, "prefix %s not found",
			    rp->prefixname);
		return;
	}

	dp = &plp->addr_array;
	for (i = 0, a = plp->addr; i < dp->len; ++i, ++a)
		ioappendfmt(&rp->payload, "%s", addr2str(a));
	rp->flags |= RFLAG_CONTINUE;
	clientsend(cl, rp);
}

static int
prefixlistcmp(const void *arg1, const void *arg2)
{
	struct prefixlist *plp1, *plp2;

	plp1 = (struct prefixlist *)arg1;
	plp2 = (struct prefixlist *)arg2;
	return (strcmp(plp1->name, plp2->name));
}

void
prefixstate(struct iobuf *ip)
{
	int i;
	char **prefixes;
	struct array *dp;
	struct prefixlist *plp;

	dp = &cf->c_prefixes_array;
	for (i = 0, prefixes = cf->c_prefixes; i < dp->len; ++i, ++prefixes)
		ioappendfmt(ip, "prefix %s", *prefixes);

	dp = &prefixlists_array;
	for (i = 0, plp = prefixlists; i < dp->len; ++i, ++plp)
		ioappendfmt(ip, "# prefixlen for %s is %lu",
		    plp->name, plp->addr_array.len);
}
