/*
 * Copyright (c) 2013, 2014, 2015, 2016, 2018, 2019, 2020, 2024
 * The Regents of the University of California. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *     * Redistributions of source code must retain the above copyright
 *       notice, this list of conditions and the following disclaimer.
 *     * Redistributions in binary form must reproduce the above copyright
 *       notice, this list of conditions and the following disclaimer in the
 *       documentation and/or other materials provided with the distribution.
 *     * Neither the name of the University nor the names of its contributors
 *       may be used to endorse or promote products derived from this software
 *       without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#ifndef lint
static const char rcsid[] __attribute__ ((unused)) =
    "@(#) $Id$ (LBL)";
#endif

#include <sys/types.h>

#include <errno.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sysexits.h>
#include <time.h>
#include <unistd.h>

#include "acld.h"

/* Locals */
static u_int32_t comment_recookie;
static int comment_readresponse;		/* true when reading comment */

/* Forwards */
static struct rereq *replicantfindrereq( u_int32_t);

void
replicantappendreq(struct rereq *rep, struct rereq **repp)
{
	struct rereq *rep2;

	if (*repp == NULL)
		*repp = rep;
	else {
		rep2 = *repp;
		while (rep2->next != NULL)
			rep2 = rep2->next;
		rep2->next = rep;
	}
}

void replicantbump(void)
{
	struct client *cl;
	time_t delta;

	for (cl = state.clients; cl != NULL; cl = cl->next) {
		if (cl->replicant)
			continue;
		delta = time(NULL) - cl->connect_ts;
		if (delta > PERSISTENT_CLIENT_SECS) {
			lg(LOG_INFO,
			    "replicantbump: kill client #%d (%ld secs)",
			    cl->n, delta);
			if (IOBUF_LEN(&cl->wbuf) > 0)
				++cl->close;
			else
				freeclient(cl);
		}
	}
}

void
replicantdone(struct rereq *rep)
{
	struct client *cl;
	struct req *rp;
	struct timeval delta;
	struct state *sp;

	/* Mark the replicant request as done */
	sp = &state;
	rep->state = RESTATE_DONE;

	/* We're done if this is a server request */
	if (rep->type == REQ_UNKNOWN || rep->type == REQ_REPLICANT)
		return;

	/* We're also done if the local request hasn't completed yet */
	if (rep->req != NULL && rep->req->state != RSTATE_DONE)
		return;

	/* Find the client and request */
	for (cl = sp->clients; cl != NULL; cl = cl->next)
		for (rp = cl->req; rp != NULL; rp = rp->next)
			if (rp->rereq == rep)
				goto bail;

	lg(LOG_ERR, "replicantdone: Can't find matching request");
	// exit(EX_SOFTWARE);
	return;

bail:
	/* Break tie with replicant */
	// if replicant already blocked it (ignore)
	// and we just blocked it (no ignore)
	// and ...?
	if ((rep->reflags & (REFLAG_FAILED | REFLAG_IGNORE)) == REFLAG_IGNORE &&
	    (rp->flags & (RFLAG_FAILED | RFLAG_IGNORE)) == 0 &&
	    (rep->cts.tv_sec != 0 || rep->cts.tv_usec != 0) &&
	    (rp->cts.tv_sec != 0 || rp->cts.tv_usec != 0)) {
		/* Positive delta means were first */
		timersub(&rep->cts, &rp->cts, &delta);
		lg(LOG_DEBUG,
		    "replicantdone: local %ld.%06ld,"
		    " replicant %ld.%06ld (delta %ld.%06ld)",
		    rp->cts.tv_sec, rp->cts.tv_usec,
		    rep->cts.tv_sec, rep->cts.tv_usec,
		    delta.tv_sec, delta.tv_usec);
	}

	/* Attempt to complete this client request */
	clientsend(cl, rp);
}

void
replicantfreerereq(struct rereq *orep)
{
	struct rereq *rep, **repp;
	struct state *sp;

	sp = &state;
	if (orep->aclname != NULL) {
		free(orep->aclname);
		orep->aclname = NULL;
	}
	if (orep->prefixname != NULL) {
		free(orep->prefixname);
		orep->prefixname = NULL;
	}
	if (orep->comment != NULL) {
		free(orep->comment);
		orep->comment = NULL;
	}
	if (orep->response != NULL) {
		free(orep->response);
		orep->response = NULL;
	}

	/* Unlink this replicant request */
	repp = &sp->new_rereq;
	rep = sp->new_rereq;
	while (rep != NULL) {
		if (rep == orep) {
			*repp = rep->next;
			break;
		}
		repp = &rep->next;
		rep = rep->next;
	}
	repp = &sp->sent_rereq;
	rep = sp->sent_rereq;
	while (rep != NULL) {
		if (rep == orep) {
			*repp = rep->next;
			break;
		}
		repp = &rep->next;
		rep = rep->next;
	}
	free(orep);
}

void
replicantinit(void)
{
	struct state *sp;

	sp = &state;
	sp->re_secs = 1;
	if (cf->c_portreplicant > 0)
		timerset(&sp->t_re_connect, sp->re_secs);
	else
		timerreset(&sp->t_re_connect);
	timerreset(&sp->t_re_timeout);
}

static struct rereq *
replicantfindrereq( u_int32_t cookie)
{
	struct rereq *rep;

	for (rep = state.sent_rereq; rep != NULL; rep = rep->next)
		if (rep->cookie == cookie)
			return (rep);
	return (NULL);
}

void
replicantinput(void)
{
	int an;
	u_int32_t cookie;
	char *cp, *cp2, *ep, **av;
	struct rereq *rep;
	struct iobuf *ip;
	struct state *sp;

	sp = &state;
	ip = &sp->re_rbuf;

	while (iohaveline(ip)) {
		rep = NULL;
		if (comment_readresponse) {
			rep = replicantfindrereq(comment_recookie);
			if (rep == NULL) {
				lg(LOG_ERR, "replicantinput: "
				    "Can't find rereq for recookie %d",
				    comment_recookie);
				    comment_readresponse = 0;
			}
		}

		/* Collect response lines */
		if (rep != NULL && rep->state == RESTATE_READRESPONSE) {
			if (!iocomment(&rep->response, ip)) {
				lg(LOG_DEBUG, "replicant response \"%s\"",
				    pretty(rep->response));

				/* Keep track if replicant did anything */
				if (strstr(rep->response, "already") != NULL)
					rep->reflags |= REFLAG_IGNORE;

				/* Recover (and hide) optional timestamp */
				cp = strstr(rep->response, " @ ");
				if (cp != NULL) {
					extracttimeval(&rep->cts, cp + 3);
					/* Hide this from ioconsolidate() */
					cp2 = strchr(cp, '\r');
					if (cp2 != NULL)
						memmove(cp, cp2,
						    strlen(cp2) + 1);
				}

				/* Try to complete this request */
				replicantdone(rep);

				/* Unlink */
				--sp->replicantdepth;
				if (sp->replicantdepth <= 0)
					timerreset(&sp->t_re_timeout);
				if (rep->req == NULL)
					replicantfreerereq(rep);
				comment_readresponse = 0;
			}
			continue;
		}

		cp = iogetstr(ip);
		if (cp == NULL)
			break;

		/* Break into arguments */
		trimws(cp, strlen(cp));
		av = NULL;
		an = makeargv(cp, &av);
		if (an == 0) {
			lg(LOG_DEBUG, "replicantinput: blank input line");
			freeargv(av);
			continue;
		}
		if (an < 3 || an > 4) {
			lg(LOG_DEBUG, "replicantinput: garbage: \"%s\"",
			    pretty(cp));
			freeargv(av);
			continue;
		}

		/* Parse cookie */
		cookie = strtoul(av[1], &ep, 10);
		if (*ep != '\0') {
			lg(LOG_ERR, "replicantinput: non-numeric cookie: %s",
			    pretty(cp));
			freeargv(av);
			continue;
		}

		/* Find rereq that matches this cookie */
		rep = replicantfindrereq(cookie);
		if (rep == NULL) {
			lg(LOG_ERR,
			    "replicantinput: Can't find rereq for cookie %d",
				cookie);
			freeargv(av);
			continue;
		}

		if (rep->req != NULL && failed(rep->req->cmd, av[2]))
			rep->reflags |= REFLAG_FAILED;

		if (an == 4) {
			if (strcmp(av[an - 1], "-") != 0) {
				lg(LOG_ERR,
				    "replicantinput: garbage 2: %s",
				    pretty(cp));
				freeargv(av);
				continue;
			}
			rep->state = RESTATE_READRESPONSE;
			comment_recookie = cookie;
			++comment_readresponse;
			freeargv(av);
			continue;
		}

		/* Try to complete this request */
		replicantdone(rep);
		--sp->replicantdepth;
		if (sp->replicantdepth <= 0)
			timerreset(&sp->t_re_timeout);
		if (rep->req == NULL)
			replicantfreerereq(rep);
		freeargv(av);
	}
}

void
replicantlogin(void)
{
	int s;
	u_short port;
	socklen_t salen;
	struct rereq *rep, *rep2;
	struct addr *addr;
	struct sockaddr_storage sa;
	struct sockaddr *sap;
	struct state *sp;
	struct timeval tv;
	fd_set readfds, writefds;
	char buf[64];

	sp = &state;
	addr = &cf->c_replicantaddr;
	port = cf->c_portreplicant;
	sap = (struct sockaddr *)&sa;
	salen = addr2sa(addr, port, sap);
	(void)snprintf(buf, sizeof(buf), "%s.%hu", addr2str(addr), port);

	s = socket(addr->family, SOCK_STREAM, 0);
	if (s < 0) {
		lg(LOG_ERR, "socket: %s: %s", buf, strerror(errno));
		exit(EX_OSERR);
	}

	/* Switch to non-blocking I/O before attempting to connect */
	if (blocking(s, 0) < 0)
		exit(EX_OSERR);
	if (connect(s, sap, salen) < 0 && errno != EINPROGRESS) {
		lg(LOG_ERR, "connect: %s: %s", buf, strerror(errno));
		(void)close(s);
		s = -1;
	} else {
		/* Give the connect a few seconds */
		tv.tv_sec = 3;
		tv.tv_usec = 0;
		FD_ZERO(&readfds);
		FD_SET(s, &readfds);
		FD_ZERO(&writefds);
		FD_SET(s, &writefds);
		if (select(s + 1, &readfds, &writefds, NULL, &tv) == 0) {
			lg(LOG_ERR, "connect: %s: timeout", buf);
			(void)close(s);
			s = -1;
		}
	}

	/* Setup the retry if we failed */
	if (s < 0) {
		sp->re_secs = sp->re_secs * 2;
		if (sp->re_secs > 8)
			sp->re_secs = 8;
		timerset(&sp->t_re_connect, sp->re_secs);
		timerreset(&sp->t_re_timeout);
		return;
	}

	sp->refd = s;
	sp->re_secs = 1;
	timerreset(&sp->t_re_connect);
	lg(LOG_INFO, "replicantlogin: connect");

	/* Toss stale I/O */
	iofree(&sp->re_rbuf);
	iofree(&sp->re_wbuf);

	/* Move sent replicant requests to front of new requests queue */
	if (sp->sent_rereq != NULL) {
		replicantappendreq(sp->new_rereq, &sp->sent_rereq);
		sp->new_rereq = sp->sent_rereq;
		sp->sent_rereq = NULL;
	}

	/* Remove stale acld and replicant requests */
	rep = sp->new_rereq;
	while (rep != NULL) {
		if (strcmp(rep->cmd, "acld") != 0 &&
		    strcmp(rep->cmd, "replicant") != 0)
			rep2 = NULL;
		else
			rep2 = rep;
		rep = rep->next;
		if (rep2 != NULL) {
			rep2->next = NULL;
			replicantfreerereq(rep2);
		}
	}

	/* Fake acld request to match greeting */
	rep = new(1, sizeof(*rep), "replicantlogin: acld");
	rep->cmd = "acld";
	rep->type = REQ_UNKNOWN;
	rep->state = RESTATE_PENDING;
	rep->cookie = 0;		/* acld initial greeting cookie */
	rep->next = sp->sent_rereq;
	sp->sent_rereq = rep;
	++sp->replicantdepth;

	/* Identify ourselves before sending stale requests */
	rep = new(1, sizeof(*rep), "replicantlogin: replicant");
	rep->cmd = "replicant";
	rep->type = REQ_REPLICANT;
	rep->state = RESTATE_PENDING;
	rep->next = sp->new_rereq;
	sp->new_rereq = rep;
}

void
replicantprocess(void)
{
	struct rereq *rep, **repp;
	struct state *sp;

	/* Send as many new requests as we can */
	sp = &state;
	repp = &sp->new_rereq;
	rep = sp->new_rereq;
	while (rep != NULL && sp->replicantdepth <= REPLICANT_DEPTH) {
		replicantsend(rep);
		repp = &rep->next;
		rep = rep->next;
	}

	/* Move the newly sent replicant requests to end of sent queue */
	*repp = NULL;
	replicantappendreq(sp->new_rereq, &sp->sent_rereq);
	sp->new_rereq = rep;
}

void
replicantreqhook(struct req *rp)
{
	struct rereq *rep;
	struct state *sp;

	/* Check for interest request types */
	sp = &state;
	switch (rp->type) {

	case REQ_DROP:
	case REQ_DROPUDPPORT:
	case REQ_DROPTCPPORT:
	case REQ_DROPTCPSYNPORT:
	case REQ_PERMITUDPDSTHOSTPORT:
	case REQ_PERMITTCPDSTHOSTPORT:
	case REQ_RESTORE:
	case REQ_RESTOREUDPPORT:
	case REQ_RESTORETCPPORT:
	case REQ_RESTORETCPSYNPORT:
	case REQ_UNPERMITUDPDSTHOSTPORT:
	case REQ_UNPERMITTCPDSTHOSTPORT:
	case REQ_QUERY:
		if (!cf->c_replicate_acl)
			return;
		break;

	case REQ_NULLZERO:
	case REQ_NONULLZERO:
	case REQ_QUERYNULLZERO:
		if (!cf->c_replicate_nullzero)
			return;
		break;

	case REQ_PREFIX:
	case REQ_NOPREFIX:
		if (!cf->c_replicate_prefix)
			return;
		break;

	case REQ_ADDWHITELIST:
	case REQ_REMWHITELIST:
	case REQ_QUERYWHITELIST:
		if (!cf->c_replicate_whitelist)
			return;
		break;

	default:
		return;
	}

	rep = new(1, sizeof(*rep), "replicantreqhook: rereq");
	rep->type = rp->type;
	rep->state = RESTATE_PENDING;
	rep->cmd = rp->cmd;
	if (rp->aclname != NULL)
		rep->aclname = strsave(rp->aclname);
	if (rp->prefixname != NULL)
		rep->prefixname = strsave(rp->prefixname);
	if (rp->comment != NULL)
		rep->comment = strsave(rp->comment);

	switch (rp->type) {

	case REQ_DROP:
	case REQ_DROPUDPPORT:
	case REQ_DROPTCPPORT:
	case REQ_DROPTCPSYNPORT:
	case REQ_PERMITUDPDSTHOSTPORT:
	case REQ_PERMITTCPDSTHOSTPORT:
	case REQ_RESTORE:
	case REQ_RESTOREUDPPORT:
	case REQ_RESTORETCPPORT:
	case REQ_RESTORETCPSYNPORT:
	case REQ_UNPERMITUDPDSTHOSTPORT:
	case REQ_UNPERMITTCPDSTHOSTPORT:
	case REQ_QUERY:
		rep->acl = rp->acl;
		break;

	case REQ_NULLZERO:
	case REQ_NONULLZERO:
	case REQ_QUERYNULLZERO:
		rep->nullzero = rp->nullzero;
		break;

	case REQ_PREFIX:
	case REQ_NOPREFIX:
		rep->prefix = rp->prefix;
		rep->prefixname = strsave(rp->prefixname);
		break;

	case REQ_ADDWHITELIST:
	case REQ_REMWHITELIST:
	case REQ_QUERYWHITELIST:
		rep->whitelist = rp->whitelist;
		break;

	default:
		abort();
	}

	/* Add cross-links */
	rp->rereq = rep;
	rep->req = rp;

	/* It's an immediate fail if the replicant isn't connected */
	if (sp->refd < 0) {
		rep->response = strsave("replicant not connected\r\n");
		rep->reflags |= REFLAG_FAILED;

		/* Try to complete this request */
		replicantdone(rep);
		return;
	}

	/* Add new request to end of new replicant request queue */
	replicantappendreq(rep, &sp->new_rereq);
}

void
replicantsend(struct rereq *rep)
{
	const char *post, *primary;
	struct state *sp;

	sp = &state;
	if (rep->comment != NULL)
		post = " -";
	else
		post = "";

	switch (rep->type) {

	case REQ_DROP:
	case REQ_RESTORE:
	case REQ_QUERY:
		rep->cookie = ++sp->recookie;
		replicantwrite("%s %lu %s%s",
		    rep->cmd, rep->cookie, addr2str(&rep->acl.addr1), post);
		break;

	case REQ_DROPUDPPORT:
	case REQ_DROPTCPPORT:
	case REQ_RESTOREUDPPORT:
	case REQ_RESTORETCPPORT:
	case REQ_DROPTCPSYNPORT:
	case REQ_RESTORETCPSYNPORT:
		rep->cookie = ++sp->recookie;
		replicantwrite("%s %lu %s %s%s",
		    rep->cmd, rep->cookie, rep->acl.port1, rep->aclname, post);
		break;

	case REQ_PERMITUDPDSTHOSTPORT:
	case REQ_PERMITTCPDSTHOSTPORT:
	case REQ_UNPERMITUDPDSTHOSTPORT:
	case REQ_UNPERMITTCPDSTHOSTPORT:
		rep->cookie = ++sp->recookie;
		replicantwrite("%s %lu %s %s%s", rep->cmd, rep->cookie,
		    addr2str(&rep->acl.addr1), rep->acl.port1, post);
		break;

	case REQ_PREFIX:
	case REQ_NOPREFIX:
		rep->cookie = ++sp->recookie;
		replicantwrite("%s %lu %s %s%s", rep->cmd, rep->cookie,
		    addr2str(&rep->prefix), rep->prefixname,
			post);
		break;

	case REQ_NULLZERO:
	case REQ_NONULLZERO:
	case REQ_QUERYNULLZERO:
		rep->cookie = ++sp->recookie;
		replicantwrite("%s %lu %s%s",
		    rep->cmd, rep->cookie, addr2str(&rep->nullzero.dst), post);
		break;

	case REQ_ADDWHITELIST:
	case REQ_REMWHITELIST:
	case REQ_QUERYWHITELIST:
		rep->cookie = ++sp->recookie;
		replicantwrite("%s %lu %s%s",
		    rep->cmd, rep->cookie, addr2str(&rep->whitelist), post);
		break;

	case REQ_REPLICANT:
		rep->cookie = ++sp->recookie;
		if (cf->c_replicate_primary)
			primary = " primary ";
		else
			primary = "";
		replicantwrite("%s %lu%s%s",
		    rep->cmd, rep->cookie, primary, post);
		break;

	case REQ_UNKNOWN:
		/* Ignore greeting */
		return;

	default:
		lg(LOG_ERR, "replicantsend: unknown type %s",
		    val2str(cmd2req, rep->type));
		abort();
	}

	if (rep->comment != NULL) {
		replicantwrite("%s", rep->comment);
		replicantwrite(".");
	}
	rep->state = RESTATE_REPLICANT;
	++sp->replicantdepth;
	timerreset(&sp->t_re_timeout);
	timerset(&sp->t_re_timeout, cf->c_replicant_session_secs);
}

void
replicantterminate(const char *response)
{
	struct req *rp;
	struct rereq *rep;
	struct client *cl;
	struct state *sp;

	sp = &state;
	if (sp->refd >= 0) {
		(void)close(sp->refd);
		sp->refd = -1;
	}
	sp->re_secs = 1;
	timerset(&sp->t_re_connect, sp->re_secs);
	iofree(&sp->re_rbuf);
	iofree(&sp->re_wbuf);

	/* Wrap up the replicant side of pending requests */
	for (rep = sp->new_rereq; rep != NULL; rep = rep->next) {
		strappend(&rep->response, response);
		strappend(&rep->response, "\r\n");
		rep->state = RESTATE_DONE;
		rep->reflags |= REFLAG_FAILED;
		rp = rep->req;
		if (rp != NULL) {
			cl = getreqclient(rp);
			if (cl != NULL)
				clientsend(cl, rp);
		}
	}
	for (rep = sp->sent_rereq; rep != NULL; rep = rep->next) {
		strappend(&rep->response, response);
		strappend(&rep->response, "\r\n");
		rep->state = RESTATE_DONE;
		rep->reflags |= REFLAG_FAILED;
		rp = rep->req;
		if (rp != NULL) {
			cl = getreqclient(rp);
			if (cl != NULL)
				clientsend(cl, rp);
		}
	}
	sp->replicantdepth = 0;
	timerreset(&sp->t_re_timeout);
}

/* Queue up output */
void
replicantwrite(const char *fmt, ...)
{
	va_list ap;

	va_start(ap, fmt);
	ioappendvfmt(&state.re_wbuf, fmt, ap);
	va_end(ap);
}
