/*
 * Copyright (c) 2006, 2007, 2008, 2009, 2011, 2012, 2013, 2015, 2018, 2019, 2020, 2024
 * The Regents of the University of California. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *     * Redistributions of source code must retain the above copyright
 *       notice, this list of conditions and the following disclaimer.
 *     * Redistributions in binary form must reproduce the above copyright
 *       notice, this list of conditions and the following disclaimer in the
 *       documentation and/or other materials provided with the distribution.
 *     * Neither the name of the University nor the names of its contributors
 *       may be used to endorse or promote products derived from this software
 *       without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#ifndef lint
static const char rcsid[] __attribute__ ((unused)) =
    "@(#) $Id$ (LBL)";
#endif

#include <sys/types.h>
#include <sys/time.h>

#include <ctype.h>
#include <errno.h>
#include <signal.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sysexits.h>
#include <unistd.h>

#include "acld.h"
#include "cf.h"
#include "child.h"

/* Globals */
struct s2v str2route[] = {
	{ "S", ROUTE_STATIC },
	{ "D", ROUTE_DYNAMIC },
	{ "I", ROUTE_INTERFACE },
	{ "N", ROUTE_NULLZERO },
	{ "U", ROUTE_UNKNOWN },
	{ NULL,	0 }
};

/* Return true if the address is an allowable null zero candidate */
int
goodnullzero(struct addr *addrp)
{
	int i;
	struct aroute *rtp;
	struct array *dp;

	dp = &cf->c_nullzeronets_array;
	for (i = 0, rtp = cf->c_nullzeronets; i < dp->len; ++i, ++rtp)
		if (insubnet(&rtp->dst, addrp))
			return (1);

	return (0);
}

int
nullzeronetadd(const char *str)
{
	int an;
	char **av;
	const char *p;
	struct aroute *rtp;
	struct array *dp;

	an = makeargv(str, &av);
	if (an != 1) {
		lg(LOG_ERR, "nullzeronetadd wrong number of args: \"%s\"", str);
		freeargv(av);
		return (0);
	}

	dp = &cf->c_nullzeronets_array;
	DYNARRAY(dp, cf->c_nullzeronets, dp->len + 1, "cf nullzeronets");
	rtp = cf->c_nullzeronets + dp->len;

	p = extractaddr(av[0], NULL, &rtp->dst);
	if (p != NULL) {
		lg(LOG_ERR, "nullzeronetadd: extractaddr %s: %s", av[0], p);
		freeargv(av);
		return (0);
	}
	rtp->type = ROUTE_STATIC;
	++dp->len;

	freeargv(av);
	return (1);
}

void
routeadd(struct aroute *routep)
{
	struct aroute *rtp;
	struct array *dp;
	struct state *sp;

	sp = &state;
	dp = &sp->routes_array;
	DYNARRAY(dp, sp->routes, dp->len + 1, "routeadd");
	rtp = sp->routes + dp->len;
	*rtp = *routep;
	if (rtp->raw != NULL)
		rtp->raw = strsave(rtp->raw);
	++dp->len;
	if (routep->type == ROUTE_NULLZERO)
		++sp->nullzerolen;
	stats_sethiwater(&sp->nullzerostats, sp->nullzerolen);
}

/* Returns true if successful */
int
routedelete(struct aroute *routep)
{
	int i, n;
	struct aroute *rtp;
	struct array *dp;
	struct state *sp;

	sp = &state;
	dp = &sp->routes_array;
	for (i = 0, rtp = sp->routes; i < dp->len; ++i, ++rtp)
		if (memcmp(rtp, routep, sizeof(*rtp)) == 0) {
			n = dp->len - (i + 1);
			memmove(rtp, rtp + 1, n * sizeof(*rtp));
			--dp->len;
			memset(sp->routes + dp->len, 0, sizeof(*sp->routes));
			if (routep->type == ROUTE_NULLZERO) {
				--sp->nullzerolen;
				stats_setvalue(&sp->nullzerostats,
				    sp->nullzerolen);
			}
			return (1);
		}
	return (0);
}

struct aroute *
routefind(struct aroute *routep)
{
	int i;
	struct aroute *rtp;
	struct array *dp;
	struct state *sp;

	sp = &state;
	dp = &sp->routes_array;
	for (i = 0, rtp = sp->routes; i < dp->len; ++i, ++rtp)
		if (memcmp(rtp, routep, sizeof(*rtp)) == 0)
			return (rtp);
	return (NULL);
}

const char *
routeformat(struct aroute *rtp)
{
	size_t size, len;
	const char *p;
	char *cp;
	static char buf[256];

	if (rtp->type == ROUTE_UNKNOWN) {
		(void)snprintf(buf, sizeof(buf), "# unsupported route \"%s\"",
		    rtp->raw != NULL ? rtp->raw : "");
		return (buf);
	}
	p = val2str(str2route, rtp->type);
	if (p == NULL)
		p = "?";

	cp = buf;
	size = sizeof(buf);
	(void)snprintf(cp, size, "%s %s", p, addr2str(&rtp->dst));
	if (rtp->gw.family != 0) {
		len = strlen(cp);
		size -= len;
		cp += len;
		(void)snprintf(cp, size, " %s", addr2str(&rtp->gw));
	}

	return (buf);
}

void
routefree(void)
{
	/* Also clean up stats */
	stats_free(&state.nullzerostats);
}

void
routeinit(void)
{
	stats_init(&state.nullzerostats, 10, 60, "routeinit: stats");
}

void
routelistsfree(void)
{
	int i;
	struct aroute *rtp;
	struct array *dp;
	struct state *sp;

	sp = &state;
	if (sp->routes != NULL) {
		dp = &sp->routes_array;
		for (i = 0, rtp = sp->routes; i < dp->len; ++i, ++rtp)
			if (rtp->raw != NULL) {
				free(rtp->raw);
				rtp->raw = NULL;
			}
		FREEDYNARRAY(dp, sp->routes);
		sp->nullzerolen = 0;
	}
	sp->listedroutes = 0;
}

int
routestradd(const char *str)
{
	int an;
	char **av;
	const char *p;
	struct aroute *rtp;
	struct aroute route;

	/* Eat leading dot; done if there was only one */
	if (*str == '.') {
		++str;
		if (*str == '\n'|| *str == '\0')
			return (1);
	}

	/* syslog any comments we get */
	if (*str == '#') {
		++str;
		while (*str == ' ')
			++str;
		lg(LOG_INFO, "routestradd: \"%s\"", str);
		return (0);
	}

	rtp = &route;
	memset(rtp, 0, sizeof(*rtp));
	an = makeargv(str, &av);
	if (an < 2) {
		lg(LOG_ERR, "routestradd: missing args \"%s\"", str);
		freeargv(av);
		return (0);
	}

	rtp->type = str2val(str2route, av[0]);
	switch (rtp->type) {

	case ROUTE_STATIC:
	case ROUTE_DYNAMIC:
	case ROUTE_INTERFACE:
	case ROUTE_NULLZERO:
		p = extractaddr(av[1], NULL, &rtp->dst);
		if (p != NULL) {
			lg(LOG_ERR, "routestradd: extractaddr dst %s: %s",
			    av[1], p);
			exit(EX_SOFTWARE);
		}
		if (an >= 3) {
			p = extractaddr(av[2], NULL, &rtp->gw);
			if (p != NULL) {
				lg(LOG_ERR,
				    "routestradd: extractaddr gw %s: %s",
				    av[2], p);
				exit(EX_SOFTWARE);
			}
		}
		break;

	case ROUTE_UNKNOWN:
		/* Save the raw text so we can print it out later */
		p = strchr(str, '#');
		if (p == NULL)
			p = str;
		else {
			++p;
			while (isspace((int)*p))
				++p;
		}
		rtp->raw = strsave(p);
		break;

	default:
		lg(LOG_ERR, "routestradd: no such type \"%s\"", str);
		exit(EX_SOFTWARE);
	}

	freeargv(av);
	routeadd(rtp);
	return (0);
}
