/*
 * Copyright (c) 2014, 2016, 2018, 2020, 2021, 2022, 2024
 * The Regents of the University of California. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *     * Redistributions of source code must retain the above copyright
 *       notice, this list of conditions and the following disclaimer.
 *     * Redistributions in binary form must reproduce the above copyright
 *       notice, this list of conditions and the following disclaimer in the
 *       documentation and/or other materials provided with the distribution.
 *     * Neither the name of the University nor the names of its contributors
 *       may be used to endorse or promote products derived from this software
 *       without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#ifndef lint
static const char rcsid[] __attribute__ ((unused)) =
    "@(#) $Id$ (LBL)";
#endif

/* Parse Junoscript RPCs */

#include <sys/types.h>

#include <ctype.h>
#include <stdio.h>
#include <string.h>
#include <sysexits.h>

#include "acld.h"
#include "junoscript.h"
#include "prefix.h"
#include "rpc.h"

/* List of parsed prefix lists */
struct rpc_prefix {
	const char *name;
	struct addr *addr;
	struct array addr_array;
};

/* Globals */
const char acldprefix[] = "ACLD-PREFIX-";

#define IS_ACLD_PREFIX(p) (strncmp(p, acldprefix, sizeof(acldprefix) - 1) == 0)

/* Locals */

/* Parsed prefix lists used for ACLs */
static struct array rpc_prefix_array;
static struct rpc_prefix *rpc_prefix;

/* rpc_parse() result */
static struct rpcresult rpcresult;

/* rpc_parse_filter() flags */
#define PFLAG_ACCEPT	0x01
#define PFLAG_DISCARD	0x02
#define PFLAG_TCP	0x04
#define PFLAG_UDP	0x08
#define PFLAG_SYN	0x0A

#define PFLAG_TCPSYN	(PFLAG_TCP | PFLAG_SYN)

/* rpc_parse_chassis() flags */
#define CFLAG_MPC	0x01
#define CFLAG_DPC	0x02

/* Forwards */
static xmlNode *rpc_child_element(xmlNode *);
static void rpc_expand_hostprefix(struct acl *,
    struct rpc_prefix *, int);
static void rpc_expand_ports(struct aclgroup *,
    struct acllist *, struct acl *);
static const char *rpc_get_message(xmlNode *);
static int rpc_ignore_warning(const char *);
static xmlNode *rpc_next_element(xmlNode *);
static void rpc_parse_acld_mac_filter(xmlNode *);
static void rpc_parse_bridge(xmlNode *);
static void rpc_parse_bridge_domains(xmlNode *);
static void rpc_parse_chassis(xmlNode *);
static xmlNode *rpc_parse_commit_results(xmlNode *);
static void rpc_parse_configuration(xmlNode *);
static void rpc_parse_domain(xmlNode *);
static void rpc_parse_filter(xmlNode *);
static void rpc_parse_filter_mac_address(xmlNode *);
static void rpc_parse_filter_term(xmlNode *,
    struct acl *, const char *, int *);
static void rpc_parse_firewall(xmlNode *);
static xmlNode *rpc_parse_load_configuration(xmlNode *);
static void rpc_parse_module(xmlNode *, int *);
static void rpc_parse_policy(xmlNode *);
static void rpc_parse_prefix(xmlNode *);
static void rpc_parse_proto(xmlNode *);
static void rpc_parse_routing_engine(xmlNode *);
static void rpc_parse_term_from(xmlNode *, struct acl *, int *);
static void rpc_parse_term_then(xmlNode *, struct acl *, int *);
static xmlNode * rpc_parse_uptime(xmlNode *);
static char *rpc_prefix2acllistname(char *);
static void rpc_prefixaddaddr(const char *, struct addr *);
static struct rpc_prefix *rpc_prefixfind(const char *);
static const char *rpc_stripsuffixes(const char *);

static void rpc_prefix_cleanup(void);

/* Returns the child element node or NULL */
static xmlNode *
rpc_child_element(xmlNode *root)
{
	xmlNode *np;
	np = root->children;
	if (ISXML_TEXT(np)) {
		np = np->next;
		if (ISXML_ELEMENT(np))
			return (np);
	}
	return (NULL);
}

void
rpc_cleanup(xmlDocPtr doc)
{
	xmlFreeDoc(doc);
	xmlCleanupParser();
	rpc_prefix_cleanup();
}

static void
rpc_expand_hostprefix(struct acl *al, struct rpc_prefix *pl,
    int flags)
{
	int i;
	char *prefixname;
	const char *listname, *groupname;
	struct addr *a;
	struct array *dp;
	struct acllist *ap;
	struct aclgroup *gp;
	enum acltype hosttype, nettype;

	/* raw should give us the prefix-list name */
	if (al->raw == NULL) {
		lg(LOG_ERR, "rpc_expand_hostprefix: missing prefix-list name");
		return;
	}

	if (al->port1 == NULL) {
		if ((flags & PFLAG_DISCARD) != 0) {
			hosttype = ATYPE_BLOCKHOST;
			nettype = ATYPE_BLOCKNET;
		} else {
			hosttype = ATYPE_PERMITHOST;
			nettype = ATYPE_PERMITNET;
		}
	} else if ((flags & PFLAG_TCP) != 0) {
		if ((flags & PFLAG_DISCARD) != 0) {
			hosttype = ATYPE_BLOCKTCPDSTHOSTPORT;
			nettype = ATYPE_BLOCKTCPDSTNETPORT;
		} else {
			hosttype = ATYPE_PERMITTCPDSTHOSTPORT;
			nettype = ATYPE_PERMITTCPDSTNETPORT;
		}
	} else if ((flags & PFLAG_UDP) != 0) {
		if ((flags & PFLAG_DISCARD) != 0) {
			hosttype = ATYPE_BLOCKUDPDSTHOSTPORT;
			nettype = ATYPE_BLOCKUDPDSTNETPORT;
		} else {
			hosttype = ATYPE_PERMITUDPDSTHOSTPORT;
			nettype = ATYPE_PERMITUDPDSTNETPORT;
		}
	} else {
		lg(LOG_ERR, "rpc_expand_hostprefix: Can't determine ACL type");
		return;
	}

	/* Don't include "raw" (prefix name) new ACLs */
	prefixname = al->raw;
	al->raw = NULL;

	/* Convert prefixname to ACL list and group names */
	listname = rpc_prefix2acllistname(prefixname);
	groupname = rpc_stripsuffixes(prefixname);
	ap = aclfindlistbyname(listname);
	if (ap == NULL)
		lg(LOG_ERR,
		    "rpc_expand_hostprefix: Can't find ACL list %s (%s)",
		    listname, prefixname);
	gp = aclfindgroupbyname(groupname);
	if (gp == NULL)
		lg(LOG_ERR,
		    "rpc_expand_hostprefix: Can't find ACL group %s (%s)",
		    groupname, prefixname);
	if (ap == NULL || gp == NULL)
		return;

	dp = &pl->addr_array;
	for (i = 0, a = pl->addr; i < dp->len; ++i, ++a) {
		if (ISHOST(a))
			al->type = hosttype;
		else
			al->type = nettype;
		al->addr1 = *a;
		acladdacl2(gp, ap, al);
	}
	free(prefixname);
}

static void
rpc_expand_ports(struct aclgroup *gp, struct acllist *ap,
    struct acl *al)
{
	char *cp, *ep;
	char *ports;
	long v;
	int port;
	char sport[32];

	ports = al->port1;
	cp = ports;
	while (cp != NULL && *cp != '\0') {
		v = strtol(cp, &ep, 10);
		port = v;
		snprintf(sport, sizeof(sport), "%d", port);
		al->port1 = sport;
		acladdacl2(gp, ap, al);
		if (*ep == ' ')
			++ep;
		cp = ep;
	}
	al->port1 = NULL;
	free(ports);
}

/* Return the text from this node's content or NULL */
static const char *
rpc_get_content(xmlNode *np)
{
	if (np != NULL) {
		np = np->children;
		if (ISXML_TEXT(np))
			return ((const char *)np->content);
	}
	return (NULL);
}

/* Return the text from the first <message> we find or NULL */
static const char *
rpc_get_message(xmlNode *np)
{
	np = rpc_child_element(np);
	for (; np != NULL; np = rpc_next_element(np)) {
		if (ISXML_ELEMENT_NAMED(np, "message")) {
			np = np->children;
			if (ISXML_TEXT(np))
				return ((const char *)np->content);
			return ("unknown");
		}
	}

	return (NULL);
}

static const char *ignore_warnings[] = {
	"uncommitted changes will be discarded on exit",
	"'synchronize' ignored in single routing engine chassis",
	NULL
};

/* Check for warning we ignore */
static int
rpc_ignore_warning(const char *str)
{
	const char **pp;

	for (pp = ignore_warnings; *pp != NULL; ++pp)
		if (strcmp(str, *pp) == 0)
			return (1);
	return (0);
}

/* Returns the next element node or NULL */
static xmlNode *
rpc_next_element(xmlNode *root)
{
	xmlNode *np;
	np = root->next;
	if (ISXML_TEXT(np)) {
		np = np->next;
		if (ISXML_ELEMENT(np))
			return (np);
	}
	return (NULL);
}

static void
rpc_parse_acld_mac_filter(xmlNode *np)
{
	xmlNode *np2;

	while (np != NULL) {
		if (!ISXML_ELEMENT_NAMED(np, "term")) {
			np = rpc_next_element(np);
			continue;
		}

		np2 = rpc_child_element(np);
		while (np2 != NULL) {
			if (ISXML_ELEMENT_NAMED(np2, "from")) {
				rpc_parse_filter_mac_address(np2);
				break;
			}
			np2 = rpc_next_element(np2);
		}

		np = rpc_next_element(np);
	}
}

static void
rpc_parse_bridge(xmlNode *np)
{
	const char *p;
	xmlNode *np2;

	np = rpc_child_element(np);
	while (np != NULL) {
		if (ISXML_ELEMENT_NAMED(np, "filter")) {
			np2 = rpc_child_element(np);
			p = rpc_get_content(np2);
			if (p != NULL && strcmp(p, acld_filter_name) == 0) {
				have_acld_mac_filter = 1;
				rpc_parse_acld_mac_filter(np2);
			}
		}
		np = rpc_next_element(np);
	}
}

static void
rpc_parse_bridge_domains(xmlNode *np)
{
	np = rpc_child_element(np);
	while (np != NULL) {
		if (ISXML_ELEMENT_NAMED(np, "domain"))
			rpc_parse_domain(np);
		np = rpc_next_element(np);
		continue;
	}
}

static void
rpc_parse_domain(xmlNode *np)
{
	const char *p;
	xmlNode *np2, *np3;
	char name[32];

	strlcpy(name, "?", sizeof(name));
	np = rpc_child_element(np);
	while (np != NULL) {
		if (ISXML_ELEMENT_NAMED(np, "name")) {
			p = rpc_get_content(np);
			strlcpy(name, p, sizeof(name));
			np = rpc_next_element(np);
			continue;
		}

		if (!ISXML_ELEMENT_NAMED(np, "forwarding-options")) {
			np = rpc_next_element(np);
			continue;
		}
		np2 = rpc_child_element(np);
		np = rpc_next_element(np);
		while (np2 != NULL) {
			if (ISXML_ELEMENT_NAMED(np2, "filter")) {
				np3 = rpc_child_element(np2);
				p = rpc_get_content(np3);
				if (p != NULL &&
				    strcmp(p, acld_filter_name) == 0)
					lg(LOG_INFO,
					    "blockmac is supported on %s",
					    name);
			}
			np2 = rpc_next_element(np2);
		}
	}
}

static void
rpc_parse_chassis(xmlNode *np)
{
	int flags;
	const char *p;
	char model[32];

	np = rpc_child_element(np);
	if (!ISXML_ELEMENT_NAMED(np, "chassis")) {
		lg(LOG_ERR, "rpc_parse_chassis: Can't find family node");
		return;
	}
	np = rpc_child_element(np);

	flags = 0;
	model[0] = '\0';
	while (np != NULL) {
		if (ISXML_ELEMENT_NAMED(np, "description")) {
			/* Save the first description we find */
			if (model[0] == '\0') {
				p = rpc_get_content(np);
				strlcpy(model, p, sizeof(model));
			}
			np = rpc_next_element(np);
			continue;
		}
		if (ISXML_ELEMENT_NAMED(np, "chassis-module")) {
			rpc_parse_module(np, &flags);
			np = rpc_next_element(np);
			continue;
		}

		/* Skip stuff we don't understand */
		np = rpc_next_element(np);
	}

	/* Select DPC we have both */
	if ((flags & CFLAG_DPC) != 0) {
		p = "DPC";
	} else if ((flags & CFLAG_MPC) != 0) {
		p = "MPC";
		++havempc;
	} else {
		lg(LOG_WARNING, "MPC/DPC not detected, default to DPC");
		return;
	}
	if (model[0] == '\0')
		strlcpy(model, "?", sizeof(model));
	lg(LOG_INFO, "Juniper %s chassis using %s", model, p);
}

static xmlNode *
rpc_parse_commit_results(xmlNode *root)
{
	const char *p;
	xmlNode *np;
	struct rpcresult *rrp;

	rrp = &rpcresult;
	np = rpc_child_element(root);
	for (; np != NULL; np = rpc_next_element(np)) {
		if (ISXML_ELEMENT_NAMED(np, "routing-engine")) {
			rpc_parse_routing_engine(np);
			continue;
		}
		if (ISXML_ELEMENT_NAMED(np, "error")) {
			strlcpy(rrp->errmsg, stripws(rpc_get_message(np)),
			    sizeof(rrp->errmsg));
			p = pretty(rrp->errmsg);
			lg(LOG_DEBUG,
			    "rpc_parse_commit_results: error: \"%s\"", p);
			clientsendallerr("%s", p);
			continue;
		}
		if (ISXML_ELEMENT_NAMED(np, "warning")) {
			p = stripws(rpc_get_message(np));
			if (!rpc_ignore_warning(p))
				lg(LOG_DEBUG,
				    "rpc_parse_commit_results: warning: \"%s\"",
				    pretty(p));
			continue;
		}
		if (ISXML_ELEMENT_NAMED(np, "load-success")) {
			++rrp->commitsuccess;
			continue;
		}

		lg(LOG_ERR, "rpc_parse_commit_results: skipping \"%s\"",
		    (const char *)np->name);
	}

	if (rrp->commitsuccess)
		clientsendall();
	else
		clientsendallerr("rpc_parse_commit_results: "
		    "missing commit results (commit without changes?)");

	return (rpc_next_element(root));
}

static void
rpc_parse_configuration(xmlNode *np)
{
	struct state *sp;

	sp = &state;
	np = rpc_child_element(np);
	while (np != NULL) {
		if (ISXML_ELEMENT_NAMED(np, "policy-options")) {
			rpc_parse_policy(np);
			np = rpc_next_element(np);
			continue;
		}

		if (ISXML_ELEMENT_NAMED(np, "firewall")) {
			rpc_parse_firewall(np);
			np = rpc_next_element(np);
			continue;
		}

		if (ISXML_ELEMENT_NAMED(np, "bridge-domains")) {
			rpc_parse_bridge_domains(np);
			np = rpc_next_element(np);
			continue;
		}

		if (!ISXML_ELEMENT_NAMED(np, "comment"))
			lg(LOG_ERR, "rpc_parse_configuration: skipping \"%s\"",
			    (const char *)np->name);
		np = rpc_next_element(np);
	}

	/* Unlock the front door */
	++sp->listedallacls;
	if (sp->state == ASTATE_SENTLISTACL) {
		setastate(ASTATE_LOGGEDIN);
		timerreset(&sp->t_login);
	}
}

static void
rpc_parse_filter(xmlNode *np)
{
	int flags;
	const char *p, *listname, *groupname;
	struct acllist *ap;
	struct acl acl, *al;
	struct aclgroup *gp;
	enum acltype porttype;
	struct rpcresult *rrp;

	/* Step down to child and loop over terms */
	rrp = &rpcresult;
	al = &acl;
	memset(al, 0, sizeof(*al));
	listname = "?";
	groupname = "?";
	ap = NULL;
	gp = NULL;
	np = rpc_child_element(np);
	for (; np != NULL; np = rpc_next_element(np)) {
		/* Initialize to parse next term */
		aclclean(al);
		memset(al, 0, sizeof(*al));
		al->seq = -1;
		flags = 0;

		/* Ignore */
		if (ISXML_ELEMENT_NAMED(np, "interface-specific"))
			continue;

		if (ISXML_ELEMENT_NAMED(np, "name")) {
			listname = rpc_get_content(np);
			if (listname == NULL)
				continue;

			/* Strip suffix to get ACL group name */
			p = rpc_stripsuffixes(listname);

			/* Ignore any that aren't one of our ACL names */
			if (aclfindgroupbyname(p) == NULL)
				return;
			continue;
		}

		/* Ignore */
		if (ISXML_ELEMENT_NAMED(np, "comment"))
			continue;

		if (!ISXML_ELEMENT_NAMED(np, "term")) {
			lg(LOG_DEBUG,
			    "rpc_parse_filter: ignoring \"%s\" in filter %s",
			    (const char *)np->name, listname);
			continue;
		}

		/* We should already have parsed the filter name */
		groupname = rpc_stripsuffixes(listname);
		ap = aclfindlistbyname(listname);
		if (ap == NULL)
			lg(LOG_ERR,
			    "rpc_parse_filter: Unknown ACL list %s",
			    listname);
		gp = aclfindgroupbyname(groupname);
		if (gp == NULL)
			lg(LOG_ERR,
			    "rpc_parse_filter: Unknown ACL group %s",
			    groupname);
		if (ap == NULL || gp == NULL)
			return;

		/* Parse all term children nodes */
		rpc_parse_filter_term(np, al, listname, &flags);
		if (rrp->errmsg[0] != '\0') {
			lg(LOG_ERR, "acl %s:%d %s",
			    listname, al->seq, rrp->errmsg);
			continue;
		}

		/* Ignore terms without numeric term names */
		if (al->seq < 0)
			continue;

		/* Add next acl: first do error checking */
		if ((flags & (PFLAG_ACCEPT | PFLAG_DISCARD)) == 0) {
			lg(LOG_ERR, "acl %s:%d Missing accept/discard",
			    listname, al->seq);
			continue;
		}

		/* Shift single address to addr1 */
		if (al->addr1.family == 0 && al->addr2.family != 0) {
			al->addr1 = al->addr2;
			memset(&al->addr2, 0, sizeof(al->addr2));
		}

		/* Shift single port to port1 */
		if (al->port1 == NULL && al->port2 != NULL) {
			al->port1 = al->port2;
			al->port2 = NULL;
		}

		/* Determine ACL type for port exceptions */
		if (al->type == ATYPE_UNKNOWN &&
		    al->port1 != NULL && al->raw == NULL) {
			if ((flags & PFLAG_TCPSYN) == PFLAG_TCPSYN) {
				if ((flags & PFLAG_DISCARD) != 0)
					porttype = ATYPE_BLOCKTCPSYNPORT;
				else
					porttype = ATYPE_PERMITTCPSYNPORT;
			} else if ((flags & PFLAG_TCP) != 0) {
				if ((flags & PFLAG_DISCARD) != 0)
					porttype = ATYPE_BLOCKTCPPORT;
				else
					porttype = ATYPE_PERMITTCPPORT;
			} else if ((flags & PFLAG_UDP) != 0) {
				if ((flags & PFLAG_DISCARD) != 0)
					porttype = ATYPE_BLOCKUDPPORT;
				else
					porttype = ATYPE_PERMITUDPPORT;
			} else {
				lg(LOG_ERR,
				    "acl %s:%d unhandled ACL port type %s",
				    listname, al->seq,
				    val2str(str2acl, al->type));
				return;
			}
			al->type = porttype;
		}

		/*
		 * Handle prefix-list guys (host blocks and host-port
		 * exceptions) later
		 */
		if (al->raw != NULL)
			continue;

		switch (al->type) {

		case ATYPE_BLOCKHOST:
		case ATYPE_BLOCKNET:
			break;

		case ATYPE_PERMITTCPDSTHOSTPORT:
		case ATYPE_PERMITUDPDSTHOSTPORT:
		case ATYPE_PERMITTCPDSTNETPORT:
		case ATYPE_PERMITUDPDSTNETPORT:
			break;

		case ATYPE_BLOCKTCPPORT:
		case ATYPE_BLOCKUDPPORT:
		case ATYPE_BLOCKTCPSYNPORT:
			rpc_expand_ports(gp, ap, al);
			break;

		case ATYPE_UNKNOWN:
		default:
			lg(LOG_ERR, "acl %s:%d unhandled ACL type %s",
			    listname, al->seq, val2str(str2acl, al->type));
			continue;
		}
	}
	aclclean(al);
}

static void
rpc_parse_filter_mac_address(xmlNode *np)
{
	char *cp;
	const char *p, *errmsg;
	xmlNode *np2;
	struct aclgroup *gp;
	struct acl acl, *al;
	char mac[32];

	np = rpc_child_element(np);
	while (np != NULL) {
		if (!ISXML_ELEMENT_NAMED(np, "destination-mac-address") &&
		    !ISXML_ELEMENT_NAMED(np, "source-mac-address")) {
			np = rpc_next_element(np);
			continue;
		}
		np2 = rpc_child_element(np);
		if (!ISXML_ELEMENT_NAMED(np2, "name")) {
			np = rpc_next_element(np);
			continue;
		}
		p = rpc_get_content(np2);
		if (p == NULL) {
			np = rpc_next_element(np);
			continue;
		}
		strlcpy(mac, p, sizeof(mac));
		cp = strchr(mac, '/');
		if (cp != NULL)
			*cp = '\0';

		al = &acl;
		memset(al, 0, sizeof(*al));
		al->type = ATYPE_BLOCKMAC;
		gp = NULL;
		if (gp == NULL)
			gp = aclgetmacgroup();
		if (gp == NULL) {
			lg(LOG_DEBUG, "rpc_parse_filter_mac_address: "
			    "Found mac %s without mac ACL", mac);
			np = rpc_next_element(np);
			continue;
		}

		/* Populate the ACL */
		errmsg = extractmac(mac, &al->mac);
		if (errmsg != NULL) {
			lg(LOG_DEBUG, "rpc_parse_source_address_filter: %s",
			    errmsg);
			np = rpc_next_element(np);
			continue;
		}

		/* Add the ACL (if we don't already have it) */
		if (aclfindacl(gp, al, 0) == NULL)
			acladdacl2(gp, gp->acllist, al);

		np = rpc_next_element(np);
	}
}

static void
rpc_parse_filter_term(xmlNode *root, struct acl *al,
    const char *listname, int *flagsp)
{
	long lv;
	char *ep;
	const char *p;
	xmlNode *np;
	struct rpc_prefix *pl;
	struct rpcresult *rrp;

	rrp = &rpcresult;
	np = rpc_child_element(root);
	for (; np != NULL; np = rpc_next_element(np)) {
		if (ISXML_ELEMENT_NAMED(np, "name")) {
			/* Parse sequence number */
			p = rpc_get_content(np);
			if (p == NULL) {
				lg(LOG_INFO, "rpc_parse_filter_term:"
				    " Can't find term name");
				return;
			}
			lv = strtol(p, &ep, 10);
			if (*ep != '\0') {
				lg(LOG_INFO, "skipping term %s in filter %s",
				    p, listname);
				return;
			}
			al->seq = lv;
			continue;
		}
		if (ISXML_ELEMENT_NAMED(np, "from")) {
			rpc_parse_term_from(np, al, flagsp);
			if (rrp->errmsg[0] != '\0')
				return;
			continue;
		}
		if (ISXML_ELEMENT_NAMED(np, "then")) {
			rpc_parse_term_then(np, al, flagsp);
			if (rrp->errmsg[0] != '\0')
				return;
			continue;
		}

		(void)snprintf(rrp->errmsg, sizeof(rrp->errmsg),
		    "unknown filter term \"%s\"", (const char *)np->name);
		return;
	}

	/* Expand the prefix list into multiple ACLs */
	if (al->raw != NULL) {
		pl = rpc_prefixfind(al->raw);
		/* Ok if the prefix is missing/empty */
		if (pl != NULL)
			rpc_expand_hostprefix(al, pl, *flagsp);
	}
}

static void
rpc_parse_firewall(xmlNode *np)
{
	np = rpc_child_element(np);
	if (!ISXML_ELEMENT_NAMED(np, "family")) {
		lg(LOG_ERR, "rpc_parse_firewall: Can't find family node");
		return;
	}
	np = rpc_child_element(np);

	while (np != NULL) {
		if (ISXML_ELEMENT_NAMED(np, "inet") ||
		    ISXML_ELEMENT_NAMED(np, "inet6")) {
			rpc_parse_proto(np);
			np = rpc_next_element(np);
			continue;
		}

		if (ISXML_ELEMENT_NAMED(np, "bridge")) {
			rpc_parse_bridge(np);
			np = rpc_next_element(np);
			continue;
		}

		if (!ISXML_ELEMENT_NAMED(np, "comment"))
			lg(LOG_ERR, "rpc_parse_firewall: skipping \"%s\"",
			    (const char *)np->name);
		np = rpc_next_element(np);
	}
}

static xmlNode *
rpc_parse_load_configuration(xmlNode *root)
{
	const char *p;
	xmlNode *np;
	struct rpcresult *rrp;

	rrp = &rpcresult;
	np = rpc_child_element(root);
	for (; np != NULL; np = rpc_next_element(np)) {
		if (ISXML_ELEMENT_NAMED(np, "error")) {
			/* Exit private configuration mode */
			junoscriptexitprivate();

			strlcpy(rrp->errmsg, stripws(rpc_get_message(np)),
			    sizeof(rrp->errmsg));
			p = pretty(rrp->errmsg);
			lg(LOG_DEBUG,
			    "rpc_parse_load_configuration: error: \"%s\"", p);
			clientsendallerr("%s", rrp->errmsg);
			++rrp->loaderror;
			continue;
		}
		if (ISXML_ELEMENT_NAMED(np, "warning")) {
			strlcpy(rrp->errmsg, stripws(rpc_get_message(np)),
			    sizeof(rrp->errmsg));
			p = pretty(rrp->errmsg);
			lg(LOG_DEBUG,
			    "rpc_parse_load_configuration: warning: \"%s\"", p);
			if (strstr(rrp->errmsg, "statement not found"))
				++rrp->loaderror;
			continue;
		}
		if (ISXML_ELEMENT_NAMED(np, "load-success")) {
			/* Now we can commit the configuration change */
			++rrp->loadsuccess;
			continue;
		}

		/* Ignore */
		/* XXX shouldn't we care? */
		if (ISXML_ELEMENT_NAMED(np, "load-error-count"))
			continue;

		lg(LOG_ERR,
		    "rpc_parse_load_configuration: skipping \"%s\"",
		    (const char *)np->name);
	}
	if (rrp->loadsuccess && !rrp->loaderror)
		junoscriptdocommit();
	return (rpc_next_element(root));
}

static void
rpc_parse_module(xmlNode *np, int *flagsp)
{
	const char *p;

	np = rpc_child_element(np);
	while (np != NULL) {
		if (ISXML_ELEMENT_NAMED(np, "description")) {
			p = rpc_get_content(np);
			if (p == NULL) {
				lg(LOG_INFO, "rpc_parse_module:"
				    " Can't find module description");
				continue;
			}
			if (strcasestr(p, "mpc") != NULL)
				*flagsp |= CFLAG_MPC;
			else
			if (strcasestr(p, "dpc") != NULL)
				*flagsp |= CFLAG_DPC;
			np = rpc_next_element(np);
			continue;
		}

		/* Skip stuff we don't understand */
		np = rpc_next_element(np);
	}
}

static void
rpc_parse_policy(xmlNode *np)
{
	np = rpc_child_element(np);
	while (np != NULL) {
		if (ISXML_ELEMENT_NAMED(np, "prefix-list")) {
			rpc_parse_prefix(np);
			np = rpc_next_element(np);
			continue;
		}

		if (!ISXML_ELEMENT_NAMED(np, "comment"))
			lg(LOG_ERR, "rpc_parse_policy: skipping \"%s\"",
			    (const char *)np->name);
		np = rpc_next_element(np);
	}
}

static void
rpc_parse_prefix(xmlNode *np)
{
	int isacldprefix;
	const char *name, *p, *errmsg;
	xmlNode *np2;
	struct addr addr, *a;

	name = NULL;
	isacldprefix = 0;
	np = rpc_child_element(np);
	while (np != NULL) {
		if (ISXML_ELEMENT_NAMED(np, "name")) {
			/* Ignore any that aren't one of our ACL names */
			name = rpc_get_content(np);
			isacldprefix = IS_ACLD_PREFIX(name);
			if (!isacldprefix &&
			    aclfindgroupbyname(rpc_stripsuffixes(name)) == NULL)
				return;

			np = rpc_next_element(np);
			continue;
		}
		if (ISXML_ELEMENT_NAMED(np, "prefix-list-item")) {
			if (name == NULL) {
				lg(LOG_ERR, "rpc_parse_prefix:"
				    " missing prefix-list name");
				return;
			}
			np2 = rpc_child_element(np);
			if (!ISXML_ELEMENT_NAMED(np2, "name")) {
				lg(LOG_ERR, "rpc_parse_prefix:"
				    " Can't find prefix-list-item name (%s)",
				    name);
				continue;
			}
			p = rpc_get_content(np2);
			a = &addr;
			errmsg = extractaddr(p, NULL, a);
			if (errmsg != NULL) {
				lg(LOG_ERR, "rpc_parse_prefix:"
				    " Can't parse addr \"%s\"", p);
				np = rpc_next_element(np);
				continue;
			}
			if (isacldprefix) {
				errmsg = prefixaddaddr(
				    name + sizeof(acldprefix) - 1, a);
				if (errmsg != NULL) {
					lg(LOG_ERR, "rpc_parse_prefix: %s",
					    errmsg);
				}
			} else
				rpc_prefixaddaddr(name, a);
			np = rpc_next_element(np);
			continue;
		}

		if (!ISXML_ELEMENT_NAMED(np, "comment"))
			lg(LOG_ERR, "rpc_parse_prefix: skipping \"%s\"",
			    (const char *)np->name);
		np = rpc_next_element(np);
	}
}

static void
rpc_parse_proto(xmlNode *np)
{
	np = rpc_child_element(np);
	while (np != NULL) {
		if (ISXML_ELEMENT_NAMED(np, "filter")) {
			rpc_parse_filter(np);
			np = rpc_next_element(np);
			continue;
		}

		if (!ISXML_ELEMENT_NAMED(np, "comment"))
		    lg(LOG_ERR, "rpc_parse_proto: skipping \"%s\"",
			(const char *)np->name);
		np = rpc_next_element(np);
	}
}

static void
rpc_parse_routing_engine(xmlNode *root)
{
	xmlNode *np;
	const char *p;
	struct rpcresult *rrp;

	rrp = &rpcresult;
	np = rpc_child_element(root);
	for (; np != NULL; np = rpc_next_element(np)) {
		if (ISXML_ELEMENT_NAMED(np, "error")) {
			strlcpy(rrp->errmsg, stripws(rpc_get_message(np)),
			    sizeof(rrp->errmsg));
			p = pretty(rrp->errmsg);
			lg(LOG_DEBUG,
			    "rpc_parse_routing_engine: error: \"%s\"", p);
			clientsendallerr("%s", p);
			continue;
		}
		if (ISXML_ELEMENT_NAMED(np, "warning")) {
			strlcpy(rrp->errmsg, stripws(rpc_get_message(np)),
			    sizeof(rrp->errmsg));
			p = pretty(rrp->errmsg);
			lg(LOG_DEBUG,
			    "rpc_parse_routing_engine: warning: \"%s\"", p);
			if (strstr(p, "ignored in single routing") != NULL)
				++rrp->engineignored;
			continue;
		}
		if (ISXML_ELEMENT_NAMED(np, "commit-success")) {
			++rrp->enginecommitsuccess;
			continue;
		}

		/* Ignore */
		if (ISXML_ELEMENT_NAMED(np, "name") ||
		    ISXML_ELEMENT_NAMED(np, "commit-check-success")) {
			continue;
		}

		lg(LOG_ERR, "rpc_parse_routing_engine: skipping \"%s\"",
		    (const char *)np->name);
	}
}

static void
rpc_parse_term_from(xmlNode *root, struct acl *al, int *flagsp)
{
	const char *p;
	xmlNode *np, *np2;
	struct rpcresult *rrp;

	rrp = &rpcresult;
	np = rpc_child_element(root);
	for (; np != NULL; np = rpc_next_element(np)) {
		if (ISXML_ELEMENT_NAMED(np, "protocol") ||
		    ISXML_ELEMENT_NAMED(np, "payload-protocol") ||
		    ISXML_ELEMENT_NAMED(np, "next-header")) {
			p = rpc_get_content(np);
			if (strcmp(p, "tcp") == 0)
				*flagsp |= PFLAG_TCP;
			else if (strcmp(p, "udp") == 0)
				*flagsp |= PFLAG_UDP;
			else {
				(void)snprintf(rrp->errmsg, sizeof(rrp->errmsg),
				    "unknown protocol \"%s\"", p);
				return;
			}
			continue;
		}
		if (ISXML_ELEMENT_NAMED(np, "destination-port")) {
			/* Create a blank delimiated list of ports */
			p = rpc_get_content(np);
			if (al->port1 != NULL)
				strappend(&al->port1, " ");
			strappend(&al->port1, p);
			continue;
		}
		if (ISXML_ELEMENT_NAMED(np, "prefix-list") ||
		    ISXML_ELEMENT_NAMED(np, "source-prefix-list") ||
		    ISXML_ELEMENT_NAMED(np, "destination-prefix-list")) {
			np2 = rpc_child_element(np);
			if (!ISXML_ELEMENT_NAMED(np2, "name"))
				strlcpy(rrp->errmsg,
				    "missing prefix-list name",
				    sizeof(rrp->errmsg));

			/* XXX don't save if not "ACLD-..." (or whatever) */

			p = rpc_get_content(np2);
			if (al->raw != NULL)
				free(al->raw);
			/* Hide the prefix-list name for later processing */
			al->raw = strsave(p);
			continue;
		}
		if (ISXML_ELEMENT_NAMED(np, "tcp-initial")) {
			*flagsp |= PFLAG_SYN;
			continue;
		}

		(void)snprintf(rrp->errmsg, sizeof(rrp->errmsg),
		    "unknown from \"%s\"", (const char *)np->name);
		return;
	}
}

static void
rpc_parse_term_then(xmlNode *root, struct acl *al, int *flagsp)
{
	xmlNode *np;
	struct rpcresult *rrp;

	rrp = &rpcresult;
	np = rpc_child_element(root);
	for (; np != NULL; np = rpc_next_element(np)) {
		if (ISXML_ELEMENT_NAMED(np, "discard")) {
			*flagsp |= PFLAG_DISCARD;
			continue;
		}
		if (ISXML_ELEMENT_NAMED(np, "accept")) {
			*flagsp |= PFLAG_ACCEPT;
			continue;
		}

		(void)snprintf(rrp->errmsg, sizeof(rrp->errmsg),
		    "unknown then \"%s\"", (const char *)np->name);
		return;
	}
}

static xmlNode *
rpc_parse_uptime(xmlNode *root)
{
	xmlNode *np;
	const char *p;
	const char *s_time;
	const char *s_uptime;
	const char *s_users;
	const char *s_load1;
	const char *s_load5;
	const char *s_load15;

	s_time = "?";
	s_uptime = "?";
	s_users = "?";
	s_load1 = "?";
	s_load5 = "?";
	s_load15 = "?";

	/* We only want to look at uptime-information children */
	np = rpc_child_element(root);
	for (; np != NULL; np = rpc_next_element(np)) {
		if (ISXML_ELEMENT_NAMED(np, "uptime-information")) {
			np = rpc_child_element(np);
			break;
		}
	}

	for (; np != NULL; np = rpc_next_element(np)) {
		if (ISXML_ELEMENT_NAMED(np, "date-time")) {
			s_time = rpc_get_content(np);
			continue;
		}
		if (ISXML_ELEMENT_NAMED(np, "up-time")) {
			s_uptime = rpc_get_content(np);
			continue;
		}
		if (ISXML_ELEMENT_NAMED(np,
		    "active-user-count")) {
			s_users = rpc_get_content(np);
			continue;
		}
		if (ISXML_ELEMENT_NAMED(np, "load-average-1")) {
			s_load1 = rpc_get_content(np);
			continue;
		}
		if (ISXML_ELEMENT_NAMED(np, "load-average-5")) {
			s_load5 = rpc_get_content(np);
			continue;
		}
		if (ISXML_ELEMENT_NAMED(np, "load-average-15")) {
			s_load15 = rpc_get_content(np);
			continue;
		}

		lg(LOG_ERR, "rpc_parse_uptime: skipping \"%s\"",
		    (const char *)np->name);
	}

	if (strcmp(s_users, "1") == 0)
		p = "";
	else
		p = "s";

	lg(LOG_INFO, "junos: %s up %s, %s user%s, load averages: %s, %s, %s",
	    s_time, s_uptime, s_users, p, s_load1, s_load5, s_load15);
	return (rpc_next_element(root));
}

/*
 * Map a prefix name to an ACL list name
 *
 * For example: ACLD-TCP-HOSTPORT25 -> ACLD-HOSTPORT
 */
static char *
rpc_prefix2acllistname(char *name)
{
	char *cp, *cp2;
	static char buf[128];

	strlcpy(buf, name, sizeof(buf));

	/* Truncate trailing port number */
	cp = buf + strlen(buf);
	while (cp > buf && isdigit(cp[-1]))
		--cp;
	*cp = '\0';

	/* Find "-TCP" or "-UDP" */
	cp = strchr(buf, '-');
	if (cp != NULL) {
		++cp;
		/* Find "HOSTPORT" */
		cp2 = strchr(cp, '-');
		if (cp2 != NULL) {
			++cp2;
			memmove(cp, cp2, strlen(cp2) + 1);
			return (buf);
		}
	}
	return (name);
}

static void
rpc_prefix_cleanup(void)
{
	int i;
	struct rpc_prefix *pl;
	struct array *dp, *dp2;

	dp = &rpc_prefix_array;
	for (i = 0, pl = rpc_prefix; i < dp->len; ++i, ++pl) {
		free((void *)pl->name);
		dp2 = &pl->addr_array;
		FREEDYNARRAY(dp2, pl->addr);
	}
	FREEDYNARRAY(dp, rpc_prefix);
}

#ifdef notdef
void
rpc_prefix_dump(void)
{
	int i, j;
	struct addr *a;
	struct rpc_prefix *pl;
	struct array *dp, *dp2;

	dp = &rpc_prefix_array;
	for (i = 0, pl = rpc_prefix; i < dp->len; ++i, ++pl) {
		dp2 = &pl->addr_array;
		printf("# %s\n", pl->name);
		for (j = 0, a = pl->addr; j < dp2->len; ++j, ++a)
			printf("%s\n", addr2str(a));
	}
}
#endif

static void
rpc_prefixaddaddr(const char *name, struct addr *a)
{
	struct addr *a2;
	struct array *dp;
	struct rpc_prefix *pl;

	pl = rpc_prefixfind(name);
	if (pl == NULL) {
		dp = &rpc_prefix_array;
		DYNARRAY(dp, rpc_prefix, dp->len + 1, "prefix-list");
		pl = rpc_prefix + dp->len;
		pl->name = strsave(name);
		++dp->len;

		dp = &pl->addr_array;
		dp->osize = sizeof(*a);
		dp->inclen = 8;
		DYNARRAY(dp, pl->addr, dp->len + 1, "prefix-list addr (init)");
	}
	dp = &pl->addr_array;
	DYNARRAY(dp, pl->addr, dp->len + 1, "prefix-list addr (add)");
	a2 = pl->addr + dp->len;
	*a2 = *a;
	++dp->len;
}

static struct rpc_prefix *
rpc_prefixfind(const char *name)
{
	int i;
	struct rpc_prefix *pl;
	struct array *dp;

	if (name != NULL) {
		dp = &rpc_prefix_array;
		for (i = 0, pl = rpc_prefix; i < dp->len; ++i, ++pl)
			if (strcmp(name, pl->name) == 0)
				return (pl);
	}
	return (NULL);
}

struct rpcresult *
rpc_process(xmlNode *root)
{
	const char *p;
	xmlNode *np;
	struct array *dp;
	struct rpcresult *rrp;

	rrp = &rpcresult;
	memset(&rpcresult, 0, sizeof(rpcresult));
	dp = &rpc_prefix_array;
	dp->osize = sizeof(*rpc_prefix);
	dp->inclen = 1024;

	if (!ISXML_ELEMENT_NAMED(root, "rpc-reply")) {
		(void)snprintf(rrp->errmsg, sizeof(rrp->errmsg),
		    "rpc_process: unexpected xml %s", (const char *)root->name);
		return (rrp);
	}

	/* Process reply */
	np = rpc_child_element(root);
	while (np != NULL) {
		if (ISXML_ELEMENT_NAMED(np, "configuration")) {
			rpc_parse_configuration(np);
			np = rpc_next_element(np);
			continue;
		}
		if (ISXML_ELEMENT_NAMED(np, "error")) {
			strlcpy(rrp->errmsg, stripws(rpc_get_message(np)),
			    sizeof(rrp->errmsg));
			p = pretty(rrp->errmsg);
			lg(LOG_DEBUG, "rpc_process: error: \"%s\"", p);
			clientsendallerr("%s", p);
			np = rpc_next_element(np);
			continue;
		}
		if (ISXML_ELEMENT_NAMED(np, "warning")) {
			p = stripws(rpc_get_message(np));
			if (!rpc_ignore_warning(p)) {
				lg(LOG_DEBUG,
				    "rpc_process: warning: \"%s\"", p);
			}
			np = rpc_next_element(np);
			continue;
		}
		if (ISXML_ELEMENT_NAMED(np, "chassis-inventory")) {
			rpc_parse_chassis(np);
			np = rpc_next_element(np);
			continue;
		}
		if (ISXML_ELEMENT_NAMED(np, "load-configuration-results")) {
			sentconfig = 0;
			np = rpc_parse_load_configuration(np);
			continue;
		}
		if (ISXML_ELEMENT_NAMED(np, "system-uptime-information")) {
			np = rpc_parse_uptime(np);
			continue;
		}
		if (ISXML_ELEMENT_NAMED(np, "commit-results")) {
			sentcommit = 0;
			/* Exit private configuration mode */
			junoscriptexitprivate();
			np = rpc_parse_commit_results(np);
			continue;
		}

		if (!ISXML_ELEMENT_NAMED(np, "comment"))
			lg(LOG_ERR, "rpc_process: skipping \"%s\"",
			    (const char *)np->name);
		np = rpc_next_element(np);
	}

	return (rrp);
}

static const char *
rpc_stripsuffixes(const char *name)
{
	char *cp;
	static char shortname[64];

	strlcpy(shortname, name, sizeof(shortname));
	cp = strchr(shortname, '-');
	if (cp != NULL) {
		*cp = '\0';
		return (shortname);
	}
	return (name);
}
