/*
 * Copyright (c) 2004, 2006, 2007, 2008, 2009, 2010, 2011, 2012, 2013, 2014, 2015, 2016, 2018, 2019, 2020, 2024
 * The Regents of the University of California. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *     * Redistributions of source code must retain the above copyright
 *       notice, this list of conditions and the following disclaimer.
 *     * Redistributions in binary form must reproduce the above copyright
 *       notice, this list of conditions and the following disclaimer in the
 *       documentation and/or other materials provided with the distribution.
 *     * Neither the name of the University nor the names of its contributors
 *       may be used to endorse or promote products derived from this software
 *       without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#ifndef lint
static const char rcsid[] __attribute__ ((unused)) =
    "@(#) $Id$ (LBL)";
#endif

#include <sys/types.h>
#include <sys/time.h>

#include <errno.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sysexits.h>
#include <unistd.h>

#include "acld.h"
#include "cf.h"
#include "child.h"
#include "server.h"

/* Forwards */
void servercompactone(struct aclgroup *);
void servercompactport(struct aclgroup *);
void servercompactpermithostport(struct aclgroup *);
void serverdroprestore(struct req *);
void servermoveacl(struct aclgroup *, struct acl *, int);

void
serverayt(void)
{
	struct req *rp;

	rp = newreq("serverayt req");
	rp->state = RSTATE_PENDING;
	rp->type = REQ_AYT;
	rp->cmd = "ayt";

	/* Arrival timestamp */
	getts(&rp->ats);

	/* Append the new request to end of list */
	appendreq(&state.req, rp);
}

void
servercompact(void)
{
	int i;
	struct aclgroup *gp;
	struct array *dp;
	struct state *sp;

	sp = &state;
	dp = &cf->c_aclgroup_array;
	for (i = 0, gp = cf->c_aclgroup; i < dp->len; ++i, ++gp) {
		if (gp->compactseq > 0) {
			servercompactone(gp);
			break;
		}
		if (gp->compactportseq > 0) {
			servercompactport(gp);
			break;
		}
		if (gp->compactpermithostportseq > 0) {
			servercompactpermithostport(gp);
			break;
		}
	}

	/* Reschedule if we're not done yet */
	if (gp->compactseq > 0 ||
	    gp->compactportseq > 0 ||
	    gp->compactpermithostportseq > 0) {
		timerset(&sp->t_compact, 0);
		return;
	}

	/* Terminado! */
	timerreset(&sp->t_compact);

	/* Schedule a sync */
	if (cf->c_sync_secs > 0 && timercheck(&sp->t_sync) < 0)
		timerset(&sp->t_sync, cf->c_sync_secs);

	lg(LOG_INFO, "servercompact: finished compacting ACL %s", gp->name);

	/* Tell client the good news */
	if (gp->compactclient != NULL) {
		/*
		 * XXX What happens if the client goes away
		 * before we get here?
		 */
		if (gp->compactreq == NULL)
			abort();
		clientsend(gp->compactclient, gp->compactreq);
		freereq(gp->compactreq);
		gp->compactclient = NULL;
		gp->compactreq = NULL;
	}
}

void
servercompactone(struct aclgroup *gp)
{
	int i, j, inc, seq;
	struct acllist *ap;
	struct acl *al;
	struct array *dp;
	struct state *sp;

	/* Host block range */
	sp = &state;
	j = gp->acltotallen;
	inc = 1;
	dp = &gp->acllist_array;
	for (i = 0, ap = gp->acllist; i < dp->len; ++i, ++ap) {
		if (cf->c_incrseq) {
			al = ap->acl;
			inc = 1;
		} else {
			al = ap->acl + j - 1;
			inc = -1;
		}

		for (; j > 0; --j, al += inc) {
			if (!IN_SEQRANGE(al->seq))
				continue;
			if (cf->c_incrseq) {
				if (al->seq < gp->compactseq)
					continue;
			} else {
				if (al->seq > gp->compactseq)
					continue;
			}
			if (al->type == ATYPE_UNKNOWN) {
				gp->compactseq = 0;
				timerreset(&sp->t_compact);
				lg(LOG_ERR, "servercompactone: Can't handle"
				    " unknown ACL type (seq %d)", al->seq);
				return;
			}
			seq = gp->compactseq;
			gp->compactseq += inc;
			if (al->seq != seq) {
				/* XXX Check to see if it's in use? */
				servermoveacl(gp, al, seq);
				return;
			}

			/* XXX paranoid */
			if (cf->c_incrseq) {
				if (gp->compactseq >= cf->c_highseq)
					abort();
			} else {
				if (gp->compactseq <= cf->c_lowseq)
					abort();
			}
		}
	}

	/* If we got here we're done */
	gp->lastseq = gp->compactseq - inc;
	gp->compactseq = 0;
}

void
servercompactpermithostport(struct aclgroup *gp)
{
	int i, j, seq;
	struct acllist *ap;
	struct acl *al;
	struct array *dp;
	struct state *sp;

	/* Port block range */
	sp = &state;
	dp = &gp->acllist_array;
	for (i = 0, ap = gp->acllist; i < dp->len; ++i, ++ap) {
		for (j = dp->len, al = ap->acl; j > 0; --j, ++al) {
			if (!IN_PERMITHOSTPORTSEQRANGE(al->seq))
				continue;
			if (al->seq < gp->compactpermithostportseq)
				continue;
			if (al->type == ATYPE_UNKNOWN) {
				gp->compactpermithostportseq = 0;
				timerreset(&sp->t_compact);
				lg(LOG_ERR, "servercompactport: Can't handle"
				    " unknown ACL type (seq %d)", al->seq);
				return;
			}
			seq = gp->compactpermithostportseq;
			++gp->compactpermithostportseq;
			if (al->seq != seq) {
				/* XXX Check to see if it's in use? */
				servermoveacl(gp, al, seq);
				return;
			}

			/* XXX paranoid */
			if (gp->compactpermithostportseq >=
			    cf->c_highpermithostportseq)
				abort();
		}
	}

	/* If we got here we're done */
	gp->lastpermithostportseq = gp->compactpermithostportseq - 1;
	gp->compactpermithostportseq = 0;
}

void
servercompactport(struct aclgroup *gp)
{
	int i, j, seq;
	struct acllist *ap;
	struct acl *al;
	struct array *dp;
	struct state *sp;

	/* Port block range */
	sp = &state;
	dp = &gp->acllist_array;
	for (i = 0, ap = gp->acllist; i < dp->len; ++i, ++ap) {
		for (j = dp->len, al = ap->acl; j > 0; --j, ++al) {
			if (!IN_PORTSEQRANGE(al->seq))
				continue;
			if (al->seq < gp->compactportseq)
				continue;
			if (al->type == ATYPE_UNKNOWN) {
				gp->compactportseq = 0;
				timerreset(&sp->t_compact);
				lg(LOG_ERR, "servercompactport: Can't handle"
				    " unknown ACL type (seq %d)", al->seq);
				return;
			}
			seq = gp->compactportseq;
			++gp->compactportseq;
			if (al->seq != seq) {
				/* XXX Check to see if it's in use? */
				servermoveacl(gp, al, seq);
				return;
			}

			/* XXX paranoid */
			if (gp->compactportseq >= cf->c_highportseq)
				abort();
		}
	}

	/* If we got here we're done */
	gp->lastportseq = gp->compactportseq - 1;
	gp->compactportseq = 0;
}

/* XXX unfortunately, this duplicates a lot of childdroprestore() */
void
serverdroprestore(struct req *rp)
{
	struct aclgroup *gp;
	struct acl *al;
	char buf[32];

	/* XXX server should normally supply the sequence number */
	if (rp->acl.seq == 0) {
		lg(LOG_ERR, "serverdroprestore: seq number shouldn't be zero");
		rp->state = RSTATE_DONE;
		getts(&rp->cts);
		return;
	}

	/* Find ACL list */
	switch (rp->acl.type) {

	case ATYPE_BLOCKHOST:
	case ATYPE_BLOCKNET:
	case ATYPE_BLOCKHOSTHOST:
	case ATYPE_BLOCKTCPDSTHOSTPORT:
		/* By address */
		gp = aclfindgroupbyaddr(&rp->acl.addr1);
		if (gp == NULL) {
			lg(LOG_ERR,
			    "serverdroprestore: can't find aclgroup for %s",
			    addr2str(&rp->acl.addr1));
			rp->state = RSTATE_DONE;
			getts(&rp->cts);
			return;
		}
		rp->aclgroup = gp;
		break;

	case ATYPE_BLOCKUDPPORT:
	case ATYPE_BLOCKTCPPORT:
	case ATYPE_BLOCKTCPSYNPORT:
		/* By ACL name */
		gp = aclfindgroupbyname(rp->aclname);
		if (gp == NULL) {
			lg(LOG_ERR,
			    "serverdroprestore: can't find aclgroup for %s",
			    rp->acl.port1);
			rp->state = RSTATE_DONE;
			getts(&rp->cts);
			return;
		}
		rp->aclgroup = gp;
		break;

	case ATYPE_PERMITUDPDSTHOSTPORT:
	case ATYPE_PERMITTCPDSTHOSTPORT:
		/* Use default ACL list for permit guys */
		gp = aclfindgroupdefault(&rp->acl.addr1);
		if (gp == NULL) {
			lg(LOG_ERR,
			    "serverdroprestore: can't find aclgroup for %s",
			    addr2str(&rp->acl.addr1));
			rp->state = RSTATE_DONE;
			getts(&rp->cts);
			return;
		}
		rp->aclgroup = gp;
		break;

	default:
		lg(LOG_ERR, "ACL type %d not implemented", rp->acl.type);
		rp->state = RSTATE_DONE;
		getts(&rp->cts);
		return;
	}

	/*
	 * Since we're just changing the sequence number, and we
	 * don't remove the ACL, it should always exist.
	 */
	al = aclfindacl(gp, &rp->acl, 0);
	switch (rp->type) {

	case REQ_DROP:
	case REQ_RESTORE:
		if (al == NULL) {
			lg(LOG_ERR, "serverdroprestore:"
			    " %s wasn't already blocked in ACL %s!",
			    addr2str(&rp->acl.addr1), gp->name);
			rp->state = RSTATE_DONE;
			getts(&rp->cts);
			return;
		}
		break;

	case REQ_BLOCKHOSTHOST:
	case REQ_RESTOREHOSTHOST:
		(void)snprintf(buf, sizeof(buf), "%s",
		    addr2str(&rp->acl.addr2));
		if (al == NULL) {
			lg(LOG_ERR, "serverdroprestore:"
			    " %s/%s wasn't already blocked in ACL %s!",
			    addr2str(&rp->acl.addr1), buf, gp->name);
			rp->state = RSTATE_DONE;
			getts(&rp->cts);
			return;
		}
		break;

	case REQ_DROPUDPPORT:
	case REQ_DROPTCPPORT:
	case REQ_RESTOREUDPPORT:
	case REQ_RESTORETCPPORT:
	case REQ_DROPTCPSYNPORT:
	case REQ_RESTORETCPSYNPORT:
		if (al == NULL) {
			lg(LOG_ERR, "serverdroprestore: "
			    " port %s/%s wasn't already blocked in ACL %s!",
			    tcporudpreqstr(rp->type), rp->acl.port1, gp->name);
			rp->state = RSTATE_DONE;
			getts(&rp->cts);
			return;
		}
		break;

	case REQ_PERMITUDPDSTHOSTPORT:
	case REQ_PERMITTCPDSTHOSTPORT:
	case REQ_DROPTCPDSTHOSTPORT:
		if (al == NULL) {
			lg(LOG_ERR, "serverdroprestore:"
			    " %s:%s/%s wasn't already permitted in ACL %s!",
			    addr2str(&rp->acl.addr1), tcporudpreqstr(rp->type),
			    rp->acl.port1, gp->name);
			rp->state = RSTATE_DONE;
			getts(&rp->cts);
			return;
		}
		break;

	case REQ_UNPERMITUDPDSTHOSTPORT:
	case REQ_UNPERMITTCPDSTHOSTPORT:
	case REQ_RESTORETCPDSTHOSTPORT:
		if (al == NULL) {
			lg(LOG_ERR, "serverdroprestore:"
			    " %s:%s/%s wasn't already permitted in ACL %s!",
			    addr2str(&rp->acl.addr1), tcporudpreqstr(rp->type),
			    rp->acl.port1, gp->name);
			rp->state = RSTATE_DONE;
			getts(&rp->cts);
			return;
		}
		break;

	default:
		lg(LOG_ERR, "serverdroprestore: bad type %s (1)",
		    val2str(str2acl, al->type));
		exit(EX_SOFTWARE);
	}

	/* No need to check the sequence number further, just check the type */
	switch (rp->type) {

	case REQ_DROP:
	case REQ_RESTORE:
	case REQ_BLOCKHOSTHOST:
	case REQ_RESTOREHOSTHOST:
	case REQ_DROPTCPPORT:
	case REQ_RESTORETCPPORT:
	case REQ_DROPUDPPORT:
	case REQ_RESTOREUDPPORT:
	case REQ_DROPTCPSYNPORT:
	case REQ_RESTORETCPSYNPORT:
	case REQ_PERMITUDPDSTHOSTPORT:
	case REQ_UNPERMITUDPDSTHOSTPORT:
	case REQ_PERMITTCPDSTHOSTPORT:
	case REQ_UNPERMITTCPDSTHOSTPORT:
	case REQ_DROPTCPDSTHOSTPORT:
	case REQ_RESTORETCPDSTHOSTPORT:
		break;

	default:
		lg(LOG_ERR, "serverdroprestore: bad type %s (2)",
		    val2str(str2acl, al->type));
		exit(EX_SOFTWARE);
	}

	/* Send the request to the child */
	(state.f_childrequest)(gp, rp);

	// XXX sp->f_childrequest2

	/* Update request state */
	rp->state = RSTATE_CHILD;
}

void
servermoveacl(struct aclgroup *gp, struct acl *al, int seq)
{
	struct req *rp;
	int remreq, addreq;
	char compactstr[] = "compact";
	struct state *sp;

	sp = &state;
	switch (al->type) {

	case ATYPE_BLOCKHOST:
		remreq = REQ_RESTORE;
		addreq = REQ_DROP;
		break;

	case ATYPE_BLOCKNET:
		remreq = REQ_RESTORE;
		addreq = REQ_DROP;
		break;

	case ATYPE_BLOCKHOSTHOST:
		remreq = REQ_RESTOREHOSTHOST;
		addreq = REQ_BLOCKHOSTHOST;
		break;

	case ATYPE_BLOCKUDPPORT:
		remreq = REQ_RESTOREUDPPORT;
		addreq = REQ_DROPUDPPORT;
		break;

	case ATYPE_BLOCKTCPPORT:
		remreq = REQ_RESTORETCPPORT;
		addreq = REQ_DROPTCPPORT;
		break;

	case ATYPE_BLOCKTCPSYNPORT:
		remreq = REQ_RESTORETCPSYNPORT;
		addreq = REQ_DROPTCPSYNPORT;
		break;

	case ATYPE_PERMITUDPDSTHOSTPORT:
		remreq = REQ_UNPERMITUDPDSTHOSTPORT;
		addreq = REQ_PERMITUDPDSTHOSTPORT;
		break;

	case ATYPE_PERMITTCPDSTHOSTPORT:
		remreq = REQ_UNPERMITTCPDSTHOSTPORT;
		addreq = REQ_PERMITTCPDSTHOSTPORT;
		break;

	case ATYPE_BLOCKTCPDSTHOSTPORT:
		remreq = REQ_RESTORETCPDSTHOSTPORT;
		addreq = REQ_DROPTCPDSTHOSTPORT;
		break;

	default:
		lg(LOG_ERR, "servermoveacl: bad type %s",
		    val2str(str2acl, al->type));
		exit(EX_SOFTWARE);
	}

	rp = newreq("servermoveacl req 1");
	rp->state = RSTATE_PENDING;
	rp->type = remreq;
	rp->cmd = val2str(cmd2req, rp->type);
	rp->acl = *al;
	rp->aclname = strsave(gp->name);
	aclcopy(&rp->acl, al);
	rp->payload.buf = compactstr;
	IOBUF_LEN(&rp->payload) = sizeof(compactstr) - 1;
	IOBUF_SIZE(&rp->payload) = 0;		/* don't free() */
	getts(&rp->ats);
	appendreq(&sp->req, rp);

	/* Change the sequence number */
	al->seq = seq;

	rp = newreq("servermoveacl req 2");
	rp->state = RSTATE_PENDING;
	rp->type = addreq;
	rp->cmd = val2str(cmd2req, rp->type);
	rp->acl = *al;
	rp->aclname = strsave(gp->name);
	aclcopy(&rp->acl, al);
	rp->payload.buf = compactstr;
	IOBUF_LEN(&rp->payload) = sizeof(compactstr) - 1;
	IOBUF_SIZE(&rp->payload) = 0;		/* don't free() */
	getts(&rp->ats);
	appendreq(&sp->req, rp);
}

void
serverprocess(struct req *rp)
{

	switch (rp->type) {

	case REQ_DROP:
	case REQ_RESTORE:
	case REQ_BLOCKHOSTHOST:
	case REQ_RESTOREHOSTHOST:
	case REQ_DROPUDPPORT:
	case REQ_RESTOREUDPPORT:
	case REQ_DROPTCPPORT:
	case REQ_RESTORETCPPORT:
	case REQ_DROPTCPSYNPORT:
	case REQ_RESTORETCPSYNPORT:
	case REQ_PERMITUDPDSTHOSTPORT:
	case REQ_UNPERMITUDPDSTHOSTPORT:
	case REQ_PERMITTCPDSTHOSTPORT:
	case REQ_UNPERMITTCPDSTHOSTPORT:
	case REQ_DROPTCPDSTHOSTPORT:
	case REQ_RESTORETCPDSTHOSTPORT:
		serverdroprestore(rp);
		break;

	case REQ_AYT:
	case REQ_SYNC:
		childsend("%s", rp->cmd);
		rp->state = RSTATE_CHILD;
		break;

	default:
		lg(LOG_ERR, "serverprocess: unhandled request type %d",
		    rp->type);
		exit(EX_SOFTWARE);
	}
}

void
serversync(void)
{
	struct req *rp;
	struct state *sp;

	sp = &state;
	rp = newreq("serversync req");
	rp->state = RSTATE_PENDING;
	rp->type = REQ_SYNC;
	rp->cmd = "sync";

	/* Arrival timestamp */
	getts(&rp->ats);

	/* Append the new request to end of list */
	appendreq(&sp->req, rp);

	/* Cancel possible pending "ayt" */
	timerreset(&sp->t_ayt);
}
