/*
 * Copyright (c) 2018, 2019, 2020
 * The Regents of the University of California. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *     * Redistributions of source code must retain the above copyright
 *       notice, this list of conditions and the following disclaimer.
 *     * Redistributions in binary form must reproduce the above copyright
 *       notice, this list of conditions and the following disclaimer in the
 *       documentation and/or other materials provided with the distribution.
 *     * Neither the name of the University nor the names of its contributors
 *       may be used to endorse or promote products derived from this software
 *       without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#ifndef lint
static const char rcsid[] __attribute__ ((unused)) =
    "@(#) $Id$ (LBL)";
#endif

#include <sys/types.h>
#include <sys/time.h>

#include <errno.h>
// #include <signal.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
// #include <sysexits.h>
#include <unistd.h>

#include <openssl/ssl.h>
#include <openssl/err.h>
#include <openssl/x509_vfy.h>

#include "acld.h"
#include "sslutil.h"

/* Return 1 on success and 0 on EOF or error */
int
sslhandleerr(struct sslclient *sup, int r, const char *what)
{
	int err;

	if (r <= 0) {
		*sup->rfdp = -1;
		*sup->wfdp = -1;
		err = SSL_get_error(sup->ssl, r);
		switch (err) {

		case SSL_ERROR_WANT_READ:
			*sup->rfdp = sup->s;
			break;

		case SSL_ERROR_WANT_WRITE:
			*sup->wfdp = sup->s;
			break;

		case SSL_ERROR_ZERO_RETURN:
			/* The other side went away */
			lg(LOG_ERR, "%s: %s: peer gone", sup->name, what);
			return (0);

		case SSL_ERROR_SSL:
			lg(LOG_ERR, "%s: %s: SSL failure", sup->name, what);
			return (0);

		case SSL_ERROR_SYSCALL:
			lg(LOG_ERR, "%s: %s: %s",
			    sup->name, what, strerror(errno));
			return (0);

		default:
			lg(LOG_ERR, "%s: %s: unknown err: %s (%d)",
			    sup->name, what, ERR_error_string(err, NULL), err);
			return (0);
		}
		return (1);
	}

// lg(LOG_DEBUG, "%s: %s: read %d", chp->name, what, r);

	return (1);
}

/*
 * Read from a socket and append to an iobuf
 *
 * Return -1 if there was a fatal error, 0 on EOF or else the iobuf len
 */
int
sslread(SSL * ssl, struct iobuf *ip)
{
	int r;
	struct array *dp;

	dp = &ip->io_array;
	DYNARRAY(dp, ip->buf, dp->len + 1, "ioread");
	ERR_clear_error();
	r = SSL_read(ssl, ip->buf + IOBUF_LEN(ip),
	    IOBUF_SIZE(ip) - IOBUF_LEN(ip) - 1);
	if (r <= 0)
		return (r);

// fprintf(stderr, "sslread: \"%.*s\"\n", r, ip->buf + IOBUF_LEN(ip));
	IOBUF_LEN(ip) += r;
	ip->buf[IOBUF_LEN(ip)] = '\0';
	return (IOBUF_LEN(ip));
}

/* Write from an iobuf to a socket */
int
sslwrite(SSL *ssl, struct iobuf *ip)
{
	ssize_t cc;

	if (IOBUF_LEN(ip) == 0) {
		lg(LOG_ERR, "sslwrite: len is zero");
		abort();
	}
	ERR_clear_error();
	cc = SSL_write(ssl, ip->buf, IOBUF_LEN(ip));

	/* Use sslhandleerr() to handle errors */
	if (cc <= 0)
		return (cc);
	if (cc > IOBUF_LEN(ip)) {
		lg(LOG_ERR, "sslwrite: cc > ip-len, %d > %d (Can't happen?)",
		    (int)cc, (int)IOBUF_LEN(ip));
		cc = IOBUF_LEN(ip);
	}
// fprintf(stderr, "sslwrite: send: \"%.*s\"\n", (int)cc, ip->buf);
	if (cc == IOBUF_LEN(ip)) {
		IOBUF_LEN(ip) = 0;
		return (cc);
	}

	/* Move leftover chars on over */
	memmove(ip->buf, ip->buf + cc, IOBUF_LEN(ip) - cc);
	IOBUF_LEN(ip) -= cc;
	return (cc);
}
