/*
 * Copyright (c) 2007, 2008, 2010, 2011, 2012, 2013, 2019, 2020, 2024
 * The Regents of the University of California. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *     * Redistributions of source code must retain the above copyright
 *       notice, this list of conditions and the following disclaimer.
 *     * Redistributions in binary form must reproduce the above copyright
 *       notice, this list of conditions and the following disclaimer in the
 *       documentation and/or other materials provided with the distribution.
 *     * Neither the name of the University nor the names of its contributors
 *       may be used to endorse or promote products derived from this software
 *       without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#ifndef lint
static const char rcsid[] __attribute__ ((unused)) =
    "@(#) $Id$ (LBL)";
#endif

#include <sys/types.h>

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "acld.h"

static long now = 0;

/* Forwards */
static int stats_bucket(struct stats *);
static void stats_now(void);
static void stats_updateptrs(struct stats *);

/* Return index to bucket defined by the specificed time */
static int
stats_bucket(struct stats *st)
{

	return ((now % (st->len * st->secs)) / st->secs);
}

void
stats_free(struct stats *st)
{

	if (st->buckets != NULL)
		free(st->buckets);
	memset(st, 0, sizeof(*st));
}

int
stats_gethiwater(struct stats *st)
{
	int i, hiwater;
	struct bucket *p;

	if (st->buckets == NULL)
		return (0);

	/* Update pointers and potentially zero out stale data */
	stats_updateptrs(st);

	hiwater = 0;
	for (i = 0, p = st->buckets; i < st->len; ++i, ++p)
		if (hiwater < p->hiwater)
			hiwater = p->hiwater;
	return (hiwater);
}

/* Returns rate in packets/min */
int
stats_getrate(struct stats *st, const char *what)
{
	int i, count, secs;
	struct bucket *p;

	if (st->buckets == NULL)
		return (0);

	/* Update pointers and potentially zero out stale data */
	stats_updateptrs(st);

	/*
	 * Calculate the seconds in current bucket plus seconds for
	 * the other buckets
	 */
	secs = ((now % (st->len * st->secs)) % st->secs) +
	    ((st->len - 1) * st->secs);

	count = 0;
	for (i = 0, p = st->cur; i < st->len; ++i, ++p) {
		if (p >= st->buckets + st->len)
			p = st->buckets;
		count += p->count;
	}
#ifdef notdef
	lg(LOG_DEBUG, "stats_getrate: %s %d/%d seconds (%d/(%d/60) == %f/min)",
	    what,
	    secs, (int)st->len * st->secs,
	    count, secs,
	    ((secs == 0) ? 0.0 : (double)count / (((double)secs) / 60.0)));
#endif
	return ((secs == 0) ? 0 : (count * 60) / secs);
}

/*
 * Passed in the number of buckets and width in seconds of each
 */
void
stats_init(struct stats *st, size_t len, int secs, const char *what)
{
	if (len < 1)
		abort();
	stats_free(st);
	st->len = len;
	st->secs = secs;
	st->buckets = new(len, sizeof(*st->buckets), what);
	st->cur = st->buckets;
	stats_now();
	st->cur += stats_bucket(st);
	st->lasttime = now;
}

static void
stats_now(void)
{
	struct timeval tv;

	getts(&tv);
	now = tv.tv_sec;
}

/* This gets called when an an addition is made */
void
stats_sethiwater(struct stats *st, int len)
{

	/* Update pointers and potentially zero out stale data */
	stats_updateptrs(st);
	st->lastvalue = len;
	if (st->cur->hiwater < len)
		st->cur->hiwater = len;
}

/* This gets called when a block or nullzero is added */
void
stats_setrate(struct stats *st)
{

	/* Update pointers and potentially zero out stale data */
	stats_updateptrs(st);
	++st->cur->count;
}

/* This gets called when an an deletion is made */
void
stats_setvalue(struct stats *st, int len)
{

	/* Update pointers and potentially zero out stale data */
	stats_updateptrs(st);
	st->lastvalue = len;
}

static void
stats_updateptrs(struct stats *st)
{
	int i;
	struct bucket *p;

	/* XXX if now - lasttime is big enough we should zero everything */
	stats_now();
	i = stats_bucket(st);
	p = st->buckets + i;
	while (p != st->cur) {
		++st->cur;
		if (st->cur >= st->buckets + st->len)
			st->cur = st->buckets;
		st->cur->count = 0;
		st->cur->hiwater = st->lastvalue;
	}
	st->lasttime = now;
}
