/*
 * Copyright (c) 2019, 2020
 * The Regents of the University of California. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *     * Redistributions of source code must retain the above copyright
 *       notice, this list of conditions and the following disclaimer.
 *     * Redistributions in binary form must reproduce the above copyright
 *       notice, this list of conditions and the following disclaimer in the
 *       documentation and/or other materials provided with the distribution.
 *     * Neither the name of the University nor the names of its contributors
 *       may be used to endorse or promote products derived from this software
 *       without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#ifndef lint
static const char copyright[] =
    "@(#) Copyright (c) 2019, 2020\n\
The Regents of the University of California.  All rights reserved.\n";
static const char rcsid[] =
    "@(#) $Id$ (LBL)";
#endif

/*
 * test1 - Tests and validation
 */

#include <sys/types.h>
#include <sys/time.h>
#include <sys/wait.h>

#include <arpa/inet.h>

#include <errno.h>
#include <fcntl.h>
#include <signal.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sysexits.h>
#include <time.h>
#include <unistd.h>

#include "gnuc.h"
#ifdef HAVE_OS_PROTO_H
#include "os-proto.h"
#endif

#include "acld.h"
#include "cf.h"
#include "child.h"
#include "child2.h"
#include "server.h"
#include "setsignal.h"
#include "util.h"

#define ASSERT(v, m) { \
    ++tests; \
    if (!(v)) { \
	    fprintf(stderr, "%s:%d error: %s\n", __FILE__, __LINE__, m); \
	    ++errors; \
    } \
    }

#define PLURAL(v) ((v) == 1 ? "" : "s")

/* Globals */
int debug;
int verbose;
const char *prog;
struct cf *cf;
struct state state;

/* Locals */
static int errors;
static int tests;

/* Externs */
extern int optind;
extern int opterr;
extern char *optarg;

/* Forwards */
int main(int, char **);
void test_io(void);
void test_util(void);
__dead void usage(void) __attribute__((noreturn));

int
main(int argc, char **argv)
{
	int op;
	char *cp;

	if (argv[0] == NULL)
		prog = "acld";
	else if ((cp = strrchr(argv[0], '/')) != NULL)
		prog = cp + 1;
	else
		prog = argv[0];

	opterr = 0;
	while ((op = getopt(argc, argv, "dv")) != EOF)
		switch (op) {

		case 'd':
			++debug;
			break;

		case 'v':
			++verbose;
			break;

		default:
			usage();
			/* NOTREACHED */
		}

	if (argc != optind)
		usage();

	printf("%s: version %s starting\n", prog, version);

	test_io();
	test_util();

	printf("%s: exiting (%d test%s, %d error%s)\n",
	    prog, tests, PLURAL(tests), errors, PLURAL(errors));
	exit(errors > 0);
}


/* Test io.c routines */
void
test_io()
{
	struct iobuf *ip;
	struct iobuf iobuf;
	const char *what;
	const char *s1 = "198.51.100.188 not found";
	const char *s2 = "198.51.100.188";
	const char *s3 = "198.51.100.188 not found\r\n198.51.100.188\r\n";
	const char *s4 = "198.51.100.188\r\n";

	ip = &iobuf;
	memset(ip, 0, sizeof(*ip));
	IOBUF_INIT(ip);

	/* Two different lines */
	ioappendline(ip, s1);
	ioappendline(ip, s2);
	what = "ioappendline failure";
	ASSERT(strcmp(ip->buf, s3) == 0, what);

	/* Consolidate two different lines */
	ioconsolidate(ip);
	what = "ioconsolidate failure 1";
	ASSERT(strcmp(ip->buf, s3) == 0, what);

	/* Consolidate one line */
	iofree(ip);
	ioappendline(ip, s2);
	ioconsolidate(ip);
	what = "ioconsolidate failure 2";
	ASSERT(strcmp(ip->buf, s4) == 0, what);

	/* Consolidate two same lines */
	iofree(ip);
	ioappendline(ip, s2);
	ioappendline(ip, s2);
	ioconsolidate(ip);
	what = "ioconsolidate failure 3";
	ASSERT(strcmp(ip->buf, s4) == 0, what);

	/* Consolidate three same lines */
	iofree(ip);
	ioappendline(ip, s2);
	ioappendline(ip, s2);
	ioappendline(ip, s2);
	ioconsolidate(ip);
	what = "ioconsolidate failure 4";
	ASSERT(strcmp(ip->buf, s4) == 0, what);

	/* Consolidate four same lines */
	iofree(ip);
	ioappendline(ip, s2);
	ioappendline(ip, s2);
	ioappendline(ip, s2);
	ioappendline(ip, s2);
	ioconsolidate(ip);
	what = "ioconsolidate failure 4";
	ASSERT(strcmp(ip->buf, s4) == 0, what);

	/* Cleanup */
	iofree(ip);
}

void
test_util()
{
	const char *what;
	const char *s1 = "1591397820.610427 ";
	const char *s2 = "1591397820.610427";
	const char *s3 = "1591397820.610427\rjunk";
	struct timeval tv;

	tv.tv_sec = 0;
	tv.tv_usec = 0;

	extracttimeval(&tv, s1);
	what = "extracttimeval failure";
	ASSERT(tv.tv_sec == 0 && tv.tv_usec == 0, what);

	extracttimeval(&tv, s2);
	what = "extracttimeval success 1";
	ASSERT(tv.tv_sec == 1591397820 && tv.tv_usec == 610427, what);

	extracttimeval(&tv, s3);
	what = "extracttimeval success 2";
	ASSERT(tv.tv_sec == 1591397820 && tv.tv_usec == 610427, what);
}

__dead void
usage(void)
{
	fprintf(stderr, "%s version %s\n", prog, version);
	fprintf(stderr, "Usage: %s [-dv]\n", prog);
	exit(EX_USAGE);
}
