/*
 * Copyright (c) 2002, 2003, 2004, 2005, 2006, 2008, 2009, 2010, 2011, 2012, 2013, 2014, 2015, 2016, 2017, 2018, 2019, 2020, 2021, 2022, 2023, 2024
 * The Regents of the University of California. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *     * Redistributions of source code must retain the above copyright
 *       notice, this list of conditions and the following disclaimer.
 *     * Redistributions in binary form must reproduce the above copyright
 *       notice, this list of conditions and the following disclaimer in the
 *       documentation and/or other materials provided with the distribution.
 *     * Neither the name of the University nor the names of its contributors
 *       may be used to endorse or promote products derived from this software
 *       without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#ifndef lint
static const char rcsid[] __attribute__ ((unused)) =
    "@(#) $Id$ (LBL)";
#endif

#include <sys/types.h>
#include <sys/time.h>

#include <arpa/inet.h>

#include <ctype.h>
#include <errno.h>
#include <fcntl.h>
#include <netdb.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sysexits.h>
#include <unistd.h>

#include "acld.h"

/* Globals */
struct log log_lf;

/* Locals */
struct s2v syslog2str[] = {
	{ "auth",	LOG_AUTH, },
	{ "authpriv",	LOG_AUTHPRIV, },
#ifdef LOG_CONSOLE
	{ "console",	LOG_CONSOLE, },
#endif
	{ "cron",	LOG_CRON, },
	{ "daemon",	LOG_DAEMON, },
	{ "ftp",	LOG_FTP, },
	{ "kern",	LOG_KERN, },
	{ "lpr",	LOG_LPR, },
	{ "mail",	LOG_MAIL, },
	{ "news",	LOG_NEWS, },
#ifdef LOG_NTP
	{ "ntp",	LOG_NTP, },
#endif
#ifdef LOG_SECURITY
	{ "security",	LOG_SECURITY, },
#endif
	{ "syslog",	LOG_SYSLOG, },
	{ "user",	LOG_USER, },
	{ "uucp",	LOG_UUCP, },
	{ "local0",	LOG_LOCAL0, },
	{ "local1",	LOG_LOCAL1, },
	{ "local2",	LOG_LOCAL2, },
	{ "local3",	LOG_LOCAL3, },
	{ "local4",	LOG_LOCAL4, },
	{ "local5",	LOG_LOCAL5, },
	{ "local6",	LOG_LOCAL6, },
	{ "local7",	LOG_LOCAL7, },
	{ NULL,		-1, }
};

static struct addr loopback4 = {
	AF_INET,
	{ { htonl(INADDR_LOOPBACK & 0xFF000000) } }, { { htonl(0xFF000000) } }
};

static struct addr loopback6 = {
	AF_INET6,
	{
	    ._addr6 = {
		.__u6_addr = {
		    .__u6_addr8 = {
			0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1
		    }
		}
	    }
	},
	{
	    ._mask6 = {
		.__u6_addr = {
		    .__u6_addr32 = {
			0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff,
		    }
		}
	    }
	},
};

socklen_t
addr2sa(struct addr *a, u_short port, struct sockaddr *sap)
{
	socklen_t salen;
	struct sockaddr_in *s4p;
	struct sockaddr_in6 *s6p;

	switch (a->family) {

	case AF_INET:
		s4p = (struct sockaddr_in *)sap;
		salen = sizeof(*s4p);
		memset(s4p, 0, salen);
		s4p->sin_port = htons(port);
		memmove(&s4p->sin_addr, &a->addr4, sizeof(s4p->sin_addr));
		break;

	case AF_INET6:
		s6p = (struct sockaddr_in6 *)sap;
		salen = sizeof(*s6p);
		memset(s6p, 0, salen);
		s6p->sin6_port = htons(port);
		memmove(&s6p->sin6_addr, &a->addr6, sizeof(s6p->sin6_addr));
		break;

	default:
		abort();
	}

#ifdef HAVE_SOCKADDR_SA_LEN
	sap->sa_len = salen;
#endif
	sap->sa_family = a->family;

	return (salen);
}

const char *
addr2str(const struct addr *addr)
{
	int w;
	size_t len, size;
	char *cp;
	static char buf[128];

	w = maskwidth(addr);
	switch (addr->family) {

	case AF_INET:
		if (inet_ntop(addr->family, &addr->addr4,
		    buf, sizeof(buf)) == NULL) {
			lg(LOG_ERR, "addr2str: v4 botch");
			abort();
		}
		if (w == 32)
			return (buf);
		break;

	case AF_INET6:
		if (inet_ntop(addr->family, &addr->addr6,
		    buf, sizeof(buf)) == NULL) {
			lg(LOG_ERR, "addr2str: v6 botch");
			abort();
		}
		if (w == 128)
			return (buf);
		break;

	default:
		(void)snprintf(buf, sizeof(buf), "AF_#%u-unsupported",
		    addr->family);
		return (buf);
	}

	/* Append address mask width */
	cp = buf;
	len = strlen(cp);
	cp += len;
	size = sizeof(buf) - len;
	(void)snprintf(cp, size, "/%d", w);
	return (buf);
}

int
blocking(int s, int block)
{
	int flags;

	if ((flags = fcntl(s, F_GETFL, 0)) < 0) {
		lg(LOG_ERR, "blocking: F_GETFL: %s", strerror(errno));
		return (-1);
	}
	if (block)
		flags &= ~O_NONBLOCK;
	else
		flags |= O_NONBLOCK;
	if ((flags = fcntl(s, F_SETFL, flags)) < 0) {
		lg(LOG_ERR, "blocking: F_SETFL: %s", strerror(errno));
		return (-1);
	}
	return (0);
}

/* Return and error message if there is a problem */
const char *
checklimits(struct aclgroup *gp)
{
	int i, total;
	struct array *dp;
	struct aclgroup *gp2;
	struct acllimit *alp;
	static char buf[128];

	/* Check cap on total number of ACLs */
	if (cf->c_maxseq > 0) {
		total = 0;
		dp = &cf->c_aclgroup_array;
		for (i = 0, gp2 = cf->c_aclgroup; i < dp->len; ++i, ++gp2)
			total += gp2->acltotallen;
		if (total >= cf->c_maxseq) {
			(void)snprintf(buf, sizeof(buf),
			    "No free ACL slots (%d in use, %d allowed)",
			    total, (int)cf->c_maxseq);
			return (buf);
		}
	}

	/* Check limit on ACLs group */
	if ((alp = gp->limitp) != NULL && alp->a_len >= alp->a_maxseq) {
		(void)snprintf(buf, sizeof(buf),
		    "No free slots in ACL %s (group %s)"
		    " (%d in use, %d allowed)",
		    gp->name,
		    fmtargv(alp->a_nacls, alp->a_acls),
		    alp->a_len,
		    alp->a_maxseq);
		return (buf);
	}
	return (NULL);
}

/* Reopen logfile if necessary */
void
checklog(struct log *lgp)
{
	struct stat sbuf2;

	/* Bail if not managing a logfile */
	if (lgp == NULL || lgp->fn == NULL)
		return;

	/* Do checks if not first time through */
	if (lgp->f != NULL) {
		/* Done if the file info matches */
		if (stat(lgp->fn, &sbuf2) >= 0 &&
		    sbuf2.st_ino == lgp->sbuf.st_ino &&
		    sbuf2.st_dev == lgp->sbuf.st_dev)
			return;

		/* Close file */
		if (fclose(lgp->f) == EOF)
			syslog(LOG_ERR, "checklog: fclose: %s: %m", lgp->fn);
		lgp->f = NULL;
	}

	/* Open for append in case logfile already exists */
	lgp->f = fopen(lgp->fn, "a");
	if (lgp->f == NULL) {
		fprintf(stderr, "%s: fopen: %s: %s\n",
		    prog, lgp->fn, strerror(errno));
		syslog(LOG_ERR, "checklog: fopen: %s: %m", lgp->fn);
		exit(EX_OSERR);
	}

	/* Close on exec */
	if (fcntl(fileno(lgp->f), F_SETFD, FD_CLOEXEC) < 0) {
		fprintf(stderr, "%s: fcntl FD_CLOEXEC: %s: %s\n",
		    prog, lgp->fn, strerror(errno));
		syslog(LOG_ERR, "fcntl FD_CLOEXEC: %s: %m", lgp->fn);
		exit(EX_OSERR);
	}

	/* Line buffering */
	setlinebuf(lgp->f);

	/* Sock away stat() info */
	if (fstat(fileno(lgp->f), &lgp->sbuf) < 0) {
		fprintf(stderr, "%s: fstat: %s: %s\n",
		    prog, lgp->fn, strerror(errno));
		syslog(LOG_ERR, "checklog: fstat: %s: %m", lgp->fn);
	}
}

const char *
checkmaskwidth(const struct addr *a)
{
	int w;
	static char buf[64];

	w = (a->family == AF_INET) ? cf->c_ipv4_maxwidth : cf->c_ipv6_maxwidth;
	if (maskwidth(a) < w) {
		(void)snprintf(buf, sizeof(buf),
		    "Subnet mask wider than /%d not supported", w);
		return (buf);
	}
	return (NULL);
}

/*
 *	dp	- pointer to dynamic array struct
 *	p	- pointer to dynamic buffer pointer
 *	newlen	- desired new total length (objects)
 *	what	- string for error message
 */
void *
dynarray(struct array *dp, void *p, size_t newlen, const char *what)
{
	size_t size;

	if (newlen == 0) {
		lg(LOG_ERR, "dynarray: %s: newlen is zero (%d/%d/%d/%d)",
		    what, (int)dp->osize, (int)dp->inclen,
		    (int)dp->len, (int)dp->size);
		exit(EX_SOFTWARE);
	}
	if (dp->osize == 0) {
		lg(LOG_ERR, "dynarray: %s: osize is zero (%d/%d/%d/%d)",
		    what, (int)dp->osize, (int)dp->inclen,
		    (int)dp->len, (int)dp->size);
		exit(EX_SOFTWARE);
	}

	/* Initial allocation */
	if (p == NULL) {
		if (dp->size < newlen)
			dp->size = newlen;
		if (dp->size < dp->inclen)
			dp->size = dp->inclen;
		p = new(dp->size, dp->osize, what);
		return (p);
	}

	/* Want more space (realloc) */
	if (dp->size < newlen) {
		size = newlen + dp->inclen;
		if (dp->size > 0 && dp->size == size) {
			lg(LOG_ERR,
			    "dynarray: %s: bad realloc() size (%d/%d/%d/%d)",
			    what, (int)dp->osize, (int)dp->inclen,
			    (int)dp->len, (int)dp->size);
			exit(EX_SOFTWARE);
		}
		p = realloc(p, size * dp->osize);
		if (p == NULL) {
			lg(LOG_ERR, "dynarray: realloc %s: %s",
			    what, strerror(errno));
			exit(EX_OSERR);
		}
		memset(((char *)p) + dp->len * dp->osize, 0,
		    (size - dp->len) * dp->osize);
		dp->size = size;
	}
	return (p);
}

const char *
escapestr(const char *str)
{
	char *cp;
	const char *p;
	size_t n;
	static char *result = NULL;
	static char escapes[] = "$[]{}\"\\";

	if (result != NULL) {
		free(result);
		result = NULL;
	}

	/* Return a expect style null */
	if (str == NULL || *str == '\0')
		return ("{}");

	/* Enough space for string and EOS */
	n = strlen(str) + 1;

	/* Add room for escapes */
	for (p = str; *p != '\0'; ++p)
		if (strchr(escapes, *p) != NULL)
			++n;

	result = malloc(n);
	if (result == NULL) {
		fprintf(log_lf.f, "%s: escapestr: malloc: %s",
		    prog, strerror(errno));
		exit(EX_OSERR);
	}
	p = str;
	cp = result;
	while (*p != '\0') {
		if (strchr(escapes, *p) != NULL)
			*cp++ = '\\';
		*cp++ = *p++;
	}
	*cp = '\0';

	return (result);
}

/* Extract an IPv4 or IPv6 host or network address */
const char *
extractaddr(const char *s1, const char *s2, struct addr *a)
{
	int i, w;
	long x;
	char *cp, *ep;
	const char *p;
	char temp[64];

	memset(a, 0, sizeof(*a));

	/* Let's see what we've got here */
	if (strchr(s1, '.') != NULL) {
		a->family = AF_INET;
		w = 32;
	} else if (strchr(s1, ':') != NULL) {
		a->family = AF_INET6;
		w = 128;
	} else
		return ("unrecognized address type");

	p = strchr(s1, '/');
	if (p != NULL) {
		/* Mask length was specified */
		if (s2 != NULL)
			return ("garbage following mask width");

		strlcpy(temp, s1, sizeof(temp));
		cp = strchr(temp, '/');
		if (cp == NULL)
			abort();
		*cp++ = '\0';
		ep = NULL;
		x = strtol(cp, &ep, 10);
		if (*ep != '\0')
			return ("garbage following mask width");
		s1 = temp;
		if (x > w)
			return ("mask too wide");
		w = x;
	}

	switch (a->family) {

	case AF_INET:
		if (!inet_pton(a->family, s1, &a->addr4))
			return ("cannot parse IPv4 address");
		if (s2 != NULL) {
			if (!inet_pton(a->family, s2, &a->mask4))
				return ("cannot parse IPv4 address mask");
			w = maskwidth(a);
		}

		if (w > 32)
			return ("mask length must be <= 32");
		setmaskwidth(w, a);

		if ((a->addr4 & ~a->mask4) != 0)
			return ("non-network bits set in addr");

#ifdef notdef
		if ((ntohl(a->addr4) & 0xff000000) == 0)
			return ("high octet must be non-zero");
#endif
		break;

	case AF_INET6:
		if (!inet_pton(a->family, s1, &a->addr6))
			return ("cannot parse IPv6 address");
		if (s2 != NULL) {
			if (!inet_pton(a->family, s2, &a->mask6))
				return ("cannot parse IPv6 address mask");
			w = maskwidth(a);
		}
		if (w > 128)
			return ("mask length must be <= 128");
		setmaskwidth(w, a);

		for (i = 0; i < 16; ++i) {
			if ((a->addr6[i] & ~a->mask6[i]) != 0)
				return ("non-network bits set in addr");
		}
		break;

	default:
		abort();
	}

	return (NULL);
}

/* Extract an IPv4 or IPv6 host address */
const char *
extracthost(const char *s1, const char *s2, struct addr *a)
{
	const char *p;

	p = extractaddr(s1, s2, a);
	if (p == NULL && !ISHOST(a))
		p = "must be an address, not a network";
	return (p);
}

/* Extract a timestamp (no update if not parsable) */
void
extracttimeval(struct timeval *tvp, const char *s)
{
	u_long secs;
	u_long usecs;
	char *ep;

	secs = strtoul(s, &ep, 10);
	if (*ep != '.')
		return;
	s = ep + 1;
	usecs = strtoul(s, &ep, 10);
	if (*ep != '\0' && *ep != '\r' && *ep != '\n')
		return;
	tvp->tv_sec = secs;
	tvp->tv_usec = usecs;
}

int
failed(const char *what, const char *str)
{
	size_t len;

	len = strlen(what);
	if (strncmp(what, str, len) == 0 && strcmp(str + len, "-failed") == 0)
		return (1);
	return (0);
}

const char *
family2protostr(int family)
{
	switch (family) {

	case AF_INET:
		return ("IPv4");

	case AF_INET6:
		return ("IPv6");

	default:
		abort();
	}
}

const char *
fmtargv(int ac, char **av)
{
	int i, cc;
	static char buf[256];
	char *cp;
	size_t size;

	cp = buf;
	size = sizeof(buf);

	for (i = 0; i < ac; ++i) {
		if (i > 0 && size > 1) {
			*cp++ = ' ';
			--size;
		}
		cc = snprintf(cp, size, "%s", av[i]);
		if (cc < 0)
			abort();
		cp += cc;
		size -= cc;
	}
	*cp = '\0';
	return (buf);
}

void
freeargv(char **av)
{
	char **av2;

	if (av != NULL) {
		av2 = av;
		while (*av2 != NULL)
			free(*av2++);
		free(av);
	}
}

void *
freedynarray(struct array *dp, void *p)
{
	if (p != NULL) {
		if (dp->size != 0)
			free(p);
		p = NULL;
	}
	dp->len = 0;
	dp->size = 0;
	return (p);
}

void
getts(struct timeval *tsp)
{

	if (gettimeofday(tsp, NULL) < 0) {
		/* Can't use lg() since it can indirectly call us */
		syslog(LOG_ERR, "getts: gettimeofday: %s", strerror(errno));
		exit(EX_OSERR);
	}
}

int
insubnet(const struct addr *net, const struct addr *a)
{
	int i;

	if (net->family != a->family)
		return (0);

	switch (net->family) {

	case AF_INET:
		return (net->addr4 == (a->addr4 & net->mask4));

	case AF_INET6:
		for (i = 0; i < 16; ++i)
			if (net->addr6[i] != (a->addr6[i] & net->mask6[i]))
				return (0);
		return (1);

	default:
		abort();
	}
}

/*
 * Return true if the addr is a default for its address family
 * This means all address and maskbits are zero
 */
int
isdefaultaddr(struct addr *a)
{
	int i;

	switch (a->family) {

	case AF_INET:
		return (a->addr4 == 0 && a->mask4 == 0);

	case AF_INET6:
		for (i = 0; i < 16; ++i)
			if (a->addr6[i] != 0 || a->mask6[i] != 0)
				return (0);

		/* If we got here, all bytes were zero */
		return (1);

	default:
		abort();
	}
}

/*
 * Return true if the addr is a loopback address or cidr
 */
int
isloopback(struct addr *a)
{
	int i;
	struct addr *lp;

	switch (a->family) {

	case AF_INET:
		lp = &loopback4;
		if (maskwidth(lp) > maskwidth(a))
			return (0);
		return ((lp->addr4 & lp->mask4) == (a->addr4 & lp->mask4));

	case AF_INET6:
		lp = &loopback6;
		if (maskwidth(lp) > maskwidth(a))
			return (0);
		for (i = 0; i < 16; ++i)
			if ((lp->addr6[i] & lp->mask6[i]) !=
			    (a->addr6[i] & lp->mask6[i]))
				return (0);
		return (1);

	default:
		abort();
	}
}

void
lg(int pri, const char *fmt, ...)
{
	va_list ap;

	if (debug) {
		if (log_lf.fn != NULL)
			checklog(&log_lf);
		fputs(tsstr(), log_lf.f);
		putc(' ', log_lf.f);
		va_start(ap, fmt);
		(void)vfprintf(log_lf.f, fmt, ap);
		va_end(ap);
		putc('\n', log_lf.f);
	}

	va_start(ap, fmt);
	vsyslog(pri, fmt, ap);
	va_end(ap);
}

int
makeargv(const char *args, char ***avp)
{
	const char *p, *p2, *ep;
	int n, len;
	char **av;

	if (args == NULL)
		return (0);

	/* Setup end pointer (trimming trailing whitespace) */
	p = args;
	ep = args + strlen(args);
	while (ep - 1 >= args && isspace((int)ep[-1]))
		--ep;

	/* Eat leading whitespace */
	while (isspace((int)*args))
		++args;

	/* Count arguments */
	n = 0;
	while (p < ep) {
		if (!isspace((int)*p)) {
			++n;
			while (p < ep && !isspace((int)*p))
				++p;
		}
		while (p < ep && isspace((int)*p))
			++p;
	}

	av = new(n + 1, sizeof(*av), "makeargs av");
	*avp = av;

	/* Copy arguments */
	n = 0;
	p = args;
	p2 = p;
	while (p < ep) {
		if (!isspace((int)*p)) {
			++n;
			while (p < ep && !isspace((int)*p))
				++p;
			len = p - p2;
			*av = new(len + 1, sizeof(char), "makeargs string");
			memmove(*av, p2, len);
			av[0][len] = '\0';
			++av;
		}
		while (p < ep && isspace((int)*p))
			++p;
		p2 = p;
	}

	return (n);
}

socklen_t
mask2sa(struct addr *a, u_short port, struct sockaddr *sap)
{
	socklen_t salen;
	struct sockaddr_in *s4p;
	struct sockaddr_in6 *s6p;

	switch (a->family) {

	case AF_INET:
		s4p = (struct sockaddr_in *)sap;
		salen = sizeof(*s4p);
		memset(s4p, 0, salen);
		s4p->sin_port = htons(port);
		memmove(&s4p->sin_addr, &a->mask4, sizeof(s4p->sin_addr));
		break;

	case AF_INET6:
		s6p = (struct sockaddr_in6 *)sap;
		salen = sizeof(*s6p);
		memset(s6p, 0, salen);
		s6p->sin6_port = htons(port);
		memmove(&s6p->sin6_addr, &a->mask6, sizeof(s6p->sin6_addr));
		break;

	default:
		abort();
	}

#ifdef HAVE_SOCKADDR_SA_LEN
	sap->sa_len = salen;
#endif
	sap->sa_family = a->family;

	return (salen);
}

int
maskwidth(const struct addr *a)
{
	int w;
	int i, j;
	u_int32_t m, tm;

	/* Work backwards until we find a set bit */
	switch (a->family) {

	case AF_INET:
		m = ntohl(a->mask4);
		for (w = 32; w > 0; --w) {
			tm = 0xffffffff << (32 - w);
			if (tm == m)
				break;
		}
		break;

	case AF_INET6:
		w = 128;
		for (j = 15; j >= 0; --j) {
			m = a->mask6[j];
			for (i = 8; i > 0; --w, --i) {
				tm = (0xff << (8 - i)) & 0xff;
				if (tm == m)
					return (w);
			}
		}
		break;

	default:
		abort();
	}
	return (w);
}

void
nets_log(struct client *cl, struct req *rp)
{
	int i;
	char *cp;
	const char *p;
	struct child *chp;
	struct array *dp;
	struct timeval *tvp;
	char detail[1024];
	size_t cc, len;
	struct state *sp;

	/* Completion timestamp */
	sp = &state;
	tvp = &rp->cts;
	if (tvp->tv_sec == 0 && tvp->tv_usec == 0)
		getts(tvp);

	/* Initial pointer/size left */
	cp = detail;
	cc = sizeof(detail) - 1;

	/* Optional NETS id */
	if (cf->c_id != NULL) {
		(void)snprintf(cp, cc, " id=%s", cf->c_id);
		len = strlen(cp);
		cp += len;
		cc -= len;
	}

	/* Filter device(s) or optional NETS device */
	if (rp->type == REQ_FILTER || rp->type == REQ_NOFILTER) {
		dp = &sp->chp2_array;
		for (i = 0, chp = sp->chp2; i < dp->len; ++i, ++chp) {
			(void)snprintf(cp, cc, " device=%s", chp->name);
			len = strlen(cp);
			cp += len;
			cc -= len;
		}
	} else if (cf->c_device != NULL) {
		(void)snprintf(cp, cc, " device=%s", cf->c_device);
		len = strlen(cp);
		cp += len;
		cc -= len;
	}

	/* Log the aclgroup name if we know it */
	if (rp->aclgroup != NULL) {
		(void)snprintf(cp, cc, " acl=%s", rp->aclgroup->name);
		len = strlen(cp);
		cp += len;
		cc -= len;
	}

	/* No default so we'll get an error for missing cases */
	switch (rp->type) {

	case REQ_FILTER:
	case REQ_NOFILTER:
		/* filter address detail */
		(void)snprintf(cp, cc, " ip=%s", addr2str(&rp->filter));
		len = strlen(cp);
		cp += len;
		cc -= len;
		break;

	case REQ_NULLZERO:
	case REQ_NONULLZERO:
		/* route detail */
		if (rp->nullzero.type == ROUTE_NULLZERO) {
			(void)snprintf(cp, cc, " ip=%s",
			    addr2str(&rp->nullzero.dst));
			len = strlen(cp);
			cp += len;
			cc -= len;
		}
		break;

	case REQ_PREFIX:
	case REQ_NOPREFIX:
		/* filter address detail */
		(void)snprintf(cp, cc, " ip=%s prefix=%s",
		    addr2str(&rp->prefix), rp->prefixname);
		len = strlen(cp);
		cp += len;
		cc -= len;
		break;

	case REQ_DROP:
	case REQ_RESTORE:
	case REQ_BLOCKHOSTHOST:
	case REQ_RESTOREHOSTHOST:
	case REQ_DROPUDPPORT:
	case REQ_RESTOREUDPPORT:
	case REQ_DROPTCPPORT:
	case REQ_RESTORETCPPORT:
	case REQ_DROPTCPSYNPORT:
	case REQ_RESTORETCPSYNPORT:
	case REQ_PERMITUDPDSTHOSTPORT:
	case REQ_UNPERMITUDPDSTHOSTPORT:
	case REQ_PERMITTCPDSTHOSTPORT:
	case REQ_UNPERMITTCPDSTHOSTPORT:
	case REQ_DROPTCPDSTHOSTPORT:
	case REQ_RESTORETCPDSTHOSTPORT:
	case REQ_COMPACT:
		/* ACL detail */
		switch (rp->acl.type) {

		case ATYPE_PERMITHOST:
		case ATYPE_BLOCKHOST:
			(void)snprintf(cp, cc, " ip=%s",
			    addr2str(&rp->acl.addr1));
			len = strlen(cp);
			cp += len;
			cc -= len;
			break;

		case ATYPE_PERMITNET:
		case ATYPE_BLOCKNET:
			(void)snprintf(cp, cc, " net=%s",
			    addr2str(&rp->acl.addr1));
			len = strlen(cp);
			cp += len;
			cc -= len;
			break;

		case ATYPE_PERMITHOSTHOST:
		case ATYPE_BLOCKHOSTHOST:
			(void)snprintf(cp, cc, " ip=%s",
			    addr2str(&rp->acl.addr1));
			len = strlen(cp);
			cp += len;
			cc -= len;

			(void)snprintf(cp, cc, " ip2=%s",
			    addr2str(&rp->acl.addr2));
			len = strlen(cp);
			cp += len;
			cc -= len;
			break;

		case ATYPE_PERMITUDPPORT:
		case ATYPE_BLOCKUDPPORT:
		case ATYPE_PERMITTCPPORT:
		case ATYPE_BLOCKTCPPORT:
		case ATYPE_PERMITTCPSYNPORT:
		case ATYPE_BLOCKTCPSYNPORT:
			(void)snprintf(cp, cc, " port=%s", rp->acl.port1);
			len = strlen(cp);
			cp += len;
			cc -= len;
			break;

		case ATYPE_PERMITUDPHOSTPORT:
		case ATYPE_BLOCKUDPHOSTPORT:
		case ATYPE_PERMITTCPHOSTPORT:
		case ATYPE_BLOCKTCPHOSTPORT:
		case ATYPE_PERMITUDPDSTHOSTPORT:
		case ATYPE_BLOCKUDPDSTHOSTPORT:
		case ATYPE_PERMITTCPDSTHOSTPORT:
		case ATYPE_BLOCKTCPDSTHOSTPORT:
		case ATYPE_PERMITUDPHOSTSRCPORT:
		case ATYPE_BLOCKUDPHOSTSRCPORT:
		case ATYPE_PERMITTCPHOSTSRCPORT:
		case ATYPE_BLOCKTCPHOSTSRCPORT:
			(void)snprintf(cp, cc, " ip=%s port=%s",
			    addr2str(&rp->acl.addr1), rp->acl.port1);
			len = strlen(cp);
			cp += len;
			cc -= len;
			break;

		default:
			(void)snprintf(cp, cc, " ?=ATYPE_#%d",
			    rp->acl.type);
			len = strlen(cp);
			cp += len;
			cc -= len;
			break;
		}
		break;

	case REQ_ADDMACWHITELIST:
	case REQ_REMMACWHITELIST:
	case REQ_BLOCKMAC:
	case REQ_RESTOREMAC:
		(void)snprintf(cp, cc, " mac=%s", mac2str(&rp->acl.mac));
		len = strlen(cp);
		cp += len;
		cc -= len;
		break;

	case REQ_ADDWHITELIST:
	case REQ_REMWHITELIST:
		/* whitelist detail */
		if (ISHOST(&rp->whitelist))
			p = "ip";
		else
			p = "net";
		(void)snprintf(cp, cc, " %s=%s", p, addr2str(&rp->whitelist));
		len = strlen(cp);
		cp += len;
		cc -= len;
		break;

	case REQ_AYT:
	case REQ_EXIT:
	case REQ_HELP:
	case REQ_LISTACL:
	case REQ_LISTMAC:
	case REQ_LISTNULLZERO:
	case REQ_LISTPREFIX:
	case REQ_LISTROUTE:
	case REQ_MACWHITELIST:
	case REQ_QUERY:
	case REQ_QUERYFILTER:
	case REQ_QUERYMAC:
	case REQ_QUERYMACWHITELIST:
	case REQ_QUERYNULLZERO:
	case REQ_QUERYWHITELIST:
	case REQ_RELOAD:
	case REQ_REPLICANT:
	case REQ_STATE:
	case REQ_SYNC:
	case REQ_TEST:
	case REQ_UNKNOWN:
	case REQ_WHITELIST:
		/* Don't nets_log these */
		return;
	}

	/* Replicant status (but only if failed) */
	if (cl->replicant) {
		(void)snprintf(cp, cc, " replicant=yes");
		len = strlen(cp);
		cp += len;
		cc -= len;
	}

	/* Replicant status (but only if failed) */
	if (rp->rereq && (rp->rereq->reflags & REFLAG_FAILED) != 0)
		(void)snprintf(cp, cc, " restatus=failed");

	/* Status */
	if ((rp->flags & RFLAG_FAILED) != 0)
		p = "failed";
	else if ((rp->flags & RFLAG_IGNORE) != 0)
		p = "ignore";
	else
		p = "success";

	syslog(cf->c_netsfac | LOG_INFO,
	    "NETS status=%s cmd=%s%s client=%s"
	    " ats=%lu.%06lu cts=%lu.%06lu cmt=%s",
	    p,
	    rp->cmd,
	    detail,
	    addr2str(&cl->addr),
	    (u_long)rp->ats.tv_sec, (u_long)rp->ats.tv_usec,
	    (u_long)rp->cts.tv_sec, (u_long)rp->cts.tv_usec,
	    pretty(rp->comment));
}

void *
new(size_t num, size_t size, const char *what)
{
	void *p;

	p = calloc(num, size);
	if (p == NULL) {
		lg(LOG_ERR, "new: calloc %s: %s", what, strerror(errno));
		exit(EX_OSERR);
	}
	return (p);
}

const char *
pretty(const char *str)
{
	size_t cc;
	char *cp;
	const char *p;
	static char buf[8192];

	if (str == NULL)
		return ("");

	cp = buf;
	cc = sizeof(buf) - 1;

	/* Leave room for possible escape, char and eos */
	for (p = str; *p != '\0' && cc >= 3; ++p) {
		if (*p == '\n') {
			*cp++ = '\\';
			*cp++ = 'n';
			cc -= 2;
		} else if (*p == '\r') {
			*cp++ = '\\';
			*cp++ = 'r';
			cc -= 2;
		} else if (*p == '"') {
			*cp++ = '\\';
			*cp++ = *p;
			cc -= 2;
		} else if (isprint((int)*p)) {
			*cp++ = *p;
			--cc;
		} else {
			/* DEL to ?, others to alpha */
			*cp++ = '^';
			*cp++ = *p ^ 0x40;
			cc -= 2;
		}
	}
	*cp = '\0';
	return (buf);
}

const char *
readfile(const char *fn, const char *what, int mustexist, void *opaque,
    const char * (*callback)(const char *, const char *, void *))
{
	int n;
	FILE *f;
	char *cp;
	char buf[1024];
	char prefix[128];
	static char errstr[256];

	f = fopen(fn, "r");
	if (f == NULL) {
		if (mustexist) {
			(void)snprintf(errstr, sizeof(errstr), "fopen %s: %s",
			    fn, strerror(errno));
			return (errstr);
		}
		return (NULL);
	}

	n = 0;
	while (fgets(buf, sizeof(buf), f) != NULL) {
		++n;

		/* Eat comments */
		cp = strchr(buf, '#');
		if (cp)
			*cp = '\0';

		/* Eat trailing whitesapce */
		cp = buf + strlen(buf) - 1;
		while (cp >= buf && isspace((int)*cp))
			*cp-- = '\0';
		cp = buf;

		/* Skip blank lines */
		if (*cp == '\0')
			continue;

		/* Callback */
		(void)snprintf(prefix, sizeof(prefix), "%s:%d: %s",
		    fn, n, what);
		callback(prefix, buf, opaque);
	}
	(void)fclose(f);
	return (NULL);
}

/* Subnet mask sockaddr is optional */
void
sa2addr(struct sockaddr *sap, struct sockaddr *smp, struct addr *a,
    u_int16_t *portp)
{
	struct sockaddr_in *s4p, *s4mp;
	struct sockaddr_in6 *s6p, *s6mp;

	memset(a, 0, sizeof(*a));
	a->family = sap->sa_family;

	switch (a->family) {

	case AF_INET:
		s4p = (struct sockaddr_in *)sap;
		*portp = ntohs(s4p->sin_port);
		memmove(&a->addr4, &s4p->sin_addr, sizeof(a->addr4));
		if (smp == NULL) {
			a->mask4 = 0xffffffff;
		} else {
			s4mp = (struct sockaddr_in *)smp;
			memmove(&a->mask4, &s4mp->sin_addr, sizeof(a->mask4));
		}
		break;

	case AF_INET6:
		s6p = (struct sockaddr_in6 *)sap;
		*portp = ntohs(s6p->sin6_port);
		memmove(&a->addr6, &s6p->sin6_addr, sizeof(a->addr6));
		if (smp == NULL) {
			setmaskwidth(128, a);
		} else {
			s6mp = (struct sockaddr_in6 *)smp;
			memmove(&a->mask6, &s6mp->sin6_addr, sizeof(a->mask6));
		}
		break;

	default:
		abort();
	}
}

/* Set address mask in network order */
void
setmaskwidth(u_int w, struct addr *a)
{
	int i, j;

	switch (a->family) {

	case AF_INET:
		if (w == 0)
			a->mask4 = 0;
		else
			a->mask4 = htonl(0xffffffff << (32 - w));
		break;

	case AF_INET6:
		memset(a->mask6, 0, sizeof(a->mask6));
		for (i = 0; i < w / 8; ++i)
			a->mask6[i] = 0xff;
		i = w / 8;
		j = w % 8;
		if (j > 0 && i < 16)
			a->mask6[i] = 0xff << (8 - j);
		break;

	default:
		abort();
	}
}

int
str2val(const struct s2v *lp, const char *suffix)
{
	const struct s2v *tp;

	for (tp = lp; tp->s != NULL; ++tp)
		if (strcmp(suffix, tp->s) == 0)
			break;

	return (tp->v);
}

#ifdef notdef
int
str2valcase(const struct s2v *lp, const char *suffix)
{
	const struct s2v *tp;

	for (tp = lp; tp->s != NULL; ++tp)
		if (strcasecmp(suffix, tp->s) == 0)
			break;

	return (tp->v);
}
#endif

void
strappend(char **pp, const char *str)
{
	int len;

	if (*pp == NULL)
		*pp = strsave(str);
	else {
		len = strlen(*pp);
		*pp = realloc(*pp, len + strlen(str) + 1);
		if (*pp == NULL) {
			lg(LOG_ERR, "strappend: realloc: %s", strerror(errno));
			exit(EX_OSERR);
		}
		strcpy(*pp + len, str);
	}
}

/* Strip leading and trailing white space removed */
const char *
stripws(const char *str)
{
	static char buf[1024];

	while (isspace((int)*str))
		++str;
	strlcpy(buf, str, sizeof(buf) - 1);
	trimws(buf, strlen(buf));
	return (buf);
}

char *
strsave(const char *str)
{
	char *cp;

	cp = strdup(str);
	if (cp == NULL) {
		lg(LOG_ERR, "strsave: strdup: %s", strerror(errno));
		exit(EX_OSERR);
	}
	return (cp);
}

const char *
tcporudpreqstr(int rtype)
{
	const char *p;

	p = "?";
	switch (rtype) {

	case REQ_DROPUDPPORT:
	case REQ_RESTOREUDPPORT:
	case REQ_PERMITUDPDSTHOSTPORT:
	case REQ_UNPERMITUDPDSTHOSTPORT:
		p = "UDP";
		break;

	case REQ_DROPTCPPORT:
	case REQ_RESTORETCPPORT:
	case REQ_DROPTCPSYNPORT:
	case REQ_RESTORETCPSYNPORT:
	case REQ_PERMITTCPDSTHOSTPORT:
	case REQ_UNPERMITTCPDSTHOSTPORT:
	case REQ_DROPTCPDSTHOSTPORT:
	case REQ_RESTORETCPDSTHOSTPORT:
		p = "TCP";
		break;

	case REQ_ADDWHITELIST:
	case REQ_AYT:
	case REQ_BLOCKHOSTHOST:
	case REQ_COMPACT:
	case REQ_DROP:
	case REQ_EXIT:
	case REQ_HELP:
	case REQ_LISTACL:
	case REQ_LISTMAC:
	case REQ_LISTNULLZERO:
	case REQ_LISTROUTE:
	case REQ_NOFILTER:
	case REQ_FILTER:
	case REQ_NONULLZERO:
	case REQ_NULLZERO:
	case REQ_QUERY:
	case REQ_QUERYFILTER:
	case REQ_QUERYNULLZERO:
	case REQ_QUERYWHITELIST:
	case REQ_RELOAD:
	case REQ_REMWHITELIST:
	case REQ_REPLICANT:
	case REQ_RESTORE:
	case REQ_RESTOREHOSTHOST:
	case REQ_STATE:
	case REQ_SYNC:
	case REQ_TEST:
	case REQ_UNKNOWN:
	case REQ_WHITELIST:
		break;
	}
	return (p);
}

void
trimws(char *str, size_t len)
{
	char *cp;

	cp = str + len;
	while (cp > str && isspace((int)cp[-1]))
		--cp;
	*cp = '\0';
}

const char *
tsstr(void)
{
	struct timeval tv;
	static char buf[32];

	getts(&tv);
	(void)snprintf(buf, sizeof(buf), "%lu.%06lu",
	    (u_long)tv.tv_sec, (u_long)tv.tv_usec);
	return (buf);
}

#ifdef notdef
int
typ2flag(const struct v2v *lp, int wtype)
{
	const struct v2v *vp;

	for (vp = lp; vp->v1 != 0; ++vp)
		if (vp->v1 == wtype)
			break;
	return (vp->v2);
}
#endif

const char *
val2str(const struct s2v *lp, int val)
{
	const struct s2v *tp;
	static char buf[32];

	for (tp = lp; tp->s != NULL; ++tp)
		if (val == tp->v)
			break;

	if (tp->s == NULL) {
		(void)snprintf(buf, sizeof(buf), "#%d", val);
		return (buf);
	}
	return (tp->s);
}

const char *
writefile(const char *fn, void *opaque, void (*callback)(FILE *, void *))
{
	FILE *f;
	time_t now;
	struct tm *tm;
	char date[64];
	char zone[32];
	char temp[256];
	static char errstr[512];
#ifdef HAVE_ALTZONE
	char ch;
	int z;
#endif

	(void)snprintf(temp, sizeof(temp), "%.*s-", (int)sizeof(temp) - 2, fn);
	f = fopen(temp, "w");
	if (f == NULL) {
		(void)snprintf(errstr, sizeof(errstr), "fopen %s: %s",
		    temp, strerror(errno));
		return (errstr);
	}
	fprintf(f, "# WARNING: dynamically updated; do not edit\n");
	now = time(NULL);
	tm = localtime(&now);
	(void)strftime(date, sizeof(date), "%d-%b-%Y %T", tm);
#ifndef HAVE_ALTZONE
	(void)strftime(zone, sizeof(zone), "%z", tm);
#else
	z = altzone / -36;
	ch = '+';
	if (z < 0) {
		z = -z;
		ch = '-';
	}
	(void)snprintf(zone, sizeof(zone), "%c%04d", ch, z);
#endif
	fprintf(f, "# Last update: %s %s\n", date, zone);

	/* Zeek Input Framework friendly "#fields" header line */
	fputs("#fields\tstr\n", f);

	/* Write the content */
	callback(f, opaque);

	if (fclose(f) != 0) {
		(void)snprintf(errstr, sizeof(errstr), "fclose %s: %s",
		    temp, strerror(errno));
		return (errstr);
	}
	if (rename(temp, fn) < 0) {
		(void)snprintf(errstr, sizeof(errstr), "rename %s -> %s: %s",
		    temp, fn, strerror(errno));
		return (errstr);
	}
	return (NULL);
}
