/*
 * Copyright (c) 2006, 2007, 2008, 2009, 2010, 2011, 2013, 2015, 2018, 2020, 2021, 2023, 2024
 * The Regents of the University of California. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *     * Redistributions of source code must retain the above copyright
 *       notice, this list of conditions and the following disclaimer.
 *     * Redistributions in binary form must reproduce the above copyright
 *       notice, this list of conditions and the following disclaimer in the
 *       documentation and/or other materials provided with the distribution.
 *     * Neither the name of the University nor the names of its contributors
 *       may be used to endorse or promote products derived from this software
 *       without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#ifndef lint
static const char rcsid[] __attribute__ ((unused)) =
    "@(#) $Id$ (LBL)";
#endif

#include <sys/types.h>
#include <sys/time.h>

#include <ctype.h>
#include <errno.h>
#include <signal.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sysexits.h>
#include <time.h>
#include <unistd.h>

#include "acld.h"
#include "child.h"
#include "mac.h"

/* Forwards */
static const char *macwhitelistcallback(const char *, const char *, void *);
static void macwhitelistfilecallback(FILE *, void *);
static const char *whitelistcallback(const char *, const char *, void *);
static void whitelistfilecallback(FILE *, void *);


/* Returns mac if whitelisted else NULL */
struct mac *
macwhitelist(struct mac *macaddr)
{
	int i;
	struct mac *ma;
	struct array *dp;

	dp = &cf->c_macwhitelist_array;
	if (dp->len == 0)
		return (NULL);

	for (i = 0, ma = cf->c_macwhitelist; i < dp->len; ++i, ++ma)
		if (ITEMEQ(ma, macaddr))
			return (ma);

	return (NULL);
}

/* Does not check to see if the mac is already on the list */
void
macwhitelistadd(struct mac *macaddr)
{
	struct mac *ma;
	struct array *dp;

	dp = &cf->c_macwhitelist_array;
	DYNARRAY(dp, cf->c_macwhitelist, dp->len + 1, "macwhitelistadd");
	ma = cf->c_macwhitelist + dp->len;
	*ma = *macaddr;
	++dp->len;

	/* Keep everything sorted */
	qsort(cf->c_macwhitelist, dp->len, sizeof(*ma), maccmp);
}

static const char *
macwhitelistcallback(const char *prefix, const char *s, void *opaque)
{
	struct mac *ma;
	struct mac mac;
	const char *errmsg;
	static char errstr[256];

	ma = &mac;
	ITEMZERO(ma);
	errmsg = extractmac(s, ma);
	if (errmsg != NULL) {
		(void)snprintf(errstr, sizeof(errstr), "%s: %s",
		    prefix, errmsg);
		return (errstr);
	}

	macwhitelistadd(ma);
	return (NULL);
}

int
macwhitelistdump(struct iobuf *ip)
{
	int i;
	struct mac *ma;
	struct array *dp;

	dp = &cf->c_macwhitelist_array;
	for (i = 0, ma = cf->c_macwhitelist; i < dp->len; ++i, ++ma)
		ioappendfmt(ip, "%s", mac2str(ma));
	return (dp->len);
}

static void
macwhitelistfilecallback(FILE *f, void *opaque)
{
	int i;
	struct mac *ma;
	struct array *dp;

	dp = &cf->c_macwhitelist_array;
	for (i = 0, ma = cf->c_macwhitelist; i < dp->len; ++i, ++ma)
		fprintf(f, "%s\n", mac2str(ma));
}

void
macwhitelistread(void)
{
	const char *errmsg;

	if (cf->c_macwhitelistfn == NULL)
		return;

	errmsg = readfile(cf->c_macwhitelistfn, "macwhitelistfn", 0, NULL,
	    macwhitelistcallback);
	if (errmsg != NULL) {
		lg(LOG_ERR, "macwhitelistread: %s", errmsg);
		exit(1);
	}
}

int
macwhitelistrem(struct mac *macaddr)
{
	int i, j;
	struct mac *ma;
	struct array *dp;

	dp = &cf->c_macwhitelist_array;
	if (dp->len == 0)
		return (0);

	for (i = 0, ma = cf->c_macwhitelist; i < dp->len; ++i, ++ma)
		if (ITEMEQ(ma, macaddr)) {
			j = dp->len - (i + 1);
			if (j > 0)
				memmove(cf->c_macwhitelist + i,
				    cf->c_macwhitelist + i + 1,
				    j * sizeof(*ma));
			--dp->len;
			ITEMZERO(ma + dp->len);
			return (1);
		}

	return (0);
}

const char *
macwhitelistupdate(void)
{
	if (cf->c_macwhitelistfn == NULL)
		return ("macwhitelistupdate: macwhitelist file not configured");

	return (writefile(cf->c_macwhitelistfn, NULL,
	    macwhitelistfilecallback));
}

// XXX why do we need this *and* whitelistquery()?

/* Returns addr if address or network is whitelisted else NULL */
struct addr *
whitelist(struct addr *addr)
{
	int i, w;
	struct addr *a;
	struct array *dp;

	dp = &cf->c_whitelist_array;
	if (dp->len == 0)
		return (NULL);

	/* Return on any overlap */
	w = maskwidth(addr);
	for (i = 0, a = cf->c_whitelist; i < dp->len; ++i, ++a) {
		if (a->family != addr->family)
			continue;
		if (w >= maskwidth(a)) {
			/* Test addr is more specific than whitelist addr */
			if (insubnet(a, addr))
				return (a);
		} else {
			/* Test addr is less specific than whitelist addr */
			if (insubnet(addr, a))
				return (a);
		}
	}

	return (NULL);
}

/* Does not check to see if the addr is already on the list */
void
whitelistadd(struct addr *addr)
{
	struct addr *a;
	struct array *dp;

	dp = &cf->c_whitelist_array;
	DYNARRAY(dp, cf->c_whitelist, dp->len + 1, "whitelistadd");
	a = cf->c_whitelist + dp->len;
	*a = *addr;
	++dp->len;

	/* Keep everything sorted */
	qsort(cf->c_whitelist, dp->len, sizeof(*a), addrcmp);
}

static const char *
whitelistcallback(const char *prefix, const char *s, void *opaque)
{
	struct addr *a;
	struct addr addr;
	const char *errmsg;
	static char errstr[256];

	a = &addr;
	ITEMZERO(a);
	errmsg = extractaddr(s, NULL, a);
	if (errmsg != NULL) {
		(void)snprintf(errstr, sizeof(errstr), "%s: %s",
		    prefix, errmsg);
		return (errstr);
	}

	whitelistadd(a);
	return (NULL);
}

int
whitelistdump(struct iobuf *ip)
{
	int i;
	struct addr *a;
	struct array *dp;

	dp = &cf->c_whitelist_array;
	for (i = 0, a = cf->c_whitelist; i < dp->len; ++i, ++a)
		ioappendfmt(ip, "%s", addr2str(a));
	return (dp->len);
}

static void
whitelistfilecallback(FILE *f, void *opaque)
{
	int i;
	struct addr *a;
	struct array *dp;

	dp = &cf->c_whitelist_array;
	for (i = 0, a = cf->c_whitelist; i < dp->len; ++i, ++a)
		fprintf(f, "%s\n", addr2str(a));
}

/* Returns number of matching whitelist entries reported */
int
whitelistquery(struct addr *addr, struct iobuf *ip)
{
	int i, cnt, w;
	struct addr *a;
	struct array *dp;

	dp = &cf->c_whitelist_array;
	if (dp->len == 0)
		return (0);

	/* Report on any overlap */
	w = maskwidth(addr);
	cnt = 0;
	for (i = 0, a = cf->c_whitelist; i < dp->len; ++i, ++a) {
		if (a->family != addr->family)
			continue;
		if (w >= maskwidth(a)) {
			/* Test addr is more specific than whitelist addr */
			if (insubnet(a, addr)) {
				ioappendfmt(ip, "%s", addr2str(a));
				++cnt;
			}
		} else {
			/* Test addr is less specific than whitelist addr */
			if (insubnet(addr, a)) {
				ioappendfmt(ip, "%s", addr2str(a));
				++cnt;
			}
		}
	}

	return (cnt);
}

void
whitelistread(void)
{
	const char *errmsg;

	if (cf->c_whitelistfn == NULL)
		return;

	errmsg = readfile(cf->c_whitelistfn, "whitelistfn", 0, NULL,
	    whitelistcallback);
	if (errmsg != NULL) {
		lg(LOG_ERR, "whitelistread: %s", errmsg);
		exit(1);
	}
}

/* Return true if successful */
int
whitelistrem(struct addr *addr)
{
	int i, j;
	struct addr *a;
	struct array *dp;

	dp = &cf->c_whitelist_array;
	if (dp->len == 0)
		return (0);

	for (i = 0, a = cf->c_whitelist; i < dp->len; ++i, ++a)
		if (ITEMEQ(a, addr)) {
			j = dp->len - (i + 1);
			if (j > 0)
				memmove(cf->c_whitelist + i,
				    cf->c_whitelist + i + 1, j * sizeof(*a));
			--dp->len;
			ITEMZERO(a + dp->len);
			return (1);
		}

	return (0);
}

/* Look for the exact address in the whitelist */
struct addr *
whitelistsearch(struct addr *addr)
{
	int i;
	struct addr *a;
	struct array *dp;

	dp = &cf->c_whitelist_array;
	if (dp->len == 0)
		return (NULL);

	for (i = 0, a = cf->c_whitelist; i < dp->len; ++i, ++a)
		if (ITEMEQ(a, addr))
			return (a);

	return (NULL);
}

const char *
whitelistupdate(void)
{
	if (cf->c_whitelistfn == NULL)
		return ("whitelistupdate: whitelist file not configured");

	return (writefile(cf->c_whitelistfn, NULL, whitelistfilecallback));
}
